#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util::GMP qw/is_almost_prime is_prime is_semiprime/;

my %kap=(
    0 => [1],
    1 => [2,3,5,7,11,13,17,19,23,29],
    2 => [4,6,9,10,14,15,21,22,25,26],
    3 => [8,12,18,20,27,28,30,42,44,45],
    4 => [16,24,36,40,54,56,60,81,84,88],
    5 => [32,48,72,80,108,112,120,162,168,176],
    6 => [64,96,144,160,216,224,240,324,336,352],
    7 => [128,192,288,320,432,448,480,648,672,704],
    8 => [256,384,576,640,864,896,960,1296,1344,1408],
    9 => [512,768,1152,1280,1728,1792,1920,2592,2688,2816],
   10 => [1024,1536,2304,2560,3456,3584,3840,5184,5376,5632],
   11 => [2048,3072,4608,5120,6912,7168,7680,10368,10752,11264],
   12 => [4096,6144,9216,10240,13824,14336,15360,20736,21504,22528],
   13 => [8192,12288,18432,20480,27648,28672,30720,41472,43008,45056],
   14 => [16384,24576,36864,40960,55296,57344,61440,82944,86016,90112],
   15 => [32768,49152,73728,81920,110592,114688,122880,165888,172032,180224],
   16 => [65536,98304,147456,163840,221184,229376,245760,331776,344064,360448],
   17 => [131072,196608,294912,327680,442368,458752,491520,663552,688128,720896],
   18 => [262144,393216,589824,655360,884736,917504,983040,1327104,1376256,1441792],
   19 => [524288,786432,1179648,1310720,1769472,1835008,1966080,2654208,2752512,2883584],
   20 => [1048576,1572864,2359296,2621440,3538944,3670016,3932160,5308416,5505024,5767168],
);

plan tests => 11 + scalar(keys(%kap)) + 1;

for my $k (0 .. 10) {
  my @exp = map { fac_is_almost_prime($k, $_) } 0 .. 40;
  my @got = map {     is_almost_prime($k, $_) } 0 .. 40;
  is_deeply( \@got, \@exp, "is_almost_prime($k, 0..40)" );
}

while (my($k, $pvals) = each (%kap)) {
  my $failed = 0;
  my $nvals = scalar(@$pvals);
  for my $n (@$pvals) { $failed++ if is_almost_prime($k, $n) != 1; }
  is( 0, $failed, "Test first $nvals $k-almost-primes return true" );
}

# TODO: more bignums
is( is_almost_prime(3,"250320631882951088595682695816397374705"), 1, "3*5*prime is a 3-almost-prime" );


sub fac_is_almost_prime {
  my($k, $n) = @_;
  return 0+($n==1) if $k == 0;
  return (is_prime($n) ? 1 : 0) if $k == 1;
  return is_semiprime($n) if $k == 2;

  return (scalar(Math::Prime::Util::GMP::factor($n)) == $k) ? 1 : 0;
}
