// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package builds

import (
	"net/http"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/go-scm/scm"

	"github.com/go-chi/chi"
)

// HandleCreate returns an http.HandlerFunc that processes http
// requests to create a build for the specified commit.
func HandleCreate(
	users core.UserStore,
	repos core.RepositoryStore,
	commits core.CommitService,
	triggerer core.Triggerer,
) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var (
			ctx       = r.Context()
			namespace = chi.URLParam(r, "owner")
			name      = chi.URLParam(r, "name")
			sha       = r.FormValue("commit")
			branch    = r.FormValue("branch")
			message   = r.FormValue("message")
			action    = r.FormValue("action")
			user, _   = request.UserFrom(ctx)
		)

		repo, err := repos.FindName(ctx, namespace, name)
		if err != nil {
			render.NotFound(w, err)
			return
		}

		owner, err := users.Find(ctx, repo.UserID)
		if err != nil {
			render.NotFound(w, err)
			return
		}

		// if the user does not provide a branch, assume the
		// default repository branch.
		if branch == "" {
			branch = repo.Branch
		}
		// expand the branch to a git reference.
		ref := scm.ExpandRef(branch, "refs/heads")

		var commit *core.Commit
		if sha != "" {
			commit, err = commits.Find(ctx, owner, repo.Slug, sha)
		} else {
			commit, err = commits.FindRef(ctx, owner, repo.Slug, ref)
		}
		if err != nil {
			render.NotFound(w, err)
			return
		}

		hook := &core.Hook{
			Trigger:      user.Login,
			Event:        core.EventCustom,
			Link:         commit.Link,
			Timestamp:    commit.Author.Date,
			Title:        "", // we expect this to be empty.
			Message:      commit.Message,
			Before:       commit.Sha,
			After:        commit.Sha,
			Ref:          ref,
			Source:       branch,
			Target:       branch,
			Author:       commit.Author.Login,
			AuthorName:   commit.Author.Name,
			AuthorEmail:  commit.Author.Email,
			AuthorAvatar: commit.Author.Avatar,
			Sender:       user.Login,
			Params:       map[string]string{},
		}
		if len(message) > 0 {
			hook.Message = message
		}
		if len(action) > 0 {
			hook.Action = action
		}

		for key, value := range r.URL.Query() {
			if key == "access_token" ||
				key == "commit" ||
				key == "branch" {
				continue
			}
			if len(value) == 0 {
				continue
			}
			hook.Params[key] = value[0]
		}

		result, err := triggerer.Trigger(r.Context(), repo, hook)
		if err != nil {
			render.InternalError(w, err)
		} else {
			render.JSON(w, result, 200)
		}
	}
}
