// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A6__F247

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_80_0_T_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xef47f2e4u, 0x06b4d1f5u, 0, 109 }, // ef47f2e406b4d1f5 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1201
    { 0x2751a8d4u, 0x8c0cc0d8u, 0, 126 }, // 2751a8d48c0cc0d8 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x1ebf0c8eu, 0xfad85045u, 0, 6 }, // 1ebf0c8efad85045 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0x69be2fa2u, 0x7bc23952u, 0, 69 }, // 69be2fa27bc23952 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x02440758u, 0xeb3857d0u, 0, 143 }, // 02440758eb3857d0 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0xfa0ef7b9u, 0x12737671u, 0, 86 }, // fa0ef7b912737671 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0x6e5aaea6u, 0x34f97bd7u, 0, 46 }, // 6e5aaea634f97bd7 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xb4b046ffu, 0x2f8d2fe8u, 23, 69 }, // b4b046ff2f8d2fe8 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0xed20abf5u, 0xb39683c4u, 23, 143 }, // ed20abf5b39683c4 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0xcae5bd28u, 0x98f53c69u, 23, 29 }, // cae5bd2898f53c69 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x10908375u, 0x18e50988u, 23, 86 }, // 1090837518e50988 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_fuse/FONLY__＊bf16@16_80_0_T_T_1___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1201_mod0
{{ 6, 0, 5, 5, 0, 4, 7},
 { 4, 6, 5, 5, 4, 2, 5},
 { 2, 3, 4, 0, 0, 0, 0},
 { 9, 4, 0, 0, 4, 0, 5},
 { 9, 0, 0, 0, 0, 0, 0},
 { 7, 0, 0, 0, 0, 5,10},
 { 9, 0, 0, 0, 2, 8,10}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A6__F247 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 80
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 80 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

