﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dms/DatabaseMigrationServiceRequest.h>
#include <aws/dms/DatabaseMigrationService_EXPORTS.h>
#include <aws/dms/model/Tag.h>

#include <utility>

namespace Aws {
namespace DatabaseMigrationService {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dms-2016-01-01/CreateEventSubscriptionMessage">AWS
 * API Reference</a></p>
 */
class CreateEventSubscriptionRequest : public DatabaseMigrationServiceRequest {
 public:
  AWS_DATABASEMIGRATIONSERVICE_API CreateEventSubscriptionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEventSubscription"; }

  AWS_DATABASEMIGRATIONSERVICE_API Aws::String SerializePayload() const override;

  AWS_DATABASEMIGRATIONSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the DMS event notification subscription. This name must be less
   * than 255 characters.</p>
   */
  inline const Aws::String& GetSubscriptionName() const { return m_subscriptionName; }
  inline bool SubscriptionNameHasBeenSet() const { return m_subscriptionNameHasBeenSet; }
  template <typename SubscriptionNameT = Aws::String>
  void SetSubscriptionName(SubscriptionNameT&& value) {
    m_subscriptionNameHasBeenSet = true;
    m_subscriptionName = std::forward<SubscriptionNameT>(value);
  }
  template <typename SubscriptionNameT = Aws::String>
  CreateEventSubscriptionRequest& WithSubscriptionName(SubscriptionNameT&& value) {
    SetSubscriptionName(std::forward<SubscriptionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) of the Amazon SNS topic created for event
   * notification. The ARN is created by Amazon SNS when you create a topic and
   * subscribe to it. </p>
   */
  inline const Aws::String& GetSnsTopicArn() const { return m_snsTopicArn; }
  inline bool SnsTopicArnHasBeenSet() const { return m_snsTopicArnHasBeenSet; }
  template <typename SnsTopicArnT = Aws::String>
  void SetSnsTopicArn(SnsTopicArnT&& value) {
    m_snsTopicArnHasBeenSet = true;
    m_snsTopicArn = std::forward<SnsTopicArnT>(value);
  }
  template <typename SnsTopicArnT = Aws::String>
  CreateEventSubscriptionRequest& WithSnsTopicArn(SnsTopicArnT&& value) {
    SetSnsTopicArn(std::forward<SnsTopicArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The type of DMS resource that generates the events. For example, if you want
   * to be notified of events generated by a replication instance, you set this
   * parameter to <code>replication-instance</code>. If this value isn't specified,
   * all events are returned. </p> <p>Valid values: <code>replication-instance</code>
   * | <code>replication-task</code> </p>
   */
  inline const Aws::String& GetSourceType() const { return m_sourceType; }
  inline bool SourceTypeHasBeenSet() const { return m_sourceTypeHasBeenSet; }
  template <typename SourceTypeT = Aws::String>
  void SetSourceType(SourceTypeT&& value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = std::forward<SourceTypeT>(value);
  }
  template <typename SourceTypeT = Aws::String>
  CreateEventSubscriptionRequest& WithSourceType(SourceTypeT&& value) {
    SetSourceType(std::forward<SourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of event categories for a source type that you want to subscribe to.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Events.html">Working
   * with Events and Notifications</a> in the <i>Database Migration Service User
   * Guide.</i> </p>
   */
  inline const Aws::Vector<Aws::String>& GetEventCategories() const { return m_eventCategories; }
  inline bool EventCategoriesHasBeenSet() const { return m_eventCategoriesHasBeenSet; }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  void SetEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories = std::forward<EventCategoriesT>(value);
  }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  CreateEventSubscriptionRequest& WithEventCategories(EventCategoriesT&& value) {
    SetEventCategories(std::forward<EventCategoriesT>(value));
    return *this;
  }
  template <typename EventCategoriesT = Aws::String>
  CreateEventSubscriptionRequest& AddEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories.emplace_back(std::forward<EventCategoriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of identifiers for which DMS provides notification events.</p> <p>If
   * you don't specify a value, notifications are provided for all sources.</p> <p>If
   * you specify multiple values, they must be of the same type. For example, if you
   * specify a database instance ID, then all of the other values must be database
   * instance IDs.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSourceIds() const { return m_sourceIds; }
  inline bool SourceIdsHasBeenSet() const { return m_sourceIdsHasBeenSet; }
  template <typename SourceIdsT = Aws::Vector<Aws::String>>
  void SetSourceIds(SourceIdsT&& value) {
    m_sourceIdsHasBeenSet = true;
    m_sourceIds = std::forward<SourceIdsT>(value);
  }
  template <typename SourceIdsT = Aws::Vector<Aws::String>>
  CreateEventSubscriptionRequest& WithSourceIds(SourceIdsT&& value) {
    SetSourceIds(std::forward<SourceIdsT>(value));
    return *this;
  }
  template <typename SourceIdsT = Aws::String>
  CreateEventSubscriptionRequest& AddSourceIds(SourceIdsT&& value) {
    m_sourceIdsHasBeenSet = true;
    m_sourceIds.emplace_back(std::forward<SourceIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A Boolean value; set to <code>true</code> to activate the subscription, or
   * set to <code>false</code> to create the subscription but not activate it. </p>
   */
  inline bool GetEnabled() const { return m_enabled; }
  inline bool EnabledHasBeenSet() const { return m_enabledHasBeenSet; }
  inline void SetEnabled(bool value) {
    m_enabledHasBeenSet = true;
    m_enabled = value;
  }
  inline CreateEventSubscriptionRequest& WithEnabled(bool value) {
    SetEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more tags to be assigned to the event subscription.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateEventSubscriptionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateEventSubscriptionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_subscriptionName;

  Aws::String m_snsTopicArn;

  Aws::String m_sourceType;

  Aws::Vector<Aws::String> m_eventCategories;

  Aws::Vector<Aws::String> m_sourceIds;

  bool m_enabled{false};

  Aws::Vector<Tag> m_tags;
  bool m_subscriptionNameHasBeenSet = false;
  bool m_snsTopicArnHasBeenSet = false;
  bool m_sourceTypeHasBeenSet = false;
  bool m_eventCategoriesHasBeenSet = false;
  bool m_sourceIdsHasBeenSet = false;
  bool m_enabledHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DatabaseMigrationService
}  // namespace Aws
