﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class UpdateBuildRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API UpdateBuildRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateBuild"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique identifier for the build to update. You can use either the build ID
   * or ARN value. </p>
   */
  inline const Aws::String& GetBuildId() const { return m_buildId; }
  inline bool BuildIdHasBeenSet() const { return m_buildIdHasBeenSet; }
  template <typename BuildIdT = Aws::String>
  void SetBuildId(BuildIdT&& value) {
    m_buildIdHasBeenSet = true;
    m_buildId = std::forward<BuildIdT>(value);
  }
  template <typename BuildIdT = Aws::String>
  UpdateBuildRequest& WithBuildId(BuildIdT&& value) {
    SetBuildId(std::forward<BuildIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A descriptive label that is associated with a build. Build names do not need
   * to be unique. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateBuildRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version information that is associated with a build or script. Version
   * strings do not need to be unique.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  UpdateBuildRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_buildId;

  Aws::String m_name;

  Aws::String m_version;
  bool m_buildIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
