﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lexv2-models/LexModelsV2Request.h>
#include <aws/lexv2-models/LexModelsV2_EXPORTS.h>
#include <aws/lexv2-models/model/BotMember.h>
#include <aws/lexv2-models/model/BotType.h>
#include <aws/lexv2-models/model/DataPrivacy.h>
#include <aws/lexv2-models/model/ErrorLogSettings.h>

#include <utility>

namespace Aws {
namespace LexModelsV2 {
namespace Model {

/**
 */
class CreateBotRequest : public LexModelsV2Request {
 public:
  AWS_LEXMODELSV2_API CreateBotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBot"; }

  AWS_LEXMODELSV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the bot. The bot name must be unique in the account that creates
   * the bot.</p>
   */
  inline const Aws::String& GetBotName() const { return m_botName; }
  inline bool BotNameHasBeenSet() const { return m_botNameHasBeenSet; }
  template <typename BotNameT = Aws::String>
  void SetBotName(BotNameT&& value) {
    m_botNameHasBeenSet = true;
    m_botName = std::forward<BotNameT>(value);
  }
  template <typename BotNameT = Aws::String>
  CreateBotRequest& WithBotName(BotNameT&& value) {
    SetBotName(std::forward<BotNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the bot. It appears in lists to help you identify a
   * particular bot.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateBotRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role that has permission to access
   * the bot.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateBotRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides information on additional privacy protections Amazon Lex should use
   * with the bot's data.</p>
   */
  inline const DataPrivacy& GetDataPrivacy() const { return m_dataPrivacy; }
  inline bool DataPrivacyHasBeenSet() const { return m_dataPrivacyHasBeenSet; }
  template <typename DataPrivacyT = DataPrivacy>
  void SetDataPrivacy(DataPrivacyT&& value) {
    m_dataPrivacyHasBeenSet = true;
    m_dataPrivacy = std::forward<DataPrivacyT>(value);
  }
  template <typename DataPrivacyT = DataPrivacy>
  CreateBotRequest& WithDataPrivacy(DataPrivacyT&& value) {
    SetDataPrivacy(std::forward<DataPrivacyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time, in seconds, that Amazon Lex should keep information about a user's
   * conversation with the bot. </p> <p>A user interaction remains active for the
   * amount of time specified. If no conversation occurs during this time, the
   * session expires and Amazon Lex deletes any data provided before the timeout.</p>
   * <p>You can specify between 60 (1 minute) and 86,400 (24 hours) seconds.</p>
   */
  inline int GetIdleSessionTTLInSeconds() const { return m_idleSessionTTLInSeconds; }
  inline bool IdleSessionTTLInSecondsHasBeenSet() const { return m_idleSessionTTLInSecondsHasBeenSet; }
  inline void SetIdleSessionTTLInSeconds(int value) {
    m_idleSessionTTLInSecondsHasBeenSet = true;
    m_idleSessionTTLInSeconds = value;
  }
  inline CreateBotRequest& WithIdleSessionTTLInSeconds(int value) {
    SetIdleSessionTTLInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to add to the bot. You can only add tags when you create a
   * bot. You can't use the <code>UpdateBot</code> operation to update tags. To
   * update tags, use the <code>TagResource</code> operation.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetBotTags() const { return m_botTags; }
  inline bool BotTagsHasBeenSet() const { return m_botTagsHasBeenSet; }
  template <typename BotTagsT = Aws::Map<Aws::String, Aws::String>>
  void SetBotTags(BotTagsT&& value) {
    m_botTagsHasBeenSet = true;
    m_botTags = std::forward<BotTagsT>(value);
  }
  template <typename BotTagsT = Aws::Map<Aws::String, Aws::String>>
  CreateBotRequest& WithBotTags(BotTagsT&& value) {
    SetBotTags(std::forward<BotTagsT>(value));
    return *this;
  }
  template <typename BotTagsKeyT = Aws::String, typename BotTagsValueT = Aws::String>
  CreateBotRequest& AddBotTags(BotTagsKeyT&& key, BotTagsValueT&& value) {
    m_botTagsHasBeenSet = true;
    m_botTags.emplace(std::forward<BotTagsKeyT>(key), std::forward<BotTagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to add to the test alias for a bot. You can only add tags when
   * you create a bot. You can't use the <code>UpdateAlias</code> operation to update
   * tags. To update tags on the test alias, use the <code>TagResource</code>
   * operation.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTestBotAliasTags() const { return m_testBotAliasTags; }
  inline bool TestBotAliasTagsHasBeenSet() const { return m_testBotAliasTagsHasBeenSet; }
  template <typename TestBotAliasTagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTestBotAliasTags(TestBotAliasTagsT&& value) {
    m_testBotAliasTagsHasBeenSet = true;
    m_testBotAliasTags = std::forward<TestBotAliasTagsT>(value);
  }
  template <typename TestBotAliasTagsT = Aws::Map<Aws::String, Aws::String>>
  CreateBotRequest& WithTestBotAliasTags(TestBotAliasTagsT&& value) {
    SetTestBotAliasTags(std::forward<TestBotAliasTagsT>(value));
    return *this;
  }
  template <typename TestBotAliasTagsKeyT = Aws::String, typename TestBotAliasTagsValueT = Aws::String>
  CreateBotRequest& AddTestBotAliasTags(TestBotAliasTagsKeyT&& key, TestBotAliasTagsValueT&& value) {
    m_testBotAliasTagsHasBeenSet = true;
    m_testBotAliasTags.emplace(std::forward<TestBotAliasTagsKeyT>(key), std::forward<TestBotAliasTagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of a bot to create.</p>
   */
  inline BotType GetBotType() const { return m_botType; }
  inline bool BotTypeHasBeenSet() const { return m_botTypeHasBeenSet; }
  inline void SetBotType(BotType value) {
    m_botTypeHasBeenSet = true;
    m_botType = value;
  }
  inline CreateBotRequest& WithBotType(BotType value) {
    SetBotType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of bot members in a network to be created.</p>
   */
  inline const Aws::Vector<BotMember>& GetBotMembers() const { return m_botMembers; }
  inline bool BotMembersHasBeenSet() const { return m_botMembersHasBeenSet; }
  template <typename BotMembersT = Aws::Vector<BotMember>>
  void SetBotMembers(BotMembersT&& value) {
    m_botMembersHasBeenSet = true;
    m_botMembers = std::forward<BotMembersT>(value);
  }
  template <typename BotMembersT = Aws::Vector<BotMember>>
  CreateBotRequest& WithBotMembers(BotMembersT&& value) {
    SetBotMembers(std::forward<BotMembersT>(value));
    return *this;
  }
  template <typename BotMembersT = BotMember>
  CreateBotRequest& AddBotMembers(BotMembersT&& value) {
    m_botMembersHasBeenSet = true;
    m_botMembers.emplace_back(std::forward<BotMembersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the configuration for error logging during bot creation.</p>
   */
  inline const ErrorLogSettings& GetErrorLogSettings() const { return m_errorLogSettings; }
  inline bool ErrorLogSettingsHasBeenSet() const { return m_errorLogSettingsHasBeenSet; }
  template <typename ErrorLogSettingsT = ErrorLogSettings>
  void SetErrorLogSettings(ErrorLogSettingsT&& value) {
    m_errorLogSettingsHasBeenSet = true;
    m_errorLogSettings = std::forward<ErrorLogSettingsT>(value);
  }
  template <typename ErrorLogSettingsT = ErrorLogSettings>
  CreateBotRequest& WithErrorLogSettings(ErrorLogSettingsT&& value) {
    SetErrorLogSettings(std::forward<ErrorLogSettingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_botName;

  Aws::String m_description;

  Aws::String m_roleArn;

  DataPrivacy m_dataPrivacy;

  int m_idleSessionTTLInSeconds{0};

  Aws::Map<Aws::String, Aws::String> m_botTags;

  Aws::Map<Aws::String, Aws::String> m_testBotAliasTags;

  BotType m_botType{BotType::NOT_SET};

  Aws::Vector<BotMember> m_botMembers;

  ErrorLogSettings m_errorLogSettings;
  bool m_botNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_dataPrivacyHasBeenSet = false;
  bool m_idleSessionTTLInSecondsHasBeenSet = false;
  bool m_botTagsHasBeenSet = false;
  bool m_testBotAliasTagsHasBeenSet = false;
  bool m_botTypeHasBeenSet = false;
  bool m_botMembersHasBeenSet = false;
  bool m_errorLogSettingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelsV2
}  // namespace Aws
