﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/ActivationState.h>
#include <aws/sagemaker/model/ComputeQuotaConfig.h>
#include <aws/sagemaker/model/ComputeQuotaTarget.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateComputeQuotaRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateComputeQuotaRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateComputeQuota"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Name to the compute allocation definition.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateComputeQuotaRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Description of the compute allocation definition.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateComputeQuotaRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of the cluster.</p>
   */
  inline const Aws::String& GetClusterArn() const { return m_clusterArn; }
  inline bool ClusterArnHasBeenSet() const { return m_clusterArnHasBeenSet; }
  template <typename ClusterArnT = Aws::String>
  void SetClusterArn(ClusterArnT&& value) {
    m_clusterArnHasBeenSet = true;
    m_clusterArn = std::forward<ClusterArnT>(value);
  }
  template <typename ClusterArnT = Aws::String>
  CreateComputeQuotaRequest& WithClusterArn(ClusterArnT&& value) {
    SetClusterArn(std::forward<ClusterArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration of the compute allocation definition. This includes the
   * resource sharing option, and the setting to preempt low priority tasks.</p>
   */
  inline const ComputeQuotaConfig& GetComputeQuotaConfig() const { return m_computeQuotaConfig; }
  inline bool ComputeQuotaConfigHasBeenSet() const { return m_computeQuotaConfigHasBeenSet; }
  template <typename ComputeQuotaConfigT = ComputeQuotaConfig>
  void SetComputeQuotaConfig(ComputeQuotaConfigT&& value) {
    m_computeQuotaConfigHasBeenSet = true;
    m_computeQuotaConfig = std::forward<ComputeQuotaConfigT>(value);
  }
  template <typename ComputeQuotaConfigT = ComputeQuotaConfig>
  CreateComputeQuotaRequest& WithComputeQuotaConfig(ComputeQuotaConfigT&& value) {
    SetComputeQuotaConfig(std::forward<ComputeQuotaConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The target entity to allocate compute resources to.</p>
   */
  inline const ComputeQuotaTarget& GetComputeQuotaTarget() const { return m_computeQuotaTarget; }
  inline bool ComputeQuotaTargetHasBeenSet() const { return m_computeQuotaTargetHasBeenSet; }
  template <typename ComputeQuotaTargetT = ComputeQuotaTarget>
  void SetComputeQuotaTarget(ComputeQuotaTargetT&& value) {
    m_computeQuotaTargetHasBeenSet = true;
    m_computeQuotaTarget = std::forward<ComputeQuotaTargetT>(value);
  }
  template <typename ComputeQuotaTargetT = ComputeQuotaTarget>
  CreateComputeQuotaRequest& WithComputeQuotaTarget(ComputeQuotaTargetT&& value) {
    SetComputeQuotaTarget(std::forward<ComputeQuotaTargetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the compute allocation being described. Use to enable or disable
   * compute allocation.</p> <p>Default is <code>Enabled</code>.</p>
   */
  inline ActivationState GetActivationState() const { return m_activationState; }
  inline bool ActivationStateHasBeenSet() const { return m_activationStateHasBeenSet; }
  inline void SetActivationState(ActivationState value) {
    m_activationStateHasBeenSet = true;
    m_activationState = value;
  }
  inline CreateComputeQuotaRequest& WithActivationState(ActivationState value) {
    SetActivationState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags of the compute allocation definition.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateComputeQuotaRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateComputeQuotaRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_description;

  Aws::String m_clusterArn;

  ComputeQuotaConfig m_computeQuotaConfig;

  ComputeQuotaTarget m_computeQuotaTarget;

  ActivationState m_activationState{ActivationState::NOT_SET};

  Aws::Vector<Tag> m_tags;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_clusterArnHasBeenSet = false;
  bool m_computeQuotaConfigHasBeenSet = false;
  bool m_computeQuotaTargetHasBeenSet = false;
  bool m_activationStateHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
