// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package perm

import (
	"context"

	"github.com/drone/drone/core"
	"github.com/drone/drone/store/shared/db"
)

// New returns a new PermStore.
func New(db *db.DB) core.PermStore {
	return &permStore{db}
}

type permStore struct {
	db *db.DB
}

// Find returns a project member from the datastore.
func (s *permStore) Find(ctx context.Context, repo string, user int64) (*core.Perm, error) {
	out := &core.Perm{RepoUID: repo, UserID: user}
	err := s.db.View(func(queryer db.Queryer, binder db.Binder) error {
		params := toParams(out)
		query, args, err := binder.BindNamed(queryKey, params)
		if err != nil {
			return err
		}
		row := queryer.QueryRow(query, args...)
		return scanRow(row, out)
	})
	return out, err
}

// List returns a list of project members from the datastore.
func (s *permStore) List(ctx context.Context, repo string) ([]*core.Collaborator, error) {
	var out []*core.Collaborator
	err := s.db.View(func(queryer db.Queryer, binder db.Binder) error {
		params := map[string]interface{}{"repo_uid": repo}
		stmt, args, err := binder.BindNamed(queryCollabs, params)
		if err != nil {
			return err
		}
		rows, err := queryer.Query(stmt, args...)
		if err != nil {
			return err
		}
		out, err = scanCollabRows(rows)
		return err
	})
	return out, err
}

// Create persists a project member to the datastore.
func (s *permStore) Create(ctx context.Context, perm *core.Perm) error {
	return s.db.Lock(func(execer db.Execer, binder db.Binder) error {
		params := toParams(perm)
		stmt, args, err := binder.BindNamed(stmtInsert, params)
		if err != nil {
			return err
		}
		_, err = execer.Exec(stmt, args...)
		return err
	})
}

// Update persists an updated project member to the datastore.
func (s *permStore) Update(ctx context.Context, perm *core.Perm) error {
	return s.db.Lock(func(execer db.Execer, binder db.Binder) error {
		params := toParams(perm)
		stmt, args, err := binder.BindNamed(stmtUpdate, params)
		if err != nil {
			return err
		}
		_, err = execer.Exec(stmt, args...)
		return err
	})
}

// Delete deletes a project member from the datastore.
func (s *permStore) Delete(ctx context.Context, perm *core.Perm) error {
	return s.db.Lock(func(execer db.Execer, binder db.Binder) error {
		params := toParams(perm)
		stmt, args, err := binder.BindNamed(stmtDelete, params)
		if err != nil {
			return err
		}
		_, err = execer.Exec(stmt, args...)
		return err
	})
}

const queryKey = `
SELECT
 perm_user_id
,perm_repo_uid
,perm_read
,perm_write
,perm_admin
,perm_synced
,perm_created
,perm_updated
FROM perms
WHERE perm_user_id  = :perm_user_id
  AND perm_repo_uid = :perm_repo_uid
`

const queryCollabs = `
SELECT
 perm_user_id
,perm_repo_uid
,user_login
,user_avatar
,perm_read
,perm_write
,perm_admin
,perm_synced
,perm_created
,perm_updated
FROM users
INNER JOIN perms ON perms.perm_user_id = users.user_id
WHERE perms.perm_repo_uid = :repo_uid
ORDER BY user_login ASC
`

const stmtDelete = `
DELETE FROM perms
WHERE perm_user_id  = :perm_user_id
  AND perm_repo_uid = :perm_repo_uid
`

const stmtUpdate = `
UPDATE perms
SET
 perm_read = :perm_read
,perm_write = :perm_write
,perm_admin = :perm_admin
,perm_synced = :perm_synced
,perm_updated = :perm_updated
WHERE perm_user_id = :perm_user_id
  AND perm_repo_uid = :perm_repo_uid
`

const stmtInsert = `
INSERT INTO perms (
 perm_user_id
,perm_repo_uid
,perm_read
,perm_write
,perm_admin
,perm_synced
,perm_created
,perm_updated
) VALUES (
 :perm_user_id
,:perm_repo_uid
,:perm_read
,:perm_write
,:perm_admin
,:perm_synced
,:perm_created
,:perm_updated
)
`
