{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A rectangular region with rounded corners.
-- 
-- Application code should normalize rectangles using
-- 'GI.Gsk.Structs.RoundedRect.roundedRectNormalize'; this function will ensure that
-- the bounds of the rectangle are normalized and ensure that the corner
-- values are positive and the corners do not overlap.
-- 
-- All functions taking a @GskRoundedRect@ as an argument will internally
-- operate on a normalized copy; all functions returning a @GskRoundedRect@
-- will always return a normalized one.
-- 
-- The algorithm used for normalizing corner sizes is described in
-- <https://drafts.csswg.org/css-backgrounds-3/#border-radius the CSS specification>.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Structs.RoundedRect
    ( 

-- * Exported types
    RoundedRect(..)                         ,
    newZeroRoundedRect                      ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [containsPoint]("GI.Gsk.Structs.RoundedRect#g:method:containsPoint"), [containsRect]("GI.Gsk.Structs.RoundedRect#g:method:containsRect"), [init]("GI.Gsk.Structs.RoundedRect#g:method:init"), [initCopy]("GI.Gsk.Structs.RoundedRect#g:method:initCopy"), [initFromRect]("GI.Gsk.Structs.RoundedRect#g:method:initFromRect"), [intersectsRect]("GI.Gsk.Structs.RoundedRect#g:method:intersectsRect"), [isRectilinear]("GI.Gsk.Structs.RoundedRect#g:method:isRectilinear"), [normalize]("GI.Gsk.Structs.RoundedRect#g:method:normalize"), [offset]("GI.Gsk.Structs.RoundedRect#g:method:offset"), [shrink]("GI.Gsk.Structs.RoundedRect#g:method:shrink").
-- 
-- ==== Getters
-- /None/.
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveRoundedRectMethod                ,
#endif

-- ** containsPoint #method:containsPoint#

#if defined(ENABLE_OVERLOADING)
    RoundedRectContainsPointMethodInfo      ,
#endif
    roundedRectContainsPoint                ,


-- ** containsRect #method:containsRect#

#if defined(ENABLE_OVERLOADING)
    RoundedRectContainsRectMethodInfo       ,
#endif
    roundedRectContainsRect                 ,


-- ** init #method:init#

#if defined(ENABLE_OVERLOADING)
    RoundedRectInitMethodInfo               ,
#endif
    roundedRectInit                         ,


-- ** initCopy #method:initCopy#

#if defined(ENABLE_OVERLOADING)
    RoundedRectInitCopyMethodInfo           ,
#endif
    roundedRectInitCopy                     ,


-- ** initFromRect #method:initFromRect#

#if defined(ENABLE_OVERLOADING)
    RoundedRectInitFromRectMethodInfo       ,
#endif
    roundedRectInitFromRect                 ,


-- ** intersectsRect #method:intersectsRect#

#if defined(ENABLE_OVERLOADING)
    RoundedRectIntersectsRectMethodInfo     ,
#endif
    roundedRectIntersectsRect               ,


-- ** isRectilinear #method:isRectilinear#

#if defined(ENABLE_OVERLOADING)
    RoundedRectIsRectilinearMethodInfo      ,
#endif
    roundedRectIsRectilinear                ,


-- ** normalize #method:normalize#

#if defined(ENABLE_OVERLOADING)
    RoundedRectNormalizeMethodInfo          ,
#endif
    roundedRectNormalize                    ,


-- ** offset #method:offset#

#if defined(ENABLE_OVERLOADING)
    RoundedRectOffsetMethodInfo             ,
#endif
    roundedRectOffset                       ,


-- ** shrink #method:shrink#

#if defined(ENABLE_OVERLOADING)
    RoundedRectShrinkMethodInfo             ,
#endif
    roundedRectShrink                       ,




 -- * Properties


-- ** bounds #attr:bounds#
-- | the bounds of the rectangle

    getRoundedRectBounds                    ,
#if defined(ENABLE_OVERLOADING)
    roundedRect_bounds                      ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size

#else
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size

#endif

-- | Memory-managed wrapper type.
newtype RoundedRect = RoundedRect (SP.ManagedPtr RoundedRect)
    deriving (Eq)

instance SP.ManagedPtrNewtype RoundedRect where
    toManagedPtr (RoundedRect p) = p

instance BoxedPtr RoundedRect where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 48 >=> B.ManagedPtr.wrapPtr RoundedRect)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr RoundedRect where
    boxedPtrCalloc = callocBytes 48


-- | Construct a t'RoundedRect' struct initialized to zero.
newZeroRoundedRect :: MonadIO m => m RoundedRect
newZeroRoundedRect = liftIO $ boxedPtrCalloc >>= wrapPtr RoundedRect

instance tag ~ 'AttrSet => Constructible RoundedRect tag where
    new _ attrs = do
        o <- newZeroRoundedRect
        GI.Attributes.set o attrs
        return o


-- | Get the value of the “@bounds@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' roundedRect #bounds
-- @
getRoundedRectBounds :: MonadIO m => RoundedRect -> m Graphene.Rect.Rect
getRoundedRectBounds s = liftIO $ withManagedPtr s $ \ptr -> do
    let val = ptr `plusPtr` 0 :: (Ptr Graphene.Rect.Rect)
    val' <- (newBoxed Graphene.Rect.Rect) val
    return val'

#if defined(ENABLE_OVERLOADING)
data RoundedRectBoundsFieldInfo
instance AttrInfo RoundedRectBoundsFieldInfo where
    type AttrBaseTypeConstraint RoundedRectBoundsFieldInfo = (~) RoundedRect
    type AttrAllowedOps RoundedRectBoundsFieldInfo = '[ 'AttrGet]
    type AttrSetTypeConstraint RoundedRectBoundsFieldInfo = (~) (Ptr Graphene.Rect.Rect)
    type AttrTransferTypeConstraint RoundedRectBoundsFieldInfo = (~)(Ptr Graphene.Rect.Rect)
    type AttrTransferType RoundedRectBoundsFieldInfo = (Ptr Graphene.Rect.Rect)
    type AttrGetType RoundedRectBoundsFieldInfo = Graphene.Rect.Rect
    type AttrLabel RoundedRectBoundsFieldInfo = "bounds"
    type AttrOrigin RoundedRectBoundsFieldInfo = RoundedRect
    attrGet = getRoundedRectBounds
    attrSet = undefined
    attrConstruct = undefined
    attrClear = undefined
    attrTransfer = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.bounds"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#g:attr:bounds"
        })

roundedRect_bounds :: AttrLabelProxy "bounds"
roundedRect_bounds = AttrLabelProxy

#endif


-- XXX Skipped attribute for "RoundedRect:corner"
-- Not implemented: Don't know how to unpack C array of type TCArray False 4 (-1) (TInterface (Name {namespace = "Graphene", name = "Size"}))

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList RoundedRect
type instance O.AttributeList RoundedRect = RoundedRectAttributeList
type RoundedRectAttributeList = ('[ '("bounds", RoundedRectBoundsFieldInfo)] :: [(Symbol, DK.Type)])
#endif

-- method RoundedRect::contains_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the point to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_contains_point" gsk_rounded_rect_contains_point :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr Graphene.Point.Point ->             -- point : TInterface (Name {namespace = "Graphene", name = "Point"})
    IO CInt

-- | Checks if the given point is inside the rounded rectangle.
roundedRectContainsPoint ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: a rounded rectangle
    -> Graphene.Point.Point
    -- ^ /@point@/: the point to check
    -> m Bool
    -- ^ __Returns:__ true if the point is inside the rounded rectangle
roundedRectContainsPoint self point = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    point' <- unsafeManagedPtrGetPtr point
    result <- gsk_rounded_rect_contains_point self' point'
    let result' = (/= 0) result
    touchManagedPtr self
    touchManagedPtr point
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectContainsPointMethodInfo
instance (signature ~ (Graphene.Point.Point -> m Bool), MonadIO m) => O.OverloadedMethod RoundedRectContainsPointMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectContainsPoint

instance O.OverloadedMethodInfo RoundedRectContainsPointMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectContainsPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectContainsPoint"
        })


#endif

-- method RoundedRect::contains_rect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rectangle to check"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_contains_rect" gsk_rounded_rect_contains_rect :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr Graphene.Rect.Rect ->               -- rect : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO CInt

-- | Checks if the given rectangle is contained inside the rounded rectangle.
roundedRectContainsRect ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: a rounded rectangle
    -> Graphene.Rect.Rect
    -- ^ /@rect@/: the rectangle to check
    -> m Bool
    -- ^ __Returns:__ true if the /@rect@/ is fully contained inside the rounded rectangle
roundedRectContainsRect self rect = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    rect' <- unsafeManagedPtrGetPtr rect
    result <- gsk_rounded_rect_contains_rect self' rect'
    let result' = (/= 0) result
    touchManagedPtr self
    touchManagedPtr rect
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectContainsRectMethodInfo
instance (signature ~ (Graphene.Rect.Rect -> m Bool), MonadIO m) => O.OverloadedMethod RoundedRectContainsRectMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectContainsRect

instance O.OverloadedMethodInfo RoundedRectContainsRectMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectContainsRect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectContainsRect"
        })


#endif

-- method RoundedRect::init
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounded rectangle to initialize"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bounds"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `graphene_rect_t` describing the bounds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "top_left"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Size" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounding radius of the top left corner"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "top_right"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Size" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounding radius of the top right corner"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bottom_right"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Size" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the rounding radius of the bottom right corner"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bottom_left"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Size" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounding radius of the bottom left corner"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_init" gsk_rounded_rect_init :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr Graphene.Rect.Rect ->               -- bounds : TInterface (Name {namespace = "Graphene", name = "Rect"})
    Ptr Graphene.Size.Size ->               -- top_left : TInterface (Name {namespace = "Graphene", name = "Size"})
    Ptr Graphene.Size.Size ->               -- top_right : TInterface (Name {namespace = "Graphene", name = "Size"})
    Ptr Graphene.Size.Size ->               -- bottom_right : TInterface (Name {namespace = "Graphene", name = "Size"})
    Ptr Graphene.Size.Size ->               -- bottom_left : TInterface (Name {namespace = "Graphene", name = "Size"})
    IO (Ptr RoundedRect)

-- | Initializes a rounded rectangle with the given values.
-- 
-- This function will implicitly normalize the rounded rectangle
-- before returning.
roundedRectInit ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: the rounded rectangle to initialize
    -> Graphene.Rect.Rect
    -- ^ /@bounds@/: a @graphene_rect_t@ describing the bounds
    -> Graphene.Size.Size
    -- ^ /@topLeft@/: the rounding radius of the top left corner
    -> Graphene.Size.Size
    -- ^ /@topRight@/: the rounding radius of the top right corner
    -> Graphene.Size.Size
    -- ^ /@bottomRight@/: the rounding radius of the bottom right corner
    -> Graphene.Size.Size
    -- ^ /@bottomLeft@/: the rounding radius of the bottom left corner
    -> m RoundedRect
    -- ^ __Returns:__ the initialized rounded rectangle
roundedRectInit self bounds topLeft topRight bottomRight bottomLeft = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    bounds' <- unsafeManagedPtrGetPtr bounds
    topLeft' <- unsafeManagedPtrGetPtr topLeft
    topRight' <- unsafeManagedPtrGetPtr topRight
    bottomRight' <- unsafeManagedPtrGetPtr bottomRight
    bottomLeft' <- unsafeManagedPtrGetPtr bottomLeft
    result <- gsk_rounded_rect_init self' bounds' topLeft' topRight' bottomRight' bottomLeft'
    checkUnexpectedReturnNULL "roundedRectInit" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    touchManagedPtr bounds
    touchManagedPtr topLeft
    touchManagedPtr topRight
    touchManagedPtr bottomRight
    touchManagedPtr bottomLeft
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectInitMethodInfo
instance (signature ~ (Graphene.Rect.Rect -> Graphene.Size.Size -> Graphene.Size.Size -> Graphene.Size.Size -> Graphene.Size.Size -> m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectInitMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectInit

instance O.OverloadedMethodInfo RoundedRectInitMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectInit",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectInit"
        })


#endif

-- method RoundedRect::init_copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounded rectangle to initialize"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "src"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_init_copy" gsk_rounded_rect_init_copy :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr RoundedRect ->                      -- src : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    IO (Ptr RoundedRect)

-- | Initializes a rounded rectangle with a copy.
-- 
-- This function will not normalize the rounded rectangle,
-- so make sure the source is normalized.
roundedRectInitCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: the rounded rectangle to initialize
    -> RoundedRect
    -- ^ /@src@/: another rounded rectangle
    -> m RoundedRect
    -- ^ __Returns:__ the initialized rounded rectangle
roundedRectInitCopy self src = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    src' <- unsafeManagedPtrGetPtr src
    result <- gsk_rounded_rect_init_copy self' src'
    checkUnexpectedReturnNULL "roundedRectInitCopy" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    touchManagedPtr src
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectInitCopyMethodInfo
instance (signature ~ (RoundedRect -> m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectInitCopyMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectInitCopy

instance O.OverloadedMethodInfo RoundedRectInitCopyMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectInitCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectInitCopy"
        })


#endif

-- method RoundedRect::init_from_rect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounded rectangle to initialize"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bounds"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `graphene_rect_t`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "radius"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the border radius" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_init_from_rect" gsk_rounded_rect_init_from_rect :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr Graphene.Rect.Rect ->               -- bounds : TInterface (Name {namespace = "Graphene", name = "Rect"})
    CFloat ->                               -- radius : TBasicType TFloat
    IO (Ptr RoundedRect)

-- | Initializes a rounded rectangle to the given bounds
-- and sets the radius of all four corners equally.
roundedRectInitFromRect ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: the rounded rectangle to initialize
    -> Graphene.Rect.Rect
    -- ^ /@bounds@/: a @graphene_rect_t@
    -> Float
    -- ^ /@radius@/: the border radius
    -> m RoundedRect
    -- ^ __Returns:__ the initialized rounded rectangle
roundedRectInitFromRect self bounds radius = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    bounds' <- unsafeManagedPtrGetPtr bounds
    let radius' = realToFrac radius
    result <- gsk_rounded_rect_init_from_rect self' bounds' radius'
    checkUnexpectedReturnNULL "roundedRectInitFromRect" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    touchManagedPtr bounds
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectInitFromRectMethodInfo
instance (signature ~ (Graphene.Rect.Rect -> Float -> m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectInitFromRectMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectInitFromRect

instance O.OverloadedMethodInfo RoundedRectInitFromRectMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectInitFromRect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectInitFromRect"
        })


#endif

-- method RoundedRect::intersects_rect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rectangle to check"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_intersects_rect" gsk_rounded_rect_intersects_rect :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    Ptr Graphene.Rect.Rect ->               -- rect : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO CInt

-- | Checks if part a rectangle is contained
-- inside the rounded rectangle.
roundedRectIntersectsRect ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: a rounded rectangle
    -> Graphene.Rect.Rect
    -- ^ /@rect@/: the rectangle to check
    -> m Bool
    -- ^ __Returns:__ true if the /@rect@/ intersects with the rounded rectangle
roundedRectIntersectsRect self rect = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    rect' <- unsafeManagedPtrGetPtr rect
    result <- gsk_rounded_rect_intersects_rect self' rect'
    let result' = (/= 0) result
    touchManagedPtr self
    touchManagedPtr rect
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectIntersectsRectMethodInfo
instance (signature ~ (Graphene.Rect.Rect -> m Bool), MonadIO m) => O.OverloadedMethod RoundedRectIntersectsRectMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectIntersectsRect

instance O.OverloadedMethodInfo RoundedRectIntersectsRectMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectIntersectsRect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectIntersectsRect"
        })


#endif

-- method RoundedRect::is_rectilinear
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounded rectangle to check"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_is_rectilinear" gsk_rounded_rect_is_rectilinear :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    IO CInt

-- | Checks if all corners of a rounded rectangle are right angles
-- and the rectangle covers all of its bounds.
-- 
-- This information can be used to decide if 'GI.Gsk.Objects.ClipNode.clipNodeNew'
-- or 'GI.Gsk.Objects.RoundedClipNode.roundedClipNodeNew' should be called.
roundedRectIsRectilinear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: the rounded rectangle to check
    -> m Bool
    -- ^ __Returns:__ true if the rounded rectangle is rectilinear
roundedRectIsRectilinear self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_rounded_rect_is_rectilinear self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectIsRectilinearMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod RoundedRectIsRectilinearMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectIsRectilinear

instance O.OverloadedMethodInfo RoundedRectIsRectilinearMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectIsRectilinear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectIsRectilinear"
        })


#endif

-- method RoundedRect::normalize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_normalize" gsk_rounded_rect_normalize :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    IO (Ptr RoundedRect)

-- | Normalizes a rounded rectangle.
-- 
-- This function will ensure that the bounds of the rounded rectangle
-- are normalized and ensure that the corner values are positive
-- and the corners do not overlap.
roundedRectNormalize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: a rounded rectangle
    -> m RoundedRect
    -- ^ __Returns:__ the normalized rounded rectangle
roundedRectNormalize self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_rounded_rect_normalize self'
    checkUnexpectedReturnNULL "roundedRectNormalize" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectNormalizeMethodInfo
instance (signature ~ (m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectNormalizeMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectNormalize

instance O.OverloadedMethodInfo RoundedRectNormalizeMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectNormalize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectNormalize"
        })


#endif

-- method RoundedRect::offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rounded rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the horizontal offset"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the vertical offset"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_offset" gsk_rounded_rect_offset :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    CFloat ->                               -- dx : TBasicType TFloat
    CFloat ->                               -- dy : TBasicType TFloat
    IO (Ptr RoundedRect)

-- | Offsets the rounded rectangle\'s origin by /@dx@/ and /@dy@/.
-- 
-- The size and corners of the rounded rectangle are unchanged.
roundedRectOffset ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: a rounded rectangle
    -> Float
    -- ^ /@dx@/: the horizontal offset
    -> Float
    -- ^ /@dy@/: the vertical offset
    -> m RoundedRect
    -- ^ __Returns:__ the offset rounded rectangle
roundedRectOffset self dx dy = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    let dx' = realToFrac dx
    let dy' = realToFrac dy
    result <- gsk_rounded_rect_offset self' dx' dy'
    checkUnexpectedReturnNULL "roundedRectOffset" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectOffsetMethodInfo
instance (signature ~ (Float -> Float -> m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectOffsetMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectOffset

instance O.OverloadedMethodInfo RoundedRectOffsetMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectOffset"
        })


#endif

-- method RoundedRect::shrink
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RoundedRect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rounded rectangle to shrink or grow"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "top"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "how far to move the top side downwards"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "right"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "how far to move the right side to the left"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bottom"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "how far to move the bottom side upwards"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "left"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "how far to move the left side to the right"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RoundedRect" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_rounded_rect_shrink" gsk_rounded_rect_shrink :: 
    Ptr RoundedRect ->                      -- self : TInterface (Name {namespace = "Gsk", name = "RoundedRect"})
    CFloat ->                               -- top : TBasicType TFloat
    CFloat ->                               -- right : TBasicType TFloat
    CFloat ->                               -- bottom : TBasicType TFloat
    CFloat ->                               -- left : TBasicType TFloat
    IO (Ptr RoundedRect)

-- | Shrinks (or grows) a rounded rectangle by moving the 4 sides
-- according to the offsets given.
-- 
-- The corner radii will be changed in a way that tries to keep
-- the center of the corner circle intact. This emulates CSS behavior.
-- 
-- This function also works for growing rounded rectangles
-- if you pass negative values for the /@top@/, /@right@/, /@bottom@/ or /@left@/.
roundedRectShrink ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    RoundedRect
    -- ^ /@self@/: the rounded rectangle to shrink or grow
    -> Float
    -- ^ /@top@/: how far to move the top side downwards
    -> Float
    -- ^ /@right@/: how far to move the right side to the left
    -> Float
    -- ^ /@bottom@/: how far to move the bottom side upwards
    -> Float
    -- ^ /@left@/: how far to move the left side to the right
    -> m RoundedRect
    -- ^ __Returns:__ the resized rounded rectangle
roundedRectShrink self top right bottom left = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    let top' = realToFrac top
    let right' = realToFrac right
    let bottom' = realToFrac bottom
    let left' = realToFrac left
    result <- gsk_rounded_rect_shrink self' top' right' bottom' left'
    checkUnexpectedReturnNULL "roundedRectShrink" result
    result' <- (newPtr RoundedRect) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data RoundedRectShrinkMethodInfo
instance (signature ~ (Float -> Float -> Float -> Float -> m RoundedRect), MonadIO m) => O.OverloadedMethod RoundedRectShrinkMethodInfo RoundedRect signature where
    overloadedMethod = roundedRectShrink

instance O.OverloadedMethodInfo RoundedRectShrinkMethodInfo RoundedRect where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.RoundedRect.roundedRectShrink",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-RoundedRect.html#v:roundedRectShrink"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveRoundedRectMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveRoundedRectMethod "containsPoint" o = RoundedRectContainsPointMethodInfo
    ResolveRoundedRectMethod "containsRect" o = RoundedRectContainsRectMethodInfo
    ResolveRoundedRectMethod "init" o = RoundedRectInitMethodInfo
    ResolveRoundedRectMethod "initCopy" o = RoundedRectInitCopyMethodInfo
    ResolveRoundedRectMethod "initFromRect" o = RoundedRectInitFromRectMethodInfo
    ResolveRoundedRectMethod "intersectsRect" o = RoundedRectIntersectsRectMethodInfo
    ResolveRoundedRectMethod "isRectilinear" o = RoundedRectIsRectilinearMethodInfo
    ResolveRoundedRectMethod "normalize" o = RoundedRectNormalizeMethodInfo
    ResolveRoundedRectMethod "offset" o = RoundedRectOffsetMethodInfo
    ResolveRoundedRectMethod "shrink" o = RoundedRectShrinkMethodInfo
    ResolveRoundedRectMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveRoundedRectMethod t RoundedRect, O.OverloadedMethod info RoundedRect p) => OL.IsLabel t (RoundedRect -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveRoundedRectMethod t RoundedRect, O.OverloadedMethod info RoundedRect p, R.HasField t RoundedRect p) => R.HasField t RoundedRect p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveRoundedRectMethod t RoundedRect, O.OverloadedMethodInfo info RoundedRect) => OL.IsLabel t (O.MethodProxy info RoundedRect) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


