

-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Pango.Enums
    ( 

 -- * Enumerations


-- ** Alignment #enum:Alignment#

    Alignment(..)                           ,


-- ** AttrType #enum:AttrType#

    AttrType(..)                            ,


-- ** BaselineShift #enum:BaselineShift#

    BaselineShift(..)                       ,


-- ** BidiType #enum:BidiType#

    BidiType(..)                            ,


-- ** CoverageLevel #enum:CoverageLevel#

    CoverageLevel(..)                       ,


-- ** Direction #enum:Direction#

    Direction(..)                           ,


-- ** EllipsizeMode #enum:EllipsizeMode#

    EllipsizeMode(..)                       ,


-- ** FontColor #enum:FontColor#

    FontColor(..)                           ,


-- ** FontScale #enum:FontScale#

    FontScale(..)                           ,


-- ** Gravity #enum:Gravity#

    Gravity(..)                             ,


-- ** GravityHint #enum:GravityHint#

    GravityHint(..)                         ,


-- ** LayoutDeserializeError #enum:LayoutDeserializeError#

    LayoutDeserializeError(..)              ,
    catchLayoutDeserializeError             ,
    handleLayoutDeserializeError            ,


-- ** Overline #enum:Overline#

    Overline(..)                            ,


-- ** RenderPart #enum:RenderPart#

    RenderPart(..)                          ,


-- ** Script #enum:Script#

    Script(..)                              ,


-- ** Stretch #enum:Stretch#

    Stretch(..)                             ,


-- ** Style #enum:Style#

    Style(..)                               ,


-- ** TabAlign #enum:TabAlign#

    TabAlign(..)                            ,


-- ** TextTransform #enum:TextTransform#

    TextTransform(..)                       ,


-- ** Underline #enum:Underline#

    Underline(..)                           ,


-- ** Variant #enum:Variant#

    Variant(..)                             ,


-- ** Weight #enum:Weight#

    Weight(..)                              ,


-- ** WrapMode #enum:WrapMode#

    WrapMode(..)                            ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- Enum WrapMode
-- | @PangoWrapMode@ describes how to wrap the lines of a @PangoLayout@
-- to the desired width.
-- 
-- For /@pANGOWRAPWORD@/, Pango uses break opportunities that are determined
-- by the Unicode line breaking algorithm. For /@pANGOWRAPCHAR@/, Pango allows
-- breaking at grapheme boundaries that are determined by the Unicode text
-- segmentation algorithm.
data WrapMode = 
      WrapModeWord
    -- ^ wrap lines at word boundaries.
    | WrapModeChar
    -- ^ wrap lines at character boundaries.
    | WrapModeWordChar
    -- ^ wrap lines at word boundaries, but fall back to
    --   character boundaries if there is not enough space for a full word.
    | WrapModeNone
    -- ^ do not wrap.
    -- 
    -- /Since: 1.56/
    | AnotherWrapMode Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum WrapMode where
    fromEnum WrapModeWord = 0
    fromEnum WrapModeChar = 1
    fromEnum WrapModeWordChar = 2
    fromEnum WrapModeNone = 3
    fromEnum (AnotherWrapMode k) = k

    toEnum 0 = WrapModeWord
    toEnum 1 = WrapModeChar
    toEnum 2 = WrapModeWordChar
    toEnum 3 = WrapModeNone
    toEnum k = AnotherWrapMode k

instance P.Ord WrapMode where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes WrapMode = '[]
instance O.HasParentTypes WrapMode

foreign import ccall "pango_wrap_mode_get_type" c_pango_wrap_mode_get_type :: 
    IO GType

instance B.Types.TypedObject WrapMode where
    glibType = c_pango_wrap_mode_get_type

instance B.Types.BoxedEnum WrapMode

-- Enum Weight
-- | An enumeration specifying the weight (boldness) of a font.
-- 
-- Weight is specified as a numeric value ranging from 100 to 1000.
-- This enumeration simply provides some common, predefined values.
data Weight = 
      WeightThin
    -- ^ the thin weight (= 100) Since: 1.24
    | WeightUltralight
    -- ^ the ultralight weight (= 200)
    | WeightLight
    -- ^ the light weight (= 300)
    | WeightSemilight
    -- ^ the semilight weight (= 350) Since: 1.36.7
    | WeightBook
    -- ^ the book weight (= 380) Since: 1.24)
    | WeightNormal
    -- ^ the default weight (= 400)
    | WeightMedium
    -- ^ the medium weight (= 500) Since: 1.24
    | WeightSemibold
    -- ^ the semibold weight (= 600)
    | WeightBold
    -- ^ the bold weight (= 700)
    | WeightUltrabold
    -- ^ the ultrabold weight (= 800)
    | WeightHeavy
    -- ^ the heavy weight (= 900)
    | WeightUltraheavy
    -- ^ the ultraheavy weight (= 1000) Since: 1.24
    | AnotherWeight Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Weight where
    fromEnum WeightThin = 100
    fromEnum WeightUltralight = 200
    fromEnum WeightLight = 300
    fromEnum WeightSemilight = 350
    fromEnum WeightBook = 380
    fromEnum WeightNormal = 400
    fromEnum WeightMedium = 500
    fromEnum WeightSemibold = 600
    fromEnum WeightBold = 700
    fromEnum WeightUltrabold = 800
    fromEnum WeightHeavy = 900
    fromEnum WeightUltraheavy = 1000
    fromEnum (AnotherWeight k) = k

    toEnum 100 = WeightThin
    toEnum 200 = WeightUltralight
    toEnum 300 = WeightLight
    toEnum 350 = WeightSemilight
    toEnum 380 = WeightBook
    toEnum 400 = WeightNormal
    toEnum 500 = WeightMedium
    toEnum 600 = WeightSemibold
    toEnum 700 = WeightBold
    toEnum 800 = WeightUltrabold
    toEnum 900 = WeightHeavy
    toEnum 1000 = WeightUltraheavy
    toEnum k = AnotherWeight k

instance P.Ord Weight where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Weight = '[]
instance O.HasParentTypes Weight

foreign import ccall "pango_weight_get_type" c_pango_weight_get_type :: 
    IO GType

instance B.Types.TypedObject Weight where
    glibType = c_pango_weight_get_type

instance B.Types.BoxedEnum Weight

-- Enum Variant
-- | An enumeration specifying capitalization variant of the font.
data Variant = 
      VariantNormal
    -- ^ A normal font.
    | VariantSmallCaps
    -- ^ A font with the lower case characters
    --   replaced by smaller variants of the capital characters.
    | VariantAllSmallCaps
    -- ^ A font with all characters
    --   replaced by smaller variants of the capital characters. Since: 1.50
    | VariantPetiteCaps
    -- ^ A font with the lower case characters
    --   replaced by smaller variants of the capital characters.
    --   Petite Caps can be even smaller than Small Caps. Since: 1.50
    | VariantAllPetiteCaps
    -- ^ A font with all characters
    --   replaced by smaller variants of the capital characters.
    --   Petite Caps can be even smaller than Small Caps. Since: 1.50
    | VariantUnicase
    -- ^ A font with the upper case characters
    --   replaced by smaller variants of the capital letters. Since: 1.50
    | VariantTitleCaps
    -- ^ A font with capital letters that
    --   are more suitable for all-uppercase titles. Since: 1.50
    | AnotherVariant Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Variant where
    fromEnum VariantNormal = 0
    fromEnum VariantSmallCaps = 1
    fromEnum VariantAllSmallCaps = 2
    fromEnum VariantPetiteCaps = 3
    fromEnum VariantAllPetiteCaps = 4
    fromEnum VariantUnicase = 5
    fromEnum VariantTitleCaps = 6
    fromEnum (AnotherVariant k) = k

    toEnum 0 = VariantNormal
    toEnum 1 = VariantSmallCaps
    toEnum 2 = VariantAllSmallCaps
    toEnum 3 = VariantPetiteCaps
    toEnum 4 = VariantAllPetiteCaps
    toEnum 5 = VariantUnicase
    toEnum 6 = VariantTitleCaps
    toEnum k = AnotherVariant k

instance P.Ord Variant where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Variant = '[]
instance O.HasParentTypes Variant

foreign import ccall "pango_variant_get_type" c_pango_variant_get_type :: 
    IO GType

instance B.Types.TypedObject Variant where
    glibType = c_pango_variant_get_type

instance B.Types.BoxedEnum Variant

-- Enum Underline
-- | The @PangoUnderline@ enumeration is used to specify whether text
-- should be underlined, and if so, the type of underlining.
data Underline = 
      UnderlineNone
    -- ^ no underline should be drawn
    | UnderlineSingle
    -- ^ a single underline should be drawn
    | UnderlineDouble
    -- ^ a double underline should be drawn
    | UnderlineLow
    -- ^ a single underline should be drawn at a
    --   position beneath the ink extents of the text being
    --   underlined. This should be used only for underlining
    --   single characters, such as for keyboard accelerators.
    --   'GI.Pango.Enums.UnderlineSingle' should be used for extended
    --   portions of text.
    | UnderlineError
    -- ^ an underline indicating an error should
    --   be drawn below. The exact style of rendering is up to the
    --   @PangoRenderer@ in use, but typical styles include wavy
    --   or dotted lines.
    --   This underline is typically used to indicate an error such
    --   as a possible mispelling; in some cases a contrasting color
    --   may automatically be used. This type of underlining is
    --   available since Pango 1.4.
    | UnderlineSingleLine
    -- ^ Like /@pANGOUNDERLINESINGLE@/, but
    --   drawn continuously across multiple runs. This type
    --   of underlining is available since Pango 1.46.
    | UnderlineDoubleLine
    -- ^ Like /@pANGOUNDERLINEDOUBLE@/, but
    --   drawn continuously across multiple runs. This type
    --   of underlining is available since Pango 1.46.
    | UnderlineErrorLine
    -- ^ Like /@pANGOUNDERLINEERROR@/, but
    --   drawn continuously across multiple runs. This type
    --   of underlining is available since Pango 1.46.
    | AnotherUnderline Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Underline where
    fromEnum UnderlineNone = 0
    fromEnum UnderlineSingle = 1
    fromEnum UnderlineDouble = 2
    fromEnum UnderlineLow = 3
    fromEnum UnderlineError = 4
    fromEnum UnderlineSingleLine = 5
    fromEnum UnderlineDoubleLine = 6
    fromEnum UnderlineErrorLine = 7
    fromEnum (AnotherUnderline k) = k

    toEnum 0 = UnderlineNone
    toEnum 1 = UnderlineSingle
    toEnum 2 = UnderlineDouble
    toEnum 3 = UnderlineLow
    toEnum 4 = UnderlineError
    toEnum 5 = UnderlineSingleLine
    toEnum 6 = UnderlineDoubleLine
    toEnum 7 = UnderlineErrorLine
    toEnum k = AnotherUnderline k

instance P.Ord Underline where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Underline = '[]
instance O.HasParentTypes Underline

foreign import ccall "pango_underline_get_type" c_pango_underline_get_type :: 
    IO GType

instance B.Types.TypedObject Underline where
    glibType = c_pango_underline_get_type

instance B.Types.BoxedEnum Underline

-- Enum TextTransform
-- | An enumeration that affects how Pango treats characters during shaping.
-- 
-- /Since: 1.50/
data TextTransform = 
      TextTransformNone
    -- ^ Leave text unchanged
    | TextTransformLowercase
    -- ^ Display letters and numbers as lowercase
    | TextTransformUppercase
    -- ^ Display letters and numbers as uppercase
    | TextTransformCapitalize
    -- ^ Display the first character of a word
    --   in titlecase
    | AnotherTextTransform Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum TextTransform where
    fromEnum TextTransformNone = 0
    fromEnum TextTransformLowercase = 1
    fromEnum TextTransformUppercase = 2
    fromEnum TextTransformCapitalize = 3
    fromEnum (AnotherTextTransform k) = k

    toEnum 0 = TextTransformNone
    toEnum 1 = TextTransformLowercase
    toEnum 2 = TextTransformUppercase
    toEnum 3 = TextTransformCapitalize
    toEnum k = AnotherTextTransform k

instance P.Ord TextTransform where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes TextTransform = '[]
instance O.HasParentTypes TextTransform

foreign import ccall "pango_text_transform_get_type" c_pango_text_transform_get_type :: 
    IO GType

instance B.Types.TypedObject TextTransform where
    glibType = c_pango_text_transform_get_type

instance B.Types.BoxedEnum TextTransform

-- Enum TabAlign
-- | @PangoTabAlign@ specifies where the text appears relative to the tab stop
-- position.
data TabAlign = 
      TabAlignLeft
    -- ^ the text appears to the right of the tab stop position
    | TabAlignRight
    -- ^ the text appears to the left of the tab stop position
    --   until the available space is filled. Since: 1.50
    | TabAlignCenter
    -- ^ the text is centered at the tab stop position
    --   until the available space is filled. Since: 1.50
    | TabAlignDecimal
    -- ^ text before the first occurrence of the decimal point
    --   character appears to the left of the tab stop position (until the available
    --   space is filled), the rest to the right. Since: 1.50
    | AnotherTabAlign Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum TabAlign where
    fromEnum TabAlignLeft = 0
    fromEnum TabAlignRight = 1
    fromEnum TabAlignCenter = 2
    fromEnum TabAlignDecimal = 3
    fromEnum (AnotherTabAlign k) = k

    toEnum 0 = TabAlignLeft
    toEnum 1 = TabAlignRight
    toEnum 2 = TabAlignCenter
    toEnum 3 = TabAlignDecimal
    toEnum k = AnotherTabAlign k

instance P.Ord TabAlign where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes TabAlign = '[]
instance O.HasParentTypes TabAlign

foreign import ccall "pango_tab_align_get_type" c_pango_tab_align_get_type :: 
    IO GType

instance B.Types.TypedObject TabAlign where
    glibType = c_pango_tab_align_get_type

instance B.Types.BoxedEnum TabAlign

-- Enum Style
-- | An enumeration specifying the various slant styles possible for a font.
data Style = 
      StyleNormal
    -- ^ the font is upright.
    | StyleOblique
    -- ^ the font is slanted, but in a roman style.
    | StyleItalic
    -- ^ the font is slanted in an italic style.
    | AnotherStyle Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Style where
    fromEnum StyleNormal = 0
    fromEnum StyleOblique = 1
    fromEnum StyleItalic = 2
    fromEnum (AnotherStyle k) = k

    toEnum 0 = StyleNormal
    toEnum 1 = StyleOblique
    toEnum 2 = StyleItalic
    toEnum k = AnotherStyle k

instance P.Ord Style where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Style = '[]
instance O.HasParentTypes Style

foreign import ccall "pango_style_get_type" c_pango_style_get_type :: 
    IO GType

instance B.Types.TypedObject Style where
    glibType = c_pango_style_get_type

instance B.Types.BoxedEnum Style

-- Enum Stretch
-- | An enumeration specifying the width of the font relative to other designs
-- within a family.
data Stretch = 
      StretchUltraCondensed
    -- ^ ultra condensed width
    | StretchExtraCondensed
    -- ^ extra condensed width
    | StretchCondensed
    -- ^ condensed width
    | StretchSemiCondensed
    -- ^ semi condensed width
    | StretchNormal
    -- ^ the normal width
    | StretchSemiExpanded
    -- ^ semi expanded width
    | StretchExpanded
    -- ^ expanded width
    | StretchExtraExpanded
    -- ^ extra expanded width
    | StretchUltraExpanded
    -- ^ ultra expanded width
    | AnotherStretch Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Stretch where
    fromEnum StretchUltraCondensed = 0
    fromEnum StretchExtraCondensed = 1
    fromEnum StretchCondensed = 2
    fromEnum StretchSemiCondensed = 3
    fromEnum StretchNormal = 4
    fromEnum StretchSemiExpanded = 5
    fromEnum StretchExpanded = 6
    fromEnum StretchExtraExpanded = 7
    fromEnum StretchUltraExpanded = 8
    fromEnum (AnotherStretch k) = k

    toEnum 0 = StretchUltraCondensed
    toEnum 1 = StretchExtraCondensed
    toEnum 2 = StretchCondensed
    toEnum 3 = StretchSemiCondensed
    toEnum 4 = StretchNormal
    toEnum 5 = StretchSemiExpanded
    toEnum 6 = StretchExpanded
    toEnum 7 = StretchExtraExpanded
    toEnum 8 = StretchUltraExpanded
    toEnum k = AnotherStretch k

instance P.Ord Stretch where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Stretch = '[]
instance O.HasParentTypes Stretch

foreign import ccall "pango_stretch_get_type" c_pango_stretch_get_type :: 
    IO GType

instance B.Types.TypedObject Stretch where
    glibType = c_pango_stretch_get_type

instance B.Types.BoxedEnum Stretch

-- Enum Script
-- | The @PangoScript@ enumeration identifies different writing
-- systems.
-- 
-- The values correspond to the names as defined in the Unicode standard. See
-- <http://www.unicode.org/reports/tr24/ Unicode Standard Annex 24: Script names>
-- 
-- Note that this enumeration is deprecated and will not be updated to include values
-- in newer versions of the Unicode standard. Applications should use the
-- t'GI.GLib.Enums.UnicodeScript' enumeration instead,
-- whose values are interchangeable with @PangoScript@.
data Script = 
      ScriptInvalidCode
    -- ^ a value never returned from 'GI.Pango.Functions.scriptForUnichar'
    | ScriptCommon
    -- ^ a character used by multiple different scripts
    | ScriptInherited
    -- ^ a mark glyph that takes its script from the
    -- base glyph to which it is attached
    | ScriptArabic
    -- ^ Arabic
    | ScriptArmenian
    -- ^ Armenian
    | ScriptBengali
    -- ^ Bengali
    | ScriptBopomofo
    -- ^ Bopomofo
    | ScriptCherokee
    -- ^ Cherokee
    | ScriptCoptic
    -- ^ Coptic
    | ScriptCyrillic
    -- ^ Cyrillic
    | ScriptDeseret
    -- ^ Deseret
    | ScriptDevanagari
    -- ^ Devanagari
    | ScriptEthiopic
    -- ^ Ethiopic
    | ScriptGeorgian
    -- ^ Georgian
    | ScriptGothic
    -- ^ Gothic
    | ScriptGreek
    -- ^ Greek
    | ScriptGujarati
    -- ^ Gujarati
    | ScriptGurmukhi
    -- ^ Gurmukhi
    | ScriptHan
    -- ^ Han
    | ScriptHangul
    -- ^ Hangul
    | ScriptHebrew
    -- ^ Hebrew
    | ScriptHiragana
    -- ^ Hiragana
    | ScriptKannada
    -- ^ Kannada
    | ScriptKatakana
    -- ^ Katakana
    | ScriptKhmer
    -- ^ Khmer
    | ScriptLao
    -- ^ Lao
    | ScriptLatin
    -- ^ Latin
    | ScriptMalayalam
    -- ^ Malayalam
    | ScriptMongolian
    -- ^ Mongolian
    | ScriptMyanmar
    -- ^ Myanmar
    | ScriptOgham
    -- ^ Ogham
    | ScriptOldItalic
    -- ^ Old Italic
    | ScriptOriya
    -- ^ Oriya
    | ScriptRunic
    -- ^ Runic
    | ScriptSinhala
    -- ^ Sinhala
    | ScriptSyriac
    -- ^ Syriac
    | ScriptTamil
    -- ^ Tamil
    | ScriptTelugu
    -- ^ Telugu
    | ScriptThaana
    -- ^ Thaana
    | ScriptThai
    -- ^ Thai
    | ScriptTibetan
    -- ^ Tibetan
    | ScriptCanadianAboriginal
    -- ^ Canadian Aboriginal
    | ScriptYi
    -- ^ Yi
    | ScriptTagalog
    -- ^ Tagalog
    | ScriptHanunoo
    -- ^ Hanunoo
    | ScriptBuhid
    -- ^ Buhid
    | ScriptTagbanwa
    -- ^ Tagbanwa
    | ScriptBraille
    -- ^ Braille
    | ScriptCypriot
    -- ^ Cypriot
    | ScriptLimbu
    -- ^ Limbu
    | ScriptOsmanya
    -- ^ Osmanya
    | ScriptShavian
    -- ^ Shavian
    | ScriptLinearB
    -- ^ Linear B
    | ScriptTaiLe
    -- ^ Tai Le
    | ScriptUgaritic
    -- ^ Ugaritic
    | ScriptNewTaiLue
    -- ^ New Tai Lue. Since 1.10
    | ScriptBuginese
    -- ^ Buginese. Since 1.10
    | ScriptGlagolitic
    -- ^ Glagolitic. Since 1.10
    | ScriptTifinagh
    -- ^ Tifinagh. Since 1.10
    | ScriptSylotiNagri
    -- ^ Syloti Nagri. Since 1.10
    | ScriptOldPersian
    -- ^ Old Persian. Since 1.10
    | ScriptKharoshthi
    -- ^ Kharoshthi. Since 1.10
    | ScriptUnknown
    -- ^ an unassigned code point. Since 1.14
    | ScriptBalinese
    -- ^ Balinese. Since 1.14
    | ScriptCuneiform
    -- ^ Cuneiform. Since 1.14
    | ScriptPhoenician
    -- ^ Phoenician. Since 1.14
    | ScriptPhagsPa
    -- ^ Phags-pa. Since 1.14
    | ScriptNko
    -- ^ N\'Ko. Since 1.14
    | ScriptKayahLi
    -- ^ Kayah Li. Since 1.20.1
    | ScriptLepcha
    -- ^ Lepcha. Since 1.20.1
    | ScriptRejang
    -- ^ Rejang. Since 1.20.1
    | ScriptSundanese
    -- ^ Sundanese. Since 1.20.1
    | ScriptSaurashtra
    -- ^ Saurashtra. Since 1.20.1
    | ScriptCham
    -- ^ Cham. Since 1.20.1
    | ScriptOlChiki
    -- ^ Ol Chiki. Since 1.20.1
    | ScriptVai
    -- ^ Vai. Since 1.20.1
    | ScriptCarian
    -- ^ Carian. Since 1.20.1
    | ScriptLycian
    -- ^ Lycian. Since 1.20.1
    | ScriptLydian
    -- ^ Lydian. Since 1.20.1
    | ScriptBatak
    -- ^ Batak. Since 1.32
    | ScriptBrahmi
    -- ^ Brahmi. Since 1.32
    | ScriptMandaic
    -- ^ Mandaic. Since 1.32
    | ScriptChakma
    -- ^ Chakma. Since: 1.32
    | ScriptMeroiticCursive
    -- ^ Meroitic Cursive. Since: 1.32
    | ScriptMeroiticHieroglyphs
    -- ^ Meroitic Hieroglyphs. Since: 1.32
    | ScriptMiao
    -- ^ Miao. Since: 1.32
    | ScriptSharada
    -- ^ Sharada. Since: 1.32
    | ScriptSoraSompeng
    -- ^ Sora Sompeng. Since: 1.32
    | ScriptTakri
    -- ^ Takri. Since: 1.32
    | ScriptBassaVah
    -- ^ Bassa. Since: 1.40
    | ScriptCaucasianAlbanian
    -- ^ Caucasian Albanian. Since: 1.40
    | ScriptDuployan
    -- ^ Duployan. Since: 1.40
    | ScriptElbasan
    -- ^ Elbasan. Since: 1.40
    | ScriptGrantha
    -- ^ Grantha. Since: 1.40
    | ScriptKhojki
    -- ^ Kjohki. Since: 1.40
    | ScriptKhudawadi
    -- ^ Khudawadi, Sindhi. Since: 1.40
    | ScriptLinearA
    -- ^ Linear A. Since: 1.40
    | ScriptMahajani
    -- ^ Mahajani. Since: 1.40
    | ScriptManichaean
    -- ^ Manichaean. Since: 1.40
    | ScriptMendeKikakui
    -- ^ Mende Kikakui. Since: 1.40
    | ScriptModi
    -- ^ Modi. Since: 1.40
    | ScriptMro
    -- ^ Mro. Since: 1.40
    | ScriptNabataean
    -- ^ Nabataean. Since: 1.40
    | ScriptOldNorthArabian
    -- ^ Old North Arabian. Since: 1.40
    | ScriptOldPermic
    -- ^ Old Permic. Since: 1.40
    | ScriptPahawhHmong
    -- ^ Pahawh Hmong. Since: 1.40
    | ScriptPalmyrene
    -- ^ Palmyrene. Since: 1.40
    | ScriptPauCinHau
    -- ^ Pau Cin Hau. Since: 1.40
    | ScriptPsalterPahlavi
    -- ^ Psalter Pahlavi. Since: 1.40
    | ScriptSiddham
    -- ^ Siddham. Since: 1.40
    | ScriptTirhuta
    -- ^ Tirhuta. Since: 1.40
    | ScriptWarangCiti
    -- ^ Warang Citi. Since: 1.40
    | ScriptAhom
    -- ^ Ahom. Since: 1.40
    | ScriptAnatolianHieroglyphs
    -- ^ Anatolian Hieroglyphs. Since: 1.40
    | ScriptHatran
    -- ^ Hatran. Since: 1.40
    | ScriptMultani
    -- ^ Multani. Since: 1.40
    | ScriptOldHungarian
    -- ^ Old Hungarian. Since: 1.40
    | ScriptSignwriting
    -- ^ Signwriting. Since: 1.40
    | AnotherScript Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Script where
    fromEnum ScriptInvalidCode = -1
    fromEnum ScriptCommon = 0
    fromEnum ScriptInherited = 1
    fromEnum ScriptArabic = 2
    fromEnum ScriptArmenian = 3
    fromEnum ScriptBengali = 4
    fromEnum ScriptBopomofo = 5
    fromEnum ScriptCherokee = 6
    fromEnum ScriptCoptic = 7
    fromEnum ScriptCyrillic = 8
    fromEnum ScriptDeseret = 9
    fromEnum ScriptDevanagari = 10
    fromEnum ScriptEthiopic = 11
    fromEnum ScriptGeorgian = 12
    fromEnum ScriptGothic = 13
    fromEnum ScriptGreek = 14
    fromEnum ScriptGujarati = 15
    fromEnum ScriptGurmukhi = 16
    fromEnum ScriptHan = 17
    fromEnum ScriptHangul = 18
    fromEnum ScriptHebrew = 19
    fromEnum ScriptHiragana = 20
    fromEnum ScriptKannada = 21
    fromEnum ScriptKatakana = 22
    fromEnum ScriptKhmer = 23
    fromEnum ScriptLao = 24
    fromEnum ScriptLatin = 25
    fromEnum ScriptMalayalam = 26
    fromEnum ScriptMongolian = 27
    fromEnum ScriptMyanmar = 28
    fromEnum ScriptOgham = 29
    fromEnum ScriptOldItalic = 30
    fromEnum ScriptOriya = 31
    fromEnum ScriptRunic = 32
    fromEnum ScriptSinhala = 33
    fromEnum ScriptSyriac = 34
    fromEnum ScriptTamil = 35
    fromEnum ScriptTelugu = 36
    fromEnum ScriptThaana = 37
    fromEnum ScriptThai = 38
    fromEnum ScriptTibetan = 39
    fromEnum ScriptCanadianAboriginal = 40
    fromEnum ScriptYi = 41
    fromEnum ScriptTagalog = 42
    fromEnum ScriptHanunoo = 43
    fromEnum ScriptBuhid = 44
    fromEnum ScriptTagbanwa = 45
    fromEnum ScriptBraille = 46
    fromEnum ScriptCypriot = 47
    fromEnum ScriptLimbu = 48
    fromEnum ScriptOsmanya = 49
    fromEnum ScriptShavian = 50
    fromEnum ScriptLinearB = 51
    fromEnum ScriptTaiLe = 52
    fromEnum ScriptUgaritic = 53
    fromEnum ScriptNewTaiLue = 54
    fromEnum ScriptBuginese = 55
    fromEnum ScriptGlagolitic = 56
    fromEnum ScriptTifinagh = 57
    fromEnum ScriptSylotiNagri = 58
    fromEnum ScriptOldPersian = 59
    fromEnum ScriptKharoshthi = 60
    fromEnum ScriptUnknown = 61
    fromEnum ScriptBalinese = 62
    fromEnum ScriptCuneiform = 63
    fromEnum ScriptPhoenician = 64
    fromEnum ScriptPhagsPa = 65
    fromEnum ScriptNko = 66
    fromEnum ScriptKayahLi = 67
    fromEnum ScriptLepcha = 68
    fromEnum ScriptRejang = 69
    fromEnum ScriptSundanese = 70
    fromEnum ScriptSaurashtra = 71
    fromEnum ScriptCham = 72
    fromEnum ScriptOlChiki = 73
    fromEnum ScriptVai = 74
    fromEnum ScriptCarian = 75
    fromEnum ScriptLycian = 76
    fromEnum ScriptLydian = 77
    fromEnum ScriptBatak = 78
    fromEnum ScriptBrahmi = 79
    fromEnum ScriptMandaic = 80
    fromEnum ScriptChakma = 81
    fromEnum ScriptMeroiticCursive = 82
    fromEnum ScriptMeroiticHieroglyphs = 83
    fromEnum ScriptMiao = 84
    fromEnum ScriptSharada = 85
    fromEnum ScriptSoraSompeng = 86
    fromEnum ScriptTakri = 87
    fromEnum ScriptBassaVah = 88
    fromEnum ScriptCaucasianAlbanian = 89
    fromEnum ScriptDuployan = 90
    fromEnum ScriptElbasan = 91
    fromEnum ScriptGrantha = 92
    fromEnum ScriptKhojki = 93
    fromEnum ScriptKhudawadi = 94
    fromEnum ScriptLinearA = 95
    fromEnum ScriptMahajani = 96
    fromEnum ScriptManichaean = 97
    fromEnum ScriptMendeKikakui = 98
    fromEnum ScriptModi = 99
    fromEnum ScriptMro = 100
    fromEnum ScriptNabataean = 101
    fromEnum ScriptOldNorthArabian = 102
    fromEnum ScriptOldPermic = 103
    fromEnum ScriptPahawhHmong = 104
    fromEnum ScriptPalmyrene = 105
    fromEnum ScriptPauCinHau = 106
    fromEnum ScriptPsalterPahlavi = 107
    fromEnum ScriptSiddham = 108
    fromEnum ScriptTirhuta = 109
    fromEnum ScriptWarangCiti = 110
    fromEnum ScriptAhom = 111
    fromEnum ScriptAnatolianHieroglyphs = 112
    fromEnum ScriptHatran = 113
    fromEnum ScriptMultani = 114
    fromEnum ScriptOldHungarian = 115
    fromEnum ScriptSignwriting = 116
    fromEnum (AnotherScript k) = k

    toEnum -1 = ScriptInvalidCode
    toEnum 0 = ScriptCommon
    toEnum 1 = ScriptInherited
    toEnum 2 = ScriptArabic
    toEnum 3 = ScriptArmenian
    toEnum 4 = ScriptBengali
    toEnum 5 = ScriptBopomofo
    toEnum 6 = ScriptCherokee
    toEnum 7 = ScriptCoptic
    toEnum 8 = ScriptCyrillic
    toEnum 9 = ScriptDeseret
    toEnum 10 = ScriptDevanagari
    toEnum 11 = ScriptEthiopic
    toEnum 12 = ScriptGeorgian
    toEnum 13 = ScriptGothic
    toEnum 14 = ScriptGreek
    toEnum 15 = ScriptGujarati
    toEnum 16 = ScriptGurmukhi
    toEnum 17 = ScriptHan
    toEnum 18 = ScriptHangul
    toEnum 19 = ScriptHebrew
    toEnum 20 = ScriptHiragana
    toEnum 21 = ScriptKannada
    toEnum 22 = ScriptKatakana
    toEnum 23 = ScriptKhmer
    toEnum 24 = ScriptLao
    toEnum 25 = ScriptLatin
    toEnum 26 = ScriptMalayalam
    toEnum 27 = ScriptMongolian
    toEnum 28 = ScriptMyanmar
    toEnum 29 = ScriptOgham
    toEnum 30 = ScriptOldItalic
    toEnum 31 = ScriptOriya
    toEnum 32 = ScriptRunic
    toEnum 33 = ScriptSinhala
    toEnum 34 = ScriptSyriac
    toEnum 35 = ScriptTamil
    toEnum 36 = ScriptTelugu
    toEnum 37 = ScriptThaana
    toEnum 38 = ScriptThai
    toEnum 39 = ScriptTibetan
    toEnum 40 = ScriptCanadianAboriginal
    toEnum 41 = ScriptYi
    toEnum 42 = ScriptTagalog
    toEnum 43 = ScriptHanunoo
    toEnum 44 = ScriptBuhid
    toEnum 45 = ScriptTagbanwa
    toEnum 46 = ScriptBraille
    toEnum 47 = ScriptCypriot
    toEnum 48 = ScriptLimbu
    toEnum 49 = ScriptOsmanya
    toEnum 50 = ScriptShavian
    toEnum 51 = ScriptLinearB
    toEnum 52 = ScriptTaiLe
    toEnum 53 = ScriptUgaritic
    toEnum 54 = ScriptNewTaiLue
    toEnum 55 = ScriptBuginese
    toEnum 56 = ScriptGlagolitic
    toEnum 57 = ScriptTifinagh
    toEnum 58 = ScriptSylotiNagri
    toEnum 59 = ScriptOldPersian
    toEnum 60 = ScriptKharoshthi
    toEnum 61 = ScriptUnknown
    toEnum 62 = ScriptBalinese
    toEnum 63 = ScriptCuneiform
    toEnum 64 = ScriptPhoenician
    toEnum 65 = ScriptPhagsPa
    toEnum 66 = ScriptNko
    toEnum 67 = ScriptKayahLi
    toEnum 68 = ScriptLepcha
    toEnum 69 = ScriptRejang
    toEnum 70 = ScriptSundanese
    toEnum 71 = ScriptSaurashtra
    toEnum 72 = ScriptCham
    toEnum 73 = ScriptOlChiki
    toEnum 74 = ScriptVai
    toEnum 75 = ScriptCarian
    toEnum 76 = ScriptLycian
    toEnum 77 = ScriptLydian
    toEnum 78 = ScriptBatak
    toEnum 79 = ScriptBrahmi
    toEnum 80 = ScriptMandaic
    toEnum 81 = ScriptChakma
    toEnum 82 = ScriptMeroiticCursive
    toEnum 83 = ScriptMeroiticHieroglyphs
    toEnum 84 = ScriptMiao
    toEnum 85 = ScriptSharada
    toEnum 86 = ScriptSoraSompeng
    toEnum 87 = ScriptTakri
    toEnum 88 = ScriptBassaVah
    toEnum 89 = ScriptCaucasianAlbanian
    toEnum 90 = ScriptDuployan
    toEnum 91 = ScriptElbasan
    toEnum 92 = ScriptGrantha
    toEnum 93 = ScriptKhojki
    toEnum 94 = ScriptKhudawadi
    toEnum 95 = ScriptLinearA
    toEnum 96 = ScriptMahajani
    toEnum 97 = ScriptManichaean
    toEnum 98 = ScriptMendeKikakui
    toEnum 99 = ScriptModi
    toEnum 100 = ScriptMro
    toEnum 101 = ScriptNabataean
    toEnum 102 = ScriptOldNorthArabian
    toEnum 103 = ScriptOldPermic
    toEnum 104 = ScriptPahawhHmong
    toEnum 105 = ScriptPalmyrene
    toEnum 106 = ScriptPauCinHau
    toEnum 107 = ScriptPsalterPahlavi
    toEnum 108 = ScriptSiddham
    toEnum 109 = ScriptTirhuta
    toEnum 110 = ScriptWarangCiti
    toEnum 111 = ScriptAhom
    toEnum 112 = ScriptAnatolianHieroglyphs
    toEnum 113 = ScriptHatran
    toEnum 114 = ScriptMultani
    toEnum 115 = ScriptOldHungarian
    toEnum 116 = ScriptSignwriting
    toEnum k = AnotherScript k

instance P.Ord Script where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Script = '[]
instance O.HasParentTypes Script

foreign import ccall "pango_script_get_type" c_pango_script_get_type :: 
    IO GType

instance B.Types.TypedObject Script where
    glibType = c_pango_script_get_type

instance B.Types.BoxedEnum Script

-- Enum RenderPart
-- | @PangoRenderPart@ defines different items to render for such
-- purposes as setting colors.
-- 
-- /Since: 1.8/
data RenderPart = 
      RenderPartForeground
    -- ^ the text itself
    | RenderPartBackground
    -- ^ the area behind the text
    | RenderPartUnderline
    -- ^ underlines
    | RenderPartStrikethrough
    -- ^ strikethrough lines
    | RenderPartOverline
    -- ^ overlines
    | AnotherRenderPart Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum RenderPart where
    fromEnum RenderPartForeground = 0
    fromEnum RenderPartBackground = 1
    fromEnum RenderPartUnderline = 2
    fromEnum RenderPartStrikethrough = 3
    fromEnum RenderPartOverline = 4
    fromEnum (AnotherRenderPart k) = k

    toEnum 0 = RenderPartForeground
    toEnum 1 = RenderPartBackground
    toEnum 2 = RenderPartUnderline
    toEnum 3 = RenderPartStrikethrough
    toEnum 4 = RenderPartOverline
    toEnum k = AnotherRenderPart k

instance P.Ord RenderPart where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes RenderPart = '[]
instance O.HasParentTypes RenderPart

foreign import ccall "pango_render_part_get_type" c_pango_render_part_get_type :: 
    IO GType

instance B.Types.TypedObject RenderPart where
    glibType = c_pango_render_part_get_type

instance B.Types.BoxedEnum RenderPart

-- Enum Overline
-- | The @PangoOverline@ enumeration is used to specify whether text
-- should be overlined, and if so, the type of line.
-- 
-- /Since: 1.46/
data Overline = 
      OverlineNone
    -- ^ no overline should be drawn
    | OverlineSingle
    -- ^ Draw a single line above the ink
    --   extents of the text being underlined.
    | AnotherOverline Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Overline where
    fromEnum OverlineNone = 0
    fromEnum OverlineSingle = 1
    fromEnum (AnotherOverline k) = k

    toEnum 0 = OverlineNone
    toEnum 1 = OverlineSingle
    toEnum k = AnotherOverline k

instance P.Ord Overline where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Overline = '[]
instance O.HasParentTypes Overline

foreign import ccall "pango_overline_get_type" c_pango_overline_get_type :: 
    IO GType

instance B.Types.TypedObject Overline where
    glibType = c_pango_overline_get_type

instance B.Types.BoxedEnum Overline

-- Enum LayoutDeserializeError
-- | Errors that can be returned by 'GI.Pango.Objects.Layout.layoutDeserialize'.
-- 
-- /Since: 1.50/
data LayoutDeserializeError = 
      LayoutDeserializeErrorInvalid
    -- ^ Unspecified error
    | LayoutDeserializeErrorInvalidValue
    -- ^ A JSon value could not be
    --   interpreted
    | LayoutDeserializeErrorMissingValue
    -- ^ A required JSon member was
    --   not found
    | AnotherLayoutDeserializeError Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum LayoutDeserializeError where
    fromEnum LayoutDeserializeErrorInvalid = 0
    fromEnum LayoutDeserializeErrorInvalidValue = 1
    fromEnum LayoutDeserializeErrorMissingValue = 2
    fromEnum (AnotherLayoutDeserializeError k) = k

    toEnum 0 = LayoutDeserializeErrorInvalid
    toEnum 1 = LayoutDeserializeErrorInvalidValue
    toEnum 2 = LayoutDeserializeErrorMissingValue
    toEnum k = AnotherLayoutDeserializeError k

instance P.Ord LayoutDeserializeError where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

instance GErrorClass LayoutDeserializeError where
    gerrorClassDomain _ = "pango-layout-deserialize-error-quark"

-- | Catch exceptions of type `LayoutDeserializeError`. This is a specialized version of `Data.GI.Base.GError.catchGErrorJustDomain`.
catchLayoutDeserializeError ::
    IO a ->
    (LayoutDeserializeError -> GErrorMessage -> IO a) ->
    IO a
catchLayoutDeserializeError = catchGErrorJustDomain

-- | Handle exceptions of type `LayoutDeserializeError`. This is a specialized version of `Data.GI.Base.GError.handleGErrorJustDomain`.
handleLayoutDeserializeError ::
    (LayoutDeserializeError -> GErrorMessage -> IO a) ->
    IO a ->
    IO a
handleLayoutDeserializeError = handleGErrorJustDomain

type instance O.ParentTypes LayoutDeserializeError = '[]
instance O.HasParentTypes LayoutDeserializeError

foreign import ccall "pango_layout_deserialize_error_get_type" c_pango_layout_deserialize_error_get_type :: 
    IO GType

instance B.Types.TypedObject LayoutDeserializeError where
    glibType = c_pango_layout_deserialize_error_get_type

instance B.Types.BoxedEnum LayoutDeserializeError

-- Enum GravityHint
-- | @PangoGravityHint@ defines how horizontal scripts should behave in a
-- vertical context.
-- 
-- That is, English excerpts in a vertical paragraph for example.
-- 
-- See also t'GI.Pango.Enums.Gravity'
-- 
-- /Since: 1.16/
data GravityHint = 
      GravityHintNatural
    -- ^ scripts will take their natural gravity based
    --   on the base gravity and the script.  This is the default.
    | GravityHintStrong
    -- ^ always use the base gravity set, regardless of
    --   the script.
    | GravityHintLine
    -- ^ for scripts not in their natural direction (eg.
    --   Latin in East gravity), choose per-script gravity such that every script
    --   respects the line progression. This means, Latin and Arabic will take
    --   opposite gravities and both flow top-to-bottom for example.
    | AnotherGravityHint Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum GravityHint where
    fromEnum GravityHintNatural = 0
    fromEnum GravityHintStrong = 1
    fromEnum GravityHintLine = 2
    fromEnum (AnotherGravityHint k) = k

    toEnum 0 = GravityHintNatural
    toEnum 1 = GravityHintStrong
    toEnum 2 = GravityHintLine
    toEnum k = AnotherGravityHint k

instance P.Ord GravityHint where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes GravityHint = '[]
instance O.HasParentTypes GravityHint

foreign import ccall "pango_gravity_hint_get_type" c_pango_gravity_hint_get_type :: 
    IO GType

instance B.Types.TypedObject GravityHint where
    glibType = c_pango_gravity_hint_get_type

instance B.Types.BoxedEnum GravityHint

-- Enum Gravity
-- | @PangoGravity@ represents the orientation of glyphs in a segment
-- of text.
-- 
-- This is useful when rendering vertical text layouts. In those situations,
-- the layout is rotated using a non-identity t'GI.Pango.Structs.Matrix.Matrix', and then
-- glyph orientation is controlled using @PangoGravity@.
-- 
-- Not every value in this enumeration makes sense for every usage of
-- @PangoGravity@; for example, 'GI.Pango.Enums.GravityAuto' only can be passed to
-- 'GI.Pango.Objects.Context.contextSetBaseGravity' and can only be returned by
-- 'GI.Pango.Objects.Context.contextGetBaseGravity'.
-- 
-- See also: t'GI.Pango.Enums.GravityHint'
-- 
-- /Since: 1.16/
data Gravity = 
      GravitySouth
    -- ^ Glyphs stand upright (default) \<img align=\"right\" valign=\"center\" src=\"m-south.png\">
    | GravityEast
    -- ^ Glyphs are rotated 90 degrees counter-clockwise. \<img align=\"right\" valign=\"center\" src=\"m-east.png\">
    | GravityNorth
    -- ^ Glyphs are upside-down. \<img align=\"right\" valign=\"cener\" src=\"m-north.png\">
    | GravityWest
    -- ^ Glyphs are rotated 90 degrees clockwise. \<img align=\"right\" valign=\"center\" src=\"m-west.png\">
    | GravityAuto
    -- ^ Gravity is resolved from the context matrix
    | AnotherGravity Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Gravity where
    fromEnum GravitySouth = 0
    fromEnum GravityEast = 1
    fromEnum GravityNorth = 2
    fromEnum GravityWest = 3
    fromEnum GravityAuto = 4
    fromEnum (AnotherGravity k) = k

    toEnum 0 = GravitySouth
    toEnum 1 = GravityEast
    toEnum 2 = GravityNorth
    toEnum 3 = GravityWest
    toEnum 4 = GravityAuto
    toEnum k = AnotherGravity k

instance P.Ord Gravity where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Gravity = '[]
instance O.HasParentTypes Gravity

foreign import ccall "pango_gravity_get_type" c_pango_gravity_get_type :: 
    IO GType

instance B.Types.TypedObject Gravity where
    glibType = c_pango_gravity_get_type

instance B.Types.BoxedEnum Gravity

-- Enum FontScale
-- | An enumeration that affects font sizes for superscript
-- and subscript positioning and for (emulated) Small Caps.
-- 
-- /Since: 1.50/
data FontScale = 
      FontScaleNone
    -- ^ Leave the font size unchanged
    | FontScaleSuperscript
    -- ^ Change the font to a size suitable for superscripts
    | FontScaleSubscript
    -- ^ Change the font to a size suitable for subscripts
    | FontScaleSmallCaps
    -- ^ Change the font to a size suitable for Small Caps
    | AnotherFontScale Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum FontScale where
    fromEnum FontScaleNone = 0
    fromEnum FontScaleSuperscript = 1
    fromEnum FontScaleSubscript = 2
    fromEnum FontScaleSmallCaps = 3
    fromEnum (AnotherFontScale k) = k

    toEnum 0 = FontScaleNone
    toEnum 1 = FontScaleSuperscript
    toEnum 2 = FontScaleSubscript
    toEnum 3 = FontScaleSmallCaps
    toEnum k = AnotherFontScale k

instance P.Ord FontScale where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes FontScale = '[]
instance O.HasParentTypes FontScale

foreign import ccall "pango_font_scale_get_type" c_pango_font_scale_get_type :: 
    IO GType

instance B.Types.TypedObject FontScale where
    glibType = c_pango_font_scale_get_type

instance B.Types.BoxedEnum FontScale

-- Enum FontColor
-- | Specifies whether a font should or should not have color glyphs.
-- 
-- /Since: 1.57/
data FontColor = 
      FontColorForbidden
    -- ^ The font should not have color glyphs
    | FontColorRequired
    -- ^ The font should have color glyphs
    | FontColorDontCare
    -- ^ The font may or may not use color
    | AnotherFontColor Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum FontColor where
    fromEnum FontColorForbidden = 0
    fromEnum FontColorRequired = 1
    fromEnum FontColorDontCare = 2
    fromEnum (AnotherFontColor k) = k

    toEnum 0 = FontColorForbidden
    toEnum 1 = FontColorRequired
    toEnum 2 = FontColorDontCare
    toEnum k = AnotherFontColor k

instance P.Ord FontColor where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes FontColor = '[]
instance O.HasParentTypes FontColor

foreign import ccall "pango_font_color_get_type" c_pango_font_color_get_type :: 
    IO GType

instance B.Types.TypedObject FontColor where
    glibType = c_pango_font_color_get_type

instance B.Types.BoxedEnum FontColor

-- Enum EllipsizeMode
-- | @PangoEllipsizeMode@ describes what sort of ellipsization
-- should be applied to text.
-- 
-- In the ellipsization process characters are removed from the
-- text in order to make it fit to a given width and replaced
-- with an ellipsis.
data EllipsizeMode = 
      EllipsizeModeNone
    -- ^ No ellipsization
    | EllipsizeModeStart
    -- ^ Omit characters at the start of the text
    | EllipsizeModeMiddle
    -- ^ Omit characters in the middle of the text
    | EllipsizeModeEnd
    -- ^ Omit characters at the end of the text
    | AnotherEllipsizeMode Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum EllipsizeMode where
    fromEnum EllipsizeModeNone = 0
    fromEnum EllipsizeModeStart = 1
    fromEnum EllipsizeModeMiddle = 2
    fromEnum EllipsizeModeEnd = 3
    fromEnum (AnotherEllipsizeMode k) = k

    toEnum 0 = EllipsizeModeNone
    toEnum 1 = EllipsizeModeStart
    toEnum 2 = EllipsizeModeMiddle
    toEnum 3 = EllipsizeModeEnd
    toEnum k = AnotherEllipsizeMode k

instance P.Ord EllipsizeMode where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes EllipsizeMode = '[]
instance O.HasParentTypes EllipsizeMode

foreign import ccall "pango_ellipsize_mode_get_type" c_pango_ellipsize_mode_get_type :: 
    IO GType

instance B.Types.TypedObject EllipsizeMode where
    glibType = c_pango_ellipsize_mode_get_type

instance B.Types.BoxedEnum EllipsizeMode

-- Enum Direction
-- | @PangoDirection@ represents a direction in the Unicode bidirectional
-- algorithm.
-- 
-- Not every value in this enumeration makes sense for every usage of
-- @PangoDirection@; for example, the return value of [func/@unicharDirection@/]
-- and [func/@findBaseDir@/] cannot be @PANGO_DIRECTION_WEAK_LTR@ or
-- @PANGO_DIRECTION_WEAK_RTL@, since every character is either neutral
-- or has a strong direction; on the other hand @PANGO_DIRECTION_NEUTRAL@
-- doesn\'t make sense to pass to [func/@itemizeWithBaseDir@/].
-- 
-- The @PANGO_DIRECTION_TTB_LTR@, @PANGO_DIRECTION_TTB_RTL@ values come from
-- an earlier interpretation of this enumeration as the writing direction
-- of a block of text and are no longer used. See @PangoGravity@ for how
-- vertical text is handled in Pango.
-- 
-- If you are interested in text direction, you should really use fribidi
-- directly. @PangoDirection@ is only retained because it is used in some
-- public apis.
data Direction = 
      DirectionLtr
    -- ^ A strong left-to-right direction
    | DirectionRtl
    -- ^ A strong right-to-left direction
    | DirectionTtbLtr
    -- ^ Deprecated value; treated the
    --   same as @PANGO_DIRECTION_RTL@.
    | DirectionTtbRtl
    -- ^ Deprecated value; treated the
    --   same as @PANGO_DIRECTION_LTR@
    | DirectionWeakLtr
    -- ^ A weak left-to-right direction
    | DirectionWeakRtl
    -- ^ A weak right-to-left direction
    | DirectionNeutral
    -- ^ No direction specified
    | AnotherDirection Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Direction where
    fromEnum DirectionLtr = 0
    fromEnum DirectionRtl = 1
    fromEnum DirectionTtbLtr = 2
    fromEnum DirectionTtbRtl = 3
    fromEnum DirectionWeakLtr = 4
    fromEnum DirectionWeakRtl = 5
    fromEnum DirectionNeutral = 6
    fromEnum (AnotherDirection k) = k

    toEnum 0 = DirectionLtr
    toEnum 1 = DirectionRtl
    toEnum 2 = DirectionTtbLtr
    toEnum 3 = DirectionTtbRtl
    toEnum 4 = DirectionWeakLtr
    toEnum 5 = DirectionWeakRtl
    toEnum 6 = DirectionNeutral
    toEnum k = AnotherDirection k

instance P.Ord Direction where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Direction = '[]
instance O.HasParentTypes Direction

foreign import ccall "pango_direction_get_type" c_pango_direction_get_type :: 
    IO GType

instance B.Types.TypedObject Direction where
    glibType = c_pango_direction_get_type

instance B.Types.BoxedEnum Direction

-- Enum CoverageLevel
-- | @PangoCoverageLevel@ is used to indicate how well a font can
-- represent a particular Unicode character for a particular script.
-- 
-- Since 1.44, only 'GI.Pango.Enums.CoverageLevelNone' and 'GI.Pango.Enums.CoverageLevelExact'
-- will be returned.
data CoverageLevel = 
      CoverageLevelNone
    -- ^ The character is not representable with
    --   the font.
    | CoverageLevelFallback
    -- ^ The character is represented in a
    --   way that may be comprehensible but is not the correct
    --   graphical form. For instance, a Hangul character represented
    --   as a a sequence of Jamos, or a Latin transliteration of a
    --   Cyrillic word.
    | CoverageLevelApproximate
    -- ^ The character is represented as
    --   basically the correct graphical form, but with a stylistic
    --   variant inappropriate for the current script.
    | CoverageLevelExact
    -- ^ The character is represented as the
    --   correct graphical form.
    | AnotherCoverageLevel Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum CoverageLevel where
    fromEnum CoverageLevelNone = 0
    fromEnum CoverageLevelFallback = 1
    fromEnum CoverageLevelApproximate = 2
    fromEnum CoverageLevelExact = 3
    fromEnum (AnotherCoverageLevel k) = k

    toEnum 0 = CoverageLevelNone
    toEnum 1 = CoverageLevelFallback
    toEnum 2 = CoverageLevelApproximate
    toEnum 3 = CoverageLevelExact
    toEnum k = AnotherCoverageLevel k

instance P.Ord CoverageLevel where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes CoverageLevel = '[]
instance O.HasParentTypes CoverageLevel

foreign import ccall "pango_coverage_level_get_type" c_pango_coverage_level_get_type :: 
    IO GType

instance B.Types.TypedObject CoverageLevel where
    glibType = c_pango_coverage_level_get_type

instance B.Types.BoxedEnum CoverageLevel

-- Enum BidiType
{-# DEPRECATED BidiType ["(Since version 1.44)","Use fribidi for this information"] #-}
-- | @PangoBidiType@ represents the bidirectional character
-- type of a Unicode character.
-- 
-- The values in this enumeration are specified by the
-- <http://www.unicode.org/reports/tr9/ Unicode bidirectional algorithm>.
-- 
-- /Since: 1.22/
data BidiType = 
      BidiTypeL
    -- ^ Left-to-Right
    | BidiTypeLre
    -- ^ Left-to-Right Embedding
    | BidiTypeLro
    -- ^ Left-to-Right Override
    | BidiTypeR
    -- ^ Right-to-Left
    | BidiTypeAl
    -- ^ Right-to-Left Arabic
    | BidiTypeRle
    -- ^ Right-to-Left Embedding
    | BidiTypeRlo
    -- ^ Right-to-Left Override
    | BidiTypePdf
    -- ^ Pop Directional Format
    | BidiTypeEn
    -- ^ European Number
    | BidiTypeEs
    -- ^ European Number Separator
    | BidiTypeEt
    -- ^ European Number Terminator
    | BidiTypeAn
    -- ^ Arabic Number
    | BidiTypeCs
    -- ^ Common Number Separator
    | BidiTypeNsm
    -- ^ Nonspacing Mark
    | BidiTypeBn
    -- ^ Boundary Neutral
    | BidiTypeB
    -- ^ Paragraph Separator
    | BidiTypeS
    -- ^ Segment Separator
    | BidiTypeWs
    -- ^ Whitespace
    | BidiTypeOn
    -- ^ Other Neutrals
    | BidiTypeLri
    -- ^ Left-to-Right isolate. Since 1.48.6
    | BidiTypeRli
    -- ^ Right-to-Left isolate. Since 1.48.6
    | BidiTypeFsi
    -- ^ First strong isolate. Since 1.48.6
    | BidiTypePdi
    -- ^ Pop directional isolate. Since 1.48.6
    | AnotherBidiType Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum BidiType where
    fromEnum BidiTypeL = 0
    fromEnum BidiTypeLre = 1
    fromEnum BidiTypeLro = 2
    fromEnum BidiTypeR = 3
    fromEnum BidiTypeAl = 4
    fromEnum BidiTypeRle = 5
    fromEnum BidiTypeRlo = 6
    fromEnum BidiTypePdf = 7
    fromEnum BidiTypeEn = 8
    fromEnum BidiTypeEs = 9
    fromEnum BidiTypeEt = 10
    fromEnum BidiTypeAn = 11
    fromEnum BidiTypeCs = 12
    fromEnum BidiTypeNsm = 13
    fromEnum BidiTypeBn = 14
    fromEnum BidiTypeB = 15
    fromEnum BidiTypeS = 16
    fromEnum BidiTypeWs = 17
    fromEnum BidiTypeOn = 18
    fromEnum BidiTypeLri = 19
    fromEnum BidiTypeRli = 20
    fromEnum BidiTypeFsi = 21
    fromEnum BidiTypePdi = 22
    fromEnum (AnotherBidiType k) = k

    toEnum 0 = BidiTypeL
    toEnum 1 = BidiTypeLre
    toEnum 2 = BidiTypeLro
    toEnum 3 = BidiTypeR
    toEnum 4 = BidiTypeAl
    toEnum 5 = BidiTypeRle
    toEnum 6 = BidiTypeRlo
    toEnum 7 = BidiTypePdf
    toEnum 8 = BidiTypeEn
    toEnum 9 = BidiTypeEs
    toEnum 10 = BidiTypeEt
    toEnum 11 = BidiTypeAn
    toEnum 12 = BidiTypeCs
    toEnum 13 = BidiTypeNsm
    toEnum 14 = BidiTypeBn
    toEnum 15 = BidiTypeB
    toEnum 16 = BidiTypeS
    toEnum 17 = BidiTypeWs
    toEnum 18 = BidiTypeOn
    toEnum 19 = BidiTypeLri
    toEnum 20 = BidiTypeRli
    toEnum 21 = BidiTypeFsi
    toEnum 22 = BidiTypePdi
    toEnum k = AnotherBidiType k

instance P.Ord BidiType where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes BidiType = '[]
instance O.HasParentTypes BidiType

foreign import ccall "pango_bidi_type_get_type" c_pango_bidi_type_get_type :: 
    IO GType

instance B.Types.TypedObject BidiType where
    glibType = c_pango_bidi_type_get_type

instance B.Types.BoxedEnum BidiType

-- Enum BaselineShift
-- | An enumeration that affects baseline shifts between runs.
-- 
-- /Since: 1.50/
data BaselineShift = 
      BaselineShiftNone
    -- ^ Leave the baseline unchanged
    | BaselineShiftSuperscript
    -- ^ Shift the baseline to the superscript position,
    --   relative to the previous run
    | BaselineShiftSubscript
    -- ^ Shift the baseline to the subscript position,
    --   relative to the previous run
    | AnotherBaselineShift Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum BaselineShift where
    fromEnum BaselineShiftNone = 0
    fromEnum BaselineShiftSuperscript = 1
    fromEnum BaselineShiftSubscript = 2
    fromEnum (AnotherBaselineShift k) = k

    toEnum 0 = BaselineShiftNone
    toEnum 1 = BaselineShiftSuperscript
    toEnum 2 = BaselineShiftSubscript
    toEnum k = AnotherBaselineShift k

instance P.Ord BaselineShift where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes BaselineShift = '[]
instance O.HasParentTypes BaselineShift

foreign import ccall "pango_baseline_shift_get_type" c_pango_baseline_shift_get_type :: 
    IO GType

instance B.Types.TypedObject BaselineShift where
    glibType = c_pango_baseline_shift_get_type

instance B.Types.BoxedEnum BaselineShift

-- Enum AttrType
-- | The @PangoAttrType@ distinguishes between different types of attributes.
-- 
-- Along with the predefined values, it is possible to allocate additional
-- values for custom attributes using @/AttrType.register/@. The predefined
-- values are given below. The type of structure used to store the attribute is
-- listed in parentheses after the description.
data AttrType = 
      AttrTypeInvalid
    -- ^ does not happen
    | AttrTypeLanguage
    -- ^ language (t'GI.Pango.Structs.AttrLanguage.AttrLanguage')
    | AttrTypeFamily
    -- ^ font family name list (t'GI.Pango.Structs.AttrString.AttrString')
    | AttrTypeStyle
    -- ^ font slant style (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeWeight
    -- ^ font weight (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeVariant
    -- ^ font variant (normal or small caps) (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeStretch
    -- ^ font stretch (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeSize
    -- ^ font size in points scaled by 'GI.Pango.Constants.SCALE' (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeFontDesc
    -- ^ font description (t'GI.Pango.Structs.AttrFontDesc.AttrFontDesc')
    | AttrTypeForeground
    -- ^ foreground color (t'GI.Pango.Structs.AttrColor.AttrColor')
    | AttrTypeBackground
    -- ^ background color (t'GI.Pango.Structs.AttrColor.AttrColor')
    | AttrTypeUnderline
    -- ^ whether the text has an underline (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeStrikethrough
    -- ^ whether the text is struck-through (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeRise
    -- ^ baseline displacement (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeShape
    -- ^ shape (t'GI.Pango.Structs.AttrShape.AttrShape')
    | AttrTypeScale
    -- ^ font size scale factor (t'GI.Pango.Structs.AttrFloat.AttrFloat')
    | AttrTypeFallback
    -- ^ whether fallback is enabled (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeLetterSpacing
    -- ^ letter spacing ([struct/@pangoAttrInt@/])
    | AttrTypeUnderlineColor
    -- ^ underline color (t'GI.Pango.Structs.AttrColor.AttrColor')
    | AttrTypeStrikethroughColor
    -- ^ strikethrough color (t'GI.Pango.Structs.AttrColor.AttrColor')
    | AttrTypeAbsoluteSize
    -- ^ font size in pixels scaled by 'GI.Pango.Constants.SCALE' (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeGravity
    -- ^ base text gravity (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeGravityHint
    -- ^ gravity hint (t'GI.Pango.Structs.AttrInt.AttrInt')
    | AttrTypeFontFeatures
    -- ^ OpenType font features (t'GI.Pango.Structs.AttrFontFeatures.AttrFontFeatures'). Since 1.38
    | AttrTypeForegroundAlpha
    -- ^ foreground alpha (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.38
    | AttrTypeBackgroundAlpha
    -- ^ background alpha (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.38
    | AttrTypeAllowBreaks
    -- ^ whether breaks are allowed (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.44
    | AttrTypeShow
    -- ^ how to render invisible characters (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.44
    | AttrTypeInsertHyphens
    -- ^ whether to insert hyphens at intra-word line breaks (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.44
    | AttrTypeOverline
    -- ^ whether the text has an overline (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.46
    | AttrTypeOverlineColor
    -- ^ overline color (t'GI.Pango.Structs.AttrColor.AttrColor'). Since 1.46
    | AttrTypeLineHeight
    -- ^ line height factor (t'GI.Pango.Structs.AttrFloat.AttrFloat'). Since: 1.50
    | AttrTypeAbsoluteLineHeight
    -- ^ line height (t'GI.Pango.Structs.AttrInt.AttrInt'). Since: 1.50
    | AttrTypeTextTransform
    -- ^ /No description available in the introspection data./
    | AttrTypeWord
    -- ^ override segmentation to classify the range of the attribute as a single word (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.50
    | AttrTypeSentence
    -- ^ override segmentation to classify the range of the attribute as a single sentence (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.50
    | AttrTypeBaselineShift
    -- ^ baseline displacement (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.50
    | AttrTypeFontScale
    -- ^ font-relative size change (t'GI.Pango.Structs.AttrInt.AttrInt'). Since 1.50
    | AnotherAttrType Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum AttrType where
    fromEnum AttrTypeInvalid = 0
    fromEnum AttrTypeLanguage = 1
    fromEnum AttrTypeFamily = 2
    fromEnum AttrTypeStyle = 3
    fromEnum AttrTypeWeight = 4
    fromEnum AttrTypeVariant = 5
    fromEnum AttrTypeStretch = 6
    fromEnum AttrTypeSize = 7
    fromEnum AttrTypeFontDesc = 8
    fromEnum AttrTypeForeground = 9
    fromEnum AttrTypeBackground = 10
    fromEnum AttrTypeUnderline = 11
    fromEnum AttrTypeStrikethrough = 12
    fromEnum AttrTypeRise = 13
    fromEnum AttrTypeShape = 14
    fromEnum AttrTypeScale = 15
    fromEnum AttrTypeFallback = 16
    fromEnum AttrTypeLetterSpacing = 17
    fromEnum AttrTypeUnderlineColor = 18
    fromEnum AttrTypeStrikethroughColor = 19
    fromEnum AttrTypeAbsoluteSize = 20
    fromEnum AttrTypeGravity = 21
    fromEnum AttrTypeGravityHint = 22
    fromEnum AttrTypeFontFeatures = 23
    fromEnum AttrTypeForegroundAlpha = 24
    fromEnum AttrTypeBackgroundAlpha = 25
    fromEnum AttrTypeAllowBreaks = 26
    fromEnum AttrTypeShow = 27
    fromEnum AttrTypeInsertHyphens = 28
    fromEnum AttrTypeOverline = 29
    fromEnum AttrTypeOverlineColor = 30
    fromEnum AttrTypeLineHeight = 31
    fromEnum AttrTypeAbsoluteLineHeight = 32
    fromEnum AttrTypeTextTransform = 33
    fromEnum AttrTypeWord = 34
    fromEnum AttrTypeSentence = 35
    fromEnum AttrTypeBaselineShift = 36
    fromEnum AttrTypeFontScale = 37
    fromEnum (AnotherAttrType k) = k

    toEnum 0 = AttrTypeInvalid
    toEnum 1 = AttrTypeLanguage
    toEnum 2 = AttrTypeFamily
    toEnum 3 = AttrTypeStyle
    toEnum 4 = AttrTypeWeight
    toEnum 5 = AttrTypeVariant
    toEnum 6 = AttrTypeStretch
    toEnum 7 = AttrTypeSize
    toEnum 8 = AttrTypeFontDesc
    toEnum 9 = AttrTypeForeground
    toEnum 10 = AttrTypeBackground
    toEnum 11 = AttrTypeUnderline
    toEnum 12 = AttrTypeStrikethrough
    toEnum 13 = AttrTypeRise
    toEnum 14 = AttrTypeShape
    toEnum 15 = AttrTypeScale
    toEnum 16 = AttrTypeFallback
    toEnum 17 = AttrTypeLetterSpacing
    toEnum 18 = AttrTypeUnderlineColor
    toEnum 19 = AttrTypeStrikethroughColor
    toEnum 20 = AttrTypeAbsoluteSize
    toEnum 21 = AttrTypeGravity
    toEnum 22 = AttrTypeGravityHint
    toEnum 23 = AttrTypeFontFeatures
    toEnum 24 = AttrTypeForegroundAlpha
    toEnum 25 = AttrTypeBackgroundAlpha
    toEnum 26 = AttrTypeAllowBreaks
    toEnum 27 = AttrTypeShow
    toEnum 28 = AttrTypeInsertHyphens
    toEnum 29 = AttrTypeOverline
    toEnum 30 = AttrTypeOverlineColor
    toEnum 31 = AttrTypeLineHeight
    toEnum 32 = AttrTypeAbsoluteLineHeight
    toEnum 33 = AttrTypeTextTransform
    toEnum 34 = AttrTypeWord
    toEnum 35 = AttrTypeSentence
    toEnum 36 = AttrTypeBaselineShift
    toEnum 37 = AttrTypeFontScale
    toEnum k = AnotherAttrType k

instance P.Ord AttrType where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes AttrType = '[]
instance O.HasParentTypes AttrType

foreign import ccall "pango_attr_type_get_type" c_pango_attr_type_get_type :: 
    IO GType

instance B.Types.TypedObject AttrType where
    glibType = c_pango_attr_type_get_type

instance B.Types.BoxedEnum AttrType

-- Enum Alignment
-- | @PangoAlignment@ describes how to align the lines of a @PangoLayout@
-- within the available space.
-- 
-- If the @PangoLayout@ is set to justify using 'GI.Pango.Objects.Layout.layoutSetJustify',
-- this only affects partial lines.
-- 
-- See 'GI.Pango.Objects.Layout.layoutSetAutoDir' for how text direction affects
-- the interpretation of @PangoAlignment@ values.
data Alignment = 
      AlignmentLeft
    -- ^ Put all available space on the right
    | AlignmentCenter
    -- ^ Center the line within the available space
    | AlignmentRight
    -- ^ Put all available space on the left
    | AnotherAlignment Int
    -- ^ Catch-all for unknown values
    deriving (Show, Eq)

instance P.Enum Alignment where
    fromEnum AlignmentLeft = 0
    fromEnum AlignmentCenter = 1
    fromEnum AlignmentRight = 2
    fromEnum (AnotherAlignment k) = k

    toEnum 0 = AlignmentLeft
    toEnum 1 = AlignmentCenter
    toEnum 2 = AlignmentRight
    toEnum k = AnotherAlignment k

instance P.Ord Alignment where
    compare a b = P.compare (P.fromEnum a) (P.fromEnum b)

type instance O.ParentTypes Alignment = '[]
instance O.HasParentTypes Alignment

foreign import ccall "pango_alignment_get_type" c_pango_alignment_get_type :: 
    IO GType

instance B.Types.TypedObject Alignment where
    glibType = c_pango_alignment_get_type

instance B.Types.BoxedEnum Alignment


