/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SRC_TRACE_PROCESSOR_IMPORTERS_COMMON_JIT_CACHE_H_
#define SRC_TRACE_PROCESSOR_IMPORTERS_COMMON_JIT_CACHE_H_

#include <cstdint>
#include <optional>

#include "perfetto/ext/base/flat_hash_map.h"
#include "perfetto/ext/base/string_view.h"
#include "perfetto/trace_processor/trace_blob_view.h"
#include "src/trace_processor/importers/common/address_range.h"
#include "src/trace_processor/importers/common/stack_profile_tracker.h"
#include "src/trace_processor/storage/trace_storage.h"
#include "src/trace_processor/tables/jit_tables_py.h"
#include "src/trace_processor/tables/profiler_tables_py.h"
#include "src/trace_processor/types/trace_processor_context.h"

namespace perfetto {
namespace trace_processor {

class VirtualMemoryMapping;
class UserMemoryMapping;

// JitCache represents a container of jit code generated by the same VM. Jitted
// functions can be "added" to the cache and stack frames added to the
// StackFrameTracker that lie in the code range for such a function will
// automatically be resolved (associate function name and source location).
//
// Jitted functions can also be deleted from the cache by overwriting existing
// ones.
class JitCache {
 public:
  struct SourceLocation {
    StringId file_name;
    uint32_t line_number;
  };

  JitCache(TraceProcessorContext* context,
           std::string name,
           UniquePid upid,
           AddressRange range)
      : context_(context), name_(std::move(name)), upid_(upid), range_(range) {}

  // Notify the cache that a jitted function was loaded at the given address
  // range. Any existing function that fully or partially overlaps with the new
  // function will be deleted.
  // The passed in listener will be notified each time a new Frame is created
  // for this function.
  tables::JitCodeTable::Id LoadCode(
      int64_t timestamp,
      UniqueTid utid,
      AddressRange code_range,
      StringId function_name,
      std::optional<SourceLocation> source_location,
      TraceBlobView native_code);
  tables::JitCodeTable::Id MoveCode(int64_t timestamp,
                                    UniqueTid utid,
                                    uint64_t from_code_start,
                                    uint64_t to_code_start);

  // Forward frame interning request.
  // MappingTracker allows other trackers to register ranges of memory for
  // which they need to control when a new frame is created. Jitted code can
  // move in memory over time, so the same program counter might refer to
  // different functions at different point in time. MappingTracker does
  // not keep track of such moves but instead delegates the creation of jitted
  // frames to a delegate.
  // Returns frame_id, and whether a new row as created or not.
  std::pair<FrameId, bool> InternFrame(VirtualMemoryMapping* mapping,
                                       uint64_t rel_pc,
                                       base::StringView function_name);

  // Simpleperf does not emit mmap events for jitted ranges (actually for non
  // file backed executable mappings). So have a way to generate a mapping on
  // the fly for FindMapping requests in a jitted region with no associated
  // mapping.
  UserMemoryMapping& CreateMapping();

 private:
  struct FrameKey {
    tables::StackProfileMappingTable::Id mapping_id;
    uint64_t rel_pc;
    struct Hasher {
      size_t operator()(const FrameKey& k) const {
        return static_cast<size_t>(
            base::Hasher::Combine(k.mapping_id.value, k.rel_pc));
      }
    };
    bool operator==(const FrameKey& other) const {
      return mapping_id == other.mapping_id && rel_pc == other.rel_pc;
    }
  };

  class JittedFunction {
   public:
    JittedFunction(tables::JitCodeTable::Id jit_code_id,
                   std::optional<uint32_t> symbol_set_id)
        : jit_code_id_(jit_code_id), symbol_set_id_(symbol_set_id) {}

    tables::JitCodeTable::Id jit_code_id() const { return jit_code_id_; }

    std::pair<FrameId, bool> InternFrame(TraceProcessorContext* context,
                                         FrameKey frame_key);

   private:
    const tables::JitCodeTable::Id jit_code_id_;
    const std::optional<uint32_t> symbol_set_id_;
    base::FlatHashMap<FrameKey, FrameId, FrameKey::Hasher> interned_frames_;
  };

  StringId Base64Encode(const TraceBlobView& data);

  TraceProcessorContext* const context_;
  const std::string name_;
  const UniquePid upid_;
  const AddressRange range_;
  AddressRangeMap<JittedFunction> functions_;
  base::FlatHashMap<FrameKey, FrameId, FrameKey::Hasher> unknown_frames_;
};

}  // namespace trace_processor
}  // namespace perfetto

#endif  // SRC_TRACE_PROCESSOR_IMPORTERS_COMMON_JIT_CACHE_H_
