﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock/BedrockRequest.h>
#include <aws/bedrock/Bedrock_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Bedrock {
namespace Model {

/**
 */
class GetGuardrailRequest : public BedrockRequest {
 public:
  AWS_BEDROCK_API GetGuardrailRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetGuardrail"; }

  AWS_BEDROCK_API Aws::String SerializePayload() const override;

  AWS_BEDROCK_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The unique identifier of the guardrail for which to get details. This can be
   * an ID or the ARN.</p>
   */
  inline const Aws::String& GetGuardrailIdentifier() const { return m_guardrailIdentifier; }
  inline bool GuardrailIdentifierHasBeenSet() const { return m_guardrailIdentifierHasBeenSet; }
  template <typename GuardrailIdentifierT = Aws::String>
  void SetGuardrailIdentifier(GuardrailIdentifierT&& value) {
    m_guardrailIdentifierHasBeenSet = true;
    m_guardrailIdentifier = std::forward<GuardrailIdentifierT>(value);
  }
  template <typename GuardrailIdentifierT = Aws::String>
  GetGuardrailRequest& WithGuardrailIdentifier(GuardrailIdentifierT&& value) {
    SetGuardrailIdentifier(std::forward<GuardrailIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the guardrail for which to get details. If you don't specify a
   * version, the response returns details for the <code>DRAFT</code> version.</p>
   */
  inline const Aws::String& GetGuardrailVersion() const { return m_guardrailVersion; }
  inline bool GuardrailVersionHasBeenSet() const { return m_guardrailVersionHasBeenSet; }
  template <typename GuardrailVersionT = Aws::String>
  void SetGuardrailVersion(GuardrailVersionT&& value) {
    m_guardrailVersionHasBeenSet = true;
    m_guardrailVersion = std::forward<GuardrailVersionT>(value);
  }
  template <typename GuardrailVersionT = Aws::String>
  GetGuardrailRequest& WithGuardrailVersion(GuardrailVersionT&& value) {
    SetGuardrailVersion(std::forward<GuardrailVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_guardrailIdentifier;

  Aws::String m_guardrailVersion;
  bool m_guardrailIdentifierHasBeenSet = false;
  bool m_guardrailVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Bedrock
}  // namespace Aws
