// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "google_apis/tasks/tasks_api_response_types.h"

#include <memory>

#include "base/json/json_reader.h"
#include "google_apis/common/time_util.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace google_apis::tasks {

using ::base::JSONReader;

TEST(TasksApiResponseTypesTest, CreatesTaskListsFromResponse) {
  const auto raw_task_lists = JSONReader::Read(R"(
      {
        "kind": "tasks#taskLists",
        "items": [
          {
            "kind": "tasks#taskList",
            "id": "qwerty",
            "title": "My Tasks 1",
            "updated": "2023-01-30T22:19:22.812Z"
          },
          {
            "kind": "tasks#taskList",
            "id": "asdfgh",
            "title": "My Tasks 2",
            "updated": "2022-12-21T23:38:22.590Z"
          }
        ]
      })");
  ASSERT_TRUE(raw_task_lists);

  const auto task_lists = TaskLists::CreateFrom(*raw_task_lists);
  ASSERT_TRUE(task_lists);
  EXPECT_TRUE(task_lists->next_page_token().empty());
  EXPECT_EQ(task_lists->items().size(), 2u);

  EXPECT_EQ(task_lists->items()[0]->id(), "qwerty");
  EXPECT_EQ(task_lists->items()[0]->title(), "My Tasks 1");
  EXPECT_EQ(util::FormatTimeAsString(task_lists->items()[0]->updated()),
            "2023-01-30T22:19:22.812Z");

  EXPECT_EQ(task_lists->items()[1]->id(), "asdfgh");
  EXPECT_EQ(task_lists->items()[1]->title(), "My Tasks 2");
  EXPECT_EQ(util::FormatTimeAsString(task_lists->items()[1]->updated()),
            "2022-12-21T23:38:22.590Z");
}

TEST(TasksApiResponseTypesTest, CreatesTaskListsWithNextPageTokenFromResponse) {
  const auto raw_task_lists = JSONReader::Read(R"(
      {
        "kind": "tasks#taskLists",
        "items": [],
        "nextPageToken": "qwerty"
      })");
  ASSERT_TRUE(raw_task_lists);

  const auto task_lists = TaskLists::CreateFrom(*raw_task_lists);
  ASSERT_TRUE(task_lists);
  EXPECT_EQ(task_lists->next_page_token(), "qwerty");
}

TEST(TasksApiResponseTypesTest, FailsToCreateTaskListsFromInvalidResponse) {
  const auto raw_task_lists = JSONReader::Read(R"(
      {
        "kind": "invalid_kind",
        "items": true
      })");
  ASSERT_TRUE(raw_task_lists);

  const auto task_lists = TaskLists::CreateFrom(*raw_task_lists);
  ASSERT_FALSE(task_lists);
}

TEST(TasksApiResponseTypesTest, CreatesTasksFromResponse) {
  const auto raw_tasks = JSONReader::Read(R"(
      {
        "kind": "tasks#tasks",
        "items": [
          {
            "id": "qwe",
            "title": "Completed child task",
            "parent": "asd",
            "position": "00000000000000000000",
            "status": "completed"
          },
          {
            "id": "asd",
            "title": "Parent task",
            "position": "00000000000000000001",
            "status": "needsAction",
            "due": "2023-04-19T00:00:00.000Z",
            "notes": "Lorem ipsum dolor sit amet"
          }
        ]
      })");
  ASSERT_TRUE(raw_tasks);

  const auto tasks = Tasks::CreateFrom(*raw_tasks);
  ASSERT_TRUE(tasks);
  EXPECT_TRUE(tasks->next_page_token().empty());
  EXPECT_EQ(tasks->items().size(), 2u);

  EXPECT_EQ(tasks->items()[0]->id(), "qwe");
  EXPECT_EQ(tasks->items()[0]->title(), "Completed child task");
  EXPECT_EQ(tasks->items()[0]->status(), Task::Status::kCompleted);
  EXPECT_EQ(tasks->items()[0]->parent_id(), "asd");
  EXPECT_EQ(tasks->items()[0]->position(), "00000000000000000000");
  EXPECT_FALSE(tasks->items()[0]->due());
  EXPECT_TRUE(tasks->items()[0]->notes().empty());

  EXPECT_EQ(tasks->items()[1]->id(), "asd");
  EXPECT_EQ(tasks->items()[1]->title(), "Parent task");
  EXPECT_EQ(tasks->items()[1]->status(), Task::Status::kNeedsAction);
  EXPECT_TRUE(tasks->items()[1]->parent_id().empty());
  EXPECT_EQ(tasks->items()[1]->position(), "00000000000000000001");
  EXPECT_EQ(util::FormatTimeAsString(tasks->items()[1]->due().value()),
            "2023-04-19T00:00:00.000Z");
  EXPECT_EQ(tasks->items()[1]->notes(), "Lorem ipsum dolor sit amet");
}

TEST(TasksApiResponseTypesTest, CreatesTasksWithNextPageTokenFromResponse) {
  const auto raw_tasks = JSONReader::Read(R"(
      {
        "kind": "tasks#tasks",
        "items": [],
        "nextPageToken": "qwerty"
      })");
  ASSERT_TRUE(raw_tasks);

  const auto tasks = Tasks::CreateFrom(*raw_tasks);
  ASSERT_TRUE(tasks);
  EXPECT_EQ(tasks->next_page_token(), "qwerty");
}

TEST(TasksApiResponseTypesTest, ConvertsTaskStatusToString) {
  EXPECT_EQ(Task::StatusToString(Task::Status::kCompleted), "completed");
  EXPECT_EQ(Task::StatusToString(Task::Status::kNeedsAction), "needsAction");
}

TEST(TasksApiResponseTypesTest, ConvertsTaskLinks) {
  const auto raw_tasks = JSONReader::Read(R"(
      {
        "kind": "tasks#tasks",
        "items": [
          {
            "id": "qwerty",
            "links": [
              {"type": "email"},
              {"type": "something unsupported yet"}
            ]
          }
        ]
      })");
  ASSERT_TRUE(raw_tasks);

  const auto tasks = Tasks::CreateFrom(*raw_tasks);
  ASSERT_TRUE(tasks);
  ASSERT_EQ(tasks->items().size(), 1u);

  EXPECT_EQ(tasks->items().at(0)->id(), "qwerty");

  const auto& links = tasks->items().at(0)->links();
  ASSERT_EQ(links.size(), 2u);
  EXPECT_EQ(links.at(0)->type(), TaskLink::Type::kEmail);
  EXPECT_EQ(links.at(1)->type(), TaskLink::Type::kUnknown);
}

TEST(TasksApiResponseTypesTest, FailsToCreateTasksFromInvalidResponse) {
  const auto raw_tasks = JSONReader::Read(R"(
      {
        "kind": "invalid_kind",
        "items": true
      })");
  ASSERT_TRUE(raw_tasks);

  const auto tasks = Tasks::CreateFrom(*raw_tasks);
  ASSERT_FALSE(tasks);
}

}  // namespace google_apis::tasks
