#!/bin/bash
# A script to configure the module's debug log level.
#
# Note: Please do not modify this script directly,
# as modifying this script will invalidate this script.
set -e

readonly pkg_name="systemd-coredump"
readonly conf_dir="/etc/systemd/coredump.conf.d/"
readonly override_conf_file="$conf_dir/98-deepin-debug.conf"

debug="off"

is_root() {
  local current_uid
  current_uid=$(id -u)

  if [ "$current_uid" -ne 0 ]; then
    echo "You need to have root privileges to run this script."
    return 1
  fi
  return 0
}

can_exec() {
  if ! is_root; then
    return 1
  fi
  if [ $# -ne 1 ]; then
    echo "Invalid argument"
    return 1
  fi
  if [ "$1" != "on" ] && [ "$1" != "off" ]; then
    echo "Invalid argument,use 'on' or 'off"
    return 1
  fi
  debug=$1
}

check_package_installed() {
    local package_name="$1"
    dpkg-query -Wf='${db:Status-Abbrev}' "$package_name" 2>/dev/null | grep -q '^ii'
}

install_coredump_pkg() {
  local package_version
  if check_package_installed "$pkg_name"; then
    echo "$pkg_name already installed"
    return 0
  fi
  package_version=$(dpkg -s "systemd" 2>/dev/null | grep '^Version:' | awk '{print $2}')
  if [ -z "$package_version" ]; then
    echo "Cannot find package_version for systemd"
    return 1
  fi
  apt-get install -y "${pkg_name}=${package_version}"
}

update_or_delete_conf() {
    local level=$1
    local conf_file=$2

    case "$level" in
        "on")
            # Enable: remove override file to use default settings
            if [ -f "$conf_file" ]; then
                rm -f "$conf_file"
                echo "Enabled coredump: removed $conf_file"
            else
                echo "Coredump is already enabled (no override config)"
            fi
            ;;
        "off")
            # Disable: create override with Storage=none
            {
              echo "[Coredump]"
              echo "Storage=none"
            } > "$conf_file"
            echo "Disabled coredump: created $conf_file"
            ;;
    esac
}


main() {
  if ! can_exec "$@"; then
    exit 1
  fi

  mkdir -p "$conf_dir"
  install_coredump_pkg  
  update_or_delete_conf "$debug" "$override_conf_file"
}

main "$@"