//! `POST /_matrix/identity/*/validate/email/requestToken`
//!
//! Create a session for validating an email.

pub mod v2 {
    //! `/v2/` ([spec])
    //!
    //! [spec]: https://spec.matrix.org/latest/identity-service-api/#post_matrixidentityv2validateemailrequesttoken

    use js_int::UInt;
    use ruma_common::{
        api::{request, response, Metadata},
        metadata, OwnedClientSecret, OwnedSessionId,
    };

    const METADATA: Metadata = metadata! {
        method: POST,
        rate_limited: false,
        authentication: AccessToken,
        history: {
            1.0 => "/_matrix/identity/v2/validate/email/requestToken",
        }
    };

    /// Request type for the `create_email_validation_session` endpoint.
    #[request]
    pub struct Request {
        /// A unique string generated by the client, and used to identify the validation attempt.
        pub client_secret: OwnedClientSecret,

        /// The email address to validate.
        pub email: String,

        /// The server will only send an email if the send_attempt is a number greater than the
        /// most recent one which it has seen, scoped to that email + client_secret pair.
        pub send_attempt: UInt,

        /// When the validation is completed, the identity server will redirect the user to this
        /// URL.
        #[serde(skip_serializing_if = "Option::is_none")]
        pub next_link: Option<String>,
    }

    /// Response type for the `create_email_validation_session` endpoint.
    #[response]
    pub struct Response {
        /// The session ID.
        ///
        /// Session IDs are opaque strings generated by the identity server.
        pub sid: OwnedSessionId,
    }

    impl Request {
        /// Create a new `Request` with the given client secret, email ID, `send_attempt` number,
        /// and the link to redirect to after validation.
        pub fn new(
            client_secret: OwnedClientSecret,
            email: String,
            send_attempt: UInt,
            next_link: Option<String>,
        ) -> Self {
            Self { client_secret, email, send_attempt, next_link }
        }
    }

    impl Response {
        /// Create a new `Response` with the given session ID.
        pub fn new(sid: OwnedSessionId) -> Self {
            Self { sid }
        }
    }
}
