// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- > K+K- pi0pi0
  class BESIII_2020_I1775344 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2020_I1775344);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // histograms
      for (double eVal : allowedEnergies()) {

        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        if (en == "2125"s) {
          book(_h[en+"KK"]   ,7,1,1);
          book(_h[en+"pipi"] ,7,1,2);
          book(_h[en+"Kpi"]  ,7,1,3);
          book(_h[en+"KKpi"] ,7,1,4);
          book(_h[en+"Kpipi"],7,1,5);
        }
        else if (en == "2396"s) {
          book(_h[en+"KK"]   ,8,1,1);
          book(_h[en+"pipi"] ,8,1,2);
          book(_h[en+"Kpi"]  ,8,1,3);
          book(_h[en+"KKpi"] ,8,1,4);
          book(_h[en+"Kpipi"],8,1,5);
        }
      }
      raiseBeamErrorIf(_sqs.empty());

      for (size_t ix=0; ix<6; ++ix) {
        book(_c[ix], 1+ix, 1, 1);
      }
      for (const string& en : _c[0].binning().edges<0>()) {
        const double eval = stod(en);
        if (isCompatibleWithSqrtS(eval)) {
          _edge = en; break;
        }
      }
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // find the final-state particles
      map<long,int> nCount;
      int ntotal(0);
      Particles Kp, pi0;
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
        if (p.abspid()==321)    Kp += p;
        else if (p.pid()==111)  pi0 += p;
      }
      // intermediates
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::abspid==100321 or
                                             Cuts::abspid==10323  or
                                             Cuts::abspid==20323  or
                                             Cuts::pid   ==333    or
                                             Cuts::abspid==323   )) {
        if (p.children().empty()) continue;
        map<long,int> nRes=nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        // X-/+ with K+/-
        if ((p.abspid()==100321 || p.abspid()== 10323 || p.abspid()==20323) && ncount==1) {
          bool matched = true;
          int Kid = -p.pid()/p.abspid()*321;
          for (const auto& val : nRes) {
            if (val.first==Kid) {
              if (val.second!=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            if (p.abspid()==100321)       _c[2]->fill(_edge);
            else if (p.abspid()== 20323)  _c[3]->fill(_edge);
            else if (p.abspid()==10323)   _c[4]->fill(_edge);
          }
        }
        else if (p.pid()==333 && ncount==2) { // phi + 2pi0
          bool matched = true;
          for (const auto& val : nRes) {
            if (val.first==111) {
              if (val.second!=2) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched)  _c[1]->fill(_edge);
        }
        else if (p.abspid()==323) { // K*K*
          for (const Particle& p2 : ufs.particles(Cuts::pid==-p.pid())) {
            map<long,int> nResB = nRes;
            int ncountB = ncount;
            findChildren(p2,nResB,ncountB);
            if (ncountB!=0) continue;
            bool matched = true;
            for (const auto& val : nResB) {
              if (val.second!=0) {
                matched = false;
                break;
              }
            }
            if (matched)  _c[5]->fill(_edge);
          }
        }
      }
      // final-state
      if (ntotal==4 && nCount[321]==1 && nCount[-321]==1 && nCount[111]==2) {
        _c[0]->fill(_edge);
        if (_sqs == "2125"s || _sqs == "2396"s) {
          FourMomentum pKK = Kp[0].mom()+Kp[1].mom();
          _h[_sqs+"KK"]->fill(pKK.mass());
          FourMomentum pPi = pi0[0].mom()+pi0[1].mom();
          _h[_sqs+"pipi"]->fill(pPi.mass());
          for (size_t ix=0; ix<2; ++ix) {
            _h[_sqs+"KKpi"]->fill((pKK+pi0[ix].mom()).mass());
            _h[_sqs+"Kpipi"]->fill((pPi+ Kp[ix].mom()).mass());
            for (size_t iy=0; iy<2; ++iy) {
              _h[_sqs+"Kpi"]->fill((Kp[ix].mom()+pi0[iy].mom()).mass());
            }
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_c, crossSection()/ sumOfWeights() /nanobarn);
      normalize(_h);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _c[6];
    map<string,Histo1DPtr> _h;
    string _sqs = "", _edge = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(BESIII_2020_I1775344);

}
