// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/DileptonFinder.hh"

namespace Rivet {


  /// Drell Yan measurements $pp \to \mu^+\mu^- +X $ at $\sqrt{s} = 44$ and $62$ GeV at CERN ISR
  class R209_1982_I168182 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(R209_1982_I168182);

    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      DileptonFinder zfinder(91.2*GeV, 0.1, Cuts::abseta < 10. && Cuts::abspid == PID::MUON,
                             Cuts::massIn(3.5*GeV, 18.0*GeV), LeptonOrigin::PROMPT, PhotonOrigin::NONE);
      declare(zfinder, "DileptonFinder");

      // Book histograms
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        if (en == "44"s) {
          book(_h[en+"M"], 1, 1, 2);
        }
        else {
          book(_h[en+"M"],  1, 1, 1);
          book(_h[en+"pT"], 2, 1, 1);
        }
        const int Nbin = 50;
        book(_n[en+"m"],  "DiMuon_mass_"+en, Nbin,  0.0, 30.0);
        book(_n[en+"pT"], "DiMuon_pT_"+en,   Nbin,  0.0, 20.0);
        book(_n[en+"y"],  "DiMuon_y_"+en,    Nbin, -8.0,  8.0);
        book(_n[en+"xF"], "DiMuon_xF_"+en,   Nbin, -1.5,  1.5);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const DileptonFinder& zfinder = apply<DileptonFinder>(event, "DileptonFinder");
      if (zfinder.particles().size() >= 1) {

        double Zmass = zfinder.bosons()[0].mom().mass()/GeV;
        double Zpt   = zfinder.bosons()[0].mom().pT()/GeV;
        double Zpl   = zfinder.bosons()[0].mom().pz()/GeV;
        double Zy    = zfinder.bosons()[0].mom().rap();
        double xf = 2.*Zpl/sqrtS() ;

        _n[_sqs+"xF"]->fill(xf);
        _n[_sqs+"m"]->fill(Zmass/GeV);
        _n[_sqs+"pT"]->fill(Zpt);
        _n[_sqs+"y"]->fill(Zy);
        if (Zmass > 0) _h[_sqs+"M"]->fill(Zmass);
        if (_sqs == "62"s) {
          if (Zmass > 5. && Zmass < 8.) {
            if (Zpt > 0)  _h[_sqs+"pT"]->fill(Zpt, 1.0/2.0/Zpt);
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_n);
      scale(_h, crossSection()/nanobarn/sumOfWeights());
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h, _n;
    /// @}

    /// Energy comparison tolerance
    string _sqs = "";

  };


  RIVET_DECLARE_PLUGIN(R209_1982_I168182);

}
