// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief Charged multiplicity below Z pole based on ALEPH Z pole analysis
  ///
  /// @author Peter Richardson
  class AMY_1990_I295160 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(AMY_1990_I295160);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(ChargedFinalState(), "CFS");
      book(_p, 2, 1, 1);
      size_t ie = 0;
      for (const string& en : _p.binning().edges<0>()) {
          const size_t idx = en.find("-");
          if (idx != string::npos) continue;
          const double eval = stod(en);
          if (isCompatibleWithSqrtS(eval))  _sqs = en;
          book(_c[en], "_sumW_"+en);
          if (en == "57.0"s) {
            book(_h[en+"avg"], 1, 1, 9);
          }
          book(_h[en+"ch"], 1, 1, ++ie);
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      _c[_sqs]->fill();
      const FinalState& cfs = apply<FinalState>(event, "CFS");
      _h[_sqs+"ch"]->fill(cfs.size());
      _p->fill(_sqs, cfs.size());
      if (_sqs == "57.0"s) {
        _h[_sqs+"avg"]->fill(cfs.size());
        _p->fill("50.0 - 61.4"s, cfs.size());
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_c, crossSectionPerEvent());
      scale(_h, crossSectionPerEvent());
      for (auto& item : _h) {
        const double w = _c[item.first.substr(0,4)]->sumW();
        if (!isZero(w))  scale(item.second, 100.0/w);
      }
    }

    /// @}


  private:

    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<int>> _h;
    BinnedProfilePtr<string> _p;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(AMY_1990_I295160);

}
