/*
 * endgame -- plays through a chess game ending.  enjoy.
 *
 * Copyright (C) 2002 Blair Tennessy (tennessy@cs.ubc.ca)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#ifndef __CHESSGAMES_H__
#define __CHESSGAMES_H__

#define DESC_STR_LEN 64 /* Length of the game description. */
#define SAN_STR_LEN 8   /* Length of the move description */

/** structure for a chess game */
typedef struct {
    char desc[DESC_STR_LEN + SAN_STR_LEN + 1]; /* game + move description */
    char fen[80];  /* Original board configuration / FEN format */
    int movecount; /* total moves */
    struct {       /* moves in game - as uci and san strings */
        char uci[8];
        char san[8];
    } moves[128];
} ChessGame;

#define GAMES_NUMBER 34
static ChessGame games[GAMES_NUMBER] = {
    {.desc = "A. Anderssen / L. Kieseritzky / 1851\n\"The immortal game\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 45,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "f2f4", .san = "f4"},    {.uci = "e5f4", .san = "exf4"},
             {.uci = "f1c4", .san = "Bc4"},   {.uci = "d8h4", .san = "Qh4+"},
             {.uci = "e1f1", .san = "Kf1"},   {.uci = "b7b5", .san = "b5"},
             {.uci = "c4b5", .san = "Bxb5"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "h4h6", .san = "Qh6"},
             {.uci = "d2d3", .san = "d3"},    {.uci = "f6h5", .san = "Nh5"},
             {.uci = "f3h4", .san = "Nh4"},   {.uci = "h6g5", .san = "Qg5"},
             {.uci = "h4f5", .san = "Nf5"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "g2g4", .san = "g4"},    {.uci = "h5f6", .san = "Nf6"},
             {.uci = "h1g1", .san = "Rg1"},   {.uci = "c6b5", .san = "cxb5"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "g5g6", .san = "Qg6"},
             {.uci = "h4h5", .san = "h5"},    {.uci = "g6g5", .san = "Qg5"},
             {.uci = "d1f3", .san = "Qf3"},   {.uci = "f6g8", .san = "Ng8"},
             {.uci = "c1f4", .san = "Bxf4"},  {.uci = "g5f6", .san = "Qf6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "f8c5", .san = "Bc5"},
             {.uci = "c3d5", .san = "Nd5"},   {.uci = "f6b2", .san = "Qxb2"},
             {.uci = "f4d6", .san = "Bd6"},   {.uci = "c5g1", .san = "Bxg1"},
             {.uci = "e4e5", .san = "e5"},    {.uci = "b2a1", .san = "Qxa1+"},
             {.uci = "f1e2", .san = "Ke2"},   {.uci = "b8a6", .san = "Na6"},
             {.uci = "f5g7", .san = "Nxg7+"}, {.uci = "e8d8", .san = "Kd8"},
             {.uci = "f3f6", .san = "Qf6+"},  {.uci = "g8f6", .san = "Nxf6"},
             {.uci = "d6e7", .san = "Be7#"},
         }},
    {.desc = "A. Hoffmann / A. Petrov / 1844\n\"Petrov's immortal\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 40,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},   {.uci = "e7e5", .san = "e5"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "b8c6", .san = "Nc6"},
             {.uci = "f1c4", .san = "Bc4"},  {.uci = "f8c5", .san = "Bc5"},
             {.uci = "c2c3", .san = "c3"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "d2d4", .san = "d4"},   {.uci = "e5d4", .san = "exd4"},
             {.uci = "e4e5", .san = "e5"},   {.uci = "f6e4", .san = "Ne4"},
             {.uci = "c4d5", .san = "Bd5"},  {.uci = "e4f2", .san = "Nxf2"},
             {.uci = "e1f2", .san = "Kxf2"}, {.uci = "d4c3", .san = "dxc3+"},
             {.uci = "f2g3", .san = "Kg3"},  {.uci = "c3b2", .san = "cxb2"},
             {.uci = "c1b2", .san = "Bxb2"}, {.uci = "c6e7", .san = "Ne7"},
             {.uci = "f3g5", .san = "Ng5"},  {.uci = "e7d5", .san = "Nxd5"},
             {.uci = "g5f7", .san = "Nxf7"}, {.uci = "e8g8", .san = "O-O"},
             {.uci = "f7d8", .san = "Nxd8"}, {.uci = "c5f2", .san = "Bf2+"},
             {.uci = "g3h3", .san = "Kh3"},  {.uci = "d7d6", .san = "d6+"},
             {.uci = "e5e6", .san = "e6"},   {.uci = "d5f4", .san = "Nf4+"},
             {.uci = "h3g4", .san = "Kg4"},  {.uci = "f4e6", .san = "Nxe6"},
             {.uci = "d8e6", .san = "Nxe6"}, {.uci = "c8e6", .san = "Bxe6+"},
             {.uci = "g4g5", .san = "Kg5"},  {.uci = "f8f5", .san = "Rf5+"},
             {.uci = "g5g4", .san = "Kg4"},  {.uci = "h7h5", .san = "h5+"},
             {.uci = "g4h3", .san = "Kh3"},  {.uci = "f5f3", .san = "Rf3#"},
         }},
    {.desc = "A. Anderssen / J. Dufresne / 1852\n\"The evergreen game\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 47,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "b8c6", .san = "Nc6"},
             {.uci = "f1c4", .san = "Bc4"},   {.uci = "f8c5", .san = "Bc5"},
             {.uci = "b2b4", .san = "b4"},    {.uci = "c5b4", .san = "Bxb4"},
             {.uci = "c2c3", .san = "c3"},    {.uci = "b4a5", .san = "Ba5"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "e5d4", .san = "exd4"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "d4d3", .san = "d3"},
             {.uci = "d1b3", .san = "Qb3"},   {.uci = "d8f6", .san = "Qf6"},
             {.uci = "e4e5", .san = "e5"},    {.uci = "f6g6", .san = "Qg6"},
             {.uci = "f1e1", .san = "Re1"},   {.uci = "g8e7", .san = "Nge7"},
             {.uci = "c1a3", .san = "Ba3"},   {.uci = "b7b5", .san = "b5"},
             {.uci = "b3b5", .san = "Qxb5"},  {.uci = "a8b8", .san = "Rb8"},
             {.uci = "b5a4", .san = "Qa4"},   {.uci = "a5b6", .san = "Bb6"},
             {.uci = "b1d2", .san = "Nbd2"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "d2e4", .san = "Ne4"},   {.uci = "g6f5", .san = "Qf5"},
             {.uci = "c4d3", .san = "Bxd3"},  {.uci = "f5h5", .san = "Qh5"},
             {.uci = "e4f6", .san = "Nf6+"},  {.uci = "g7f6", .san = "gxf6"},
             {.uci = "e5f6", .san = "exf6"},  {.uci = "h8g8", .san = "Rg8"},
             {.uci = "a1d1", .san = "Rad1"},  {.uci = "h5f3", .san = "Qxf3"},
             {.uci = "e1e7", .san = "Rxe7+"}, {.uci = "c6e7", .san = "Nxe7"},
             {.uci = "a4d7", .san = "Qxd7+"}, {.uci = "e8d7", .san = "Kxd7"},
             {.uci = "d3f5", .san = "Bf5+"},  {.uci = "d7e8", .san = "Ke8"},
             {.uci = "f5d7", .san = "Bd7+"},  {.uci = "e8f8", .san = "Kf8"},
             {.uci = "a3e7", .san = "Bxe7#"},
         }},
    {.desc = "P. Morphy / D. Karl / 1858\n\"The opera game\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 33,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "d7d6", .san = "d6"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "c8g4", .san = "Bg4"},
             {.uci = "d4e5", .san = "dxe5"},  {.uci = "g4f3", .san = "Bxf3"},
             {.uci = "d1f3", .san = "Qxf3"},  {.uci = "d6e5", .san = "dxe5"},
             {.uci = "f1c4", .san = "Bc4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "f3b3", .san = "Qb3"},   {.uci = "d8e7", .san = "Qe7"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "c1g5", .san = "Bg5"},   {.uci = "b7b5", .san = "b5"},
             {.uci = "c3b5", .san = "Nxb5"},  {.uci = "c6b5", .san = "cxb5"},
             {.uci = "c4b5", .san = "Bxb5+"}, {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "e1c1", .san = "O-O-O"}, {.uci = "a8d8", .san = "Rd8"},
             {.uci = "d1d7", .san = "Rxd7"},  {.uci = "d8d7", .san = "Rxd7"},
             {.uci = "h1d1", .san = "Rd1"},   {.uci = "e7e6", .san = "Qe6"},
             {.uci = "b5d7", .san = "Bxd7+"}, {.uci = "f6d7", .san = "Nxd7"},
             {.uci = "b3b8", .san = "Qb8+"},  {.uci = "d7b8", .san = "Nxb8"},
             {.uci = "d1d8", .san = "Rd8#"},
         }},
    {.desc = "G. Rotlewi / A. Rubinstein / 1907\n\"Rotlewi's immortal game\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 50,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "d7d5", .san = "d5"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "e2e3", .san = "e3"},   {.uci = "c7c5", .san = "c5"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "b8c6", .san = "Nc6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "d4c5", .san = "dxc5"}, {.uci = "f8c5", .san = "Bxc5"},
             {.uci = "a2a3", .san = "a3"},   {.uci = "a7a6", .san = "a6"},
             {.uci = "b2b4", .san = "b4"},   {.uci = "c5d6", .san = "Bd6"},
             {.uci = "c1b2", .san = "Bb2"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "d1d2", .san = "Qd2"},  {.uci = "d8e7", .san = "Qe7"},
             {.uci = "f1d3", .san = "Bd3"},  {.uci = "d5c4", .san = "dxc4"},
             {.uci = "d3c4", .san = "Bxc4"}, {.uci = "b7b5", .san = "b5"},
             {.uci = "c4d3", .san = "Bd3"},  {.uci = "f8d8", .san = "Rd8"},
             {.uci = "d2e2", .san = "Qe2"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "c6e5", .san = "Ne5"},
             {.uci = "f3e5", .san = "Nxe5"}, {.uci = "d6e5", .san = "Bxe5"},
             {.uci = "f2f4", .san = "f4"},   {.uci = "e5c7", .san = "Bc7"},
             {.uci = "e3e4", .san = "e4"},   {.uci = "a8c8", .san = "Rac8"},
             {.uci = "e4e5", .san = "e5"},   {.uci = "c7b6", .san = "Bb6+"},
             {.uci = "g1h1", .san = "Kh1"},  {.uci = "f6g4", .san = "Ng4"},
             {.uci = "d3e4", .san = "Be4"},  {.uci = "e7h4", .san = "Qh4"},
             {.uci = "g2g3", .san = "g3"},   {.uci = "c8c3", .san = "Rxc3"},
             {.uci = "g3h4", .san = "gxh4"}, {.uci = "d8d2", .san = "Rd2"},
             {.uci = "e2d2", .san = "Qxd2"}, {.uci = "b7e4", .san = "Bxe4+"},
             {.uci = "d2g2", .san = "Qg2"},  {.uci = "c3h3", .san = "Rh3"},
         }},
    {.desc = "E. Bogoljubov / A. Alekhine / 1922.09.21 / Hastings",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 106,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},
             {.uci = "f7f5", .san = "f5"},
             {.uci = "c2c4", .san = "c4"},
             {.uci = "g8f6", .san = "Nf6"},
             {.uci = "g2g3", .san = "g3"},
             {.uci = "e7e6", .san = "e6"},
             {.uci = "f1g2", .san = "Bg2"},
             {.uci = "f8b4", .san = "Bb4+"},
             {.uci = "c1d2", .san = "Bd2"},
             {.uci = "b4d2", .san = "Bxd2+"},
             {.uci = "b1d2", .san = "Nxd2"},
             {.uci = "b8c6", .san = "Nc6"},
             {.uci = "g1f3", .san = "Ngf3"},
             {.uci = "e8g8", .san = "O-O"},
             {.uci = "e1g1", .san = "O-O"},
             {.uci = "d7d6", .san = "d6"},
             {.uci = "d1b3", .san = "Qb3"},
             {.uci = "g8h8", .san = "Kh8"},
             {.uci = "b3c3", .san = "Qc3"},
             {.uci = "e6e5", .san = "e5"},
             {.uci = "e2e3", .san = "e3"},
             {.uci = "a7a5", .san = "a5"},
             {.uci = "b2b3", .san = "b3"},
             {.uci = "d8e8", .san = "Qe8"},
             {.uci = "a2a3", .san = "a3"},
             {.uci = "e8h5", .san = "Qh5"},
             {.uci = "h2h4", .san = "h4"},
             {.uci = "f6g4", .san = "Ng4"},
             {.uci = "f3g5", .san = "Ng5"},
             {.uci = "c8d7", .san = "Bd7"},
             {.uci = "f2f3", .san = "f3"},
             {.uci = "g4f6", .san = "Nf6"},
             {.uci = "f3f4", .san = "f4"},
             {.uci = "e5e4", .san = "e4"},
             {.uci = "f1d1", .san = "Rfd1"},
             {.uci = "h7h6", .san = "h6"},
             {.uci = "g5h3", .san = "Nh3"},
             {.uci = "d6d5", .san = "d5"},
             {.uci = "d2f1", .san = "Nf1"},
             {.uci = "c6e7", .san = "Ne7"},
             {.uci = "a3a4", .san = "a4"},
             {.uci = "e7c6", .san = "Nc6"},
             {.uci = "d1d2", .san = "Rd2"},
             {.uci = "c6b4", .san = "Nb4"},
             {.uci = "g2h1", .san = "Bh1"},
             {.uci = "h5e8", .san = "Qe8"},
             {.uci = "d2g2", .san = "Rg2"},
             {.uci = "d5c4", .san = "dxc4"},
             {.uci = "b3c4", .san = "bxc4"},
             {.uci = "d7a4", .san = "Bxa4"},
             {.uci = "h3f2", .san = "Nf2"},
             {.uci = "a4d7", .san = "Bd7"},
             {.uci = "f1d2", .san = "Nd2"},
             {.uci = "b7b5", .san = "b5"},
             {.uci = "f2d1", .san = "Nd1"},
             {.uci = "b4d3", .san = "Nd3"},
             {.uci = "a1a5", .san = "Rxa5"},
             {.uci = "b5b4", .san = "b4"},
             {.uci = "a5a8", .san = "Rxa8"},
             {.uci = "b4c3", .san = "bxc3"},
             {.uci = "a8e8", .san = "Rxe8"},
             {.uci = "c3c2", .san = "c2"},
             {.uci = "e8f8", .san = "Rxf8+"},
             {.uci = "h8h7", .san = "Kh7"},
             {.uci = "d1f2", .san = "Nf2"},
             {.uci = "c2c1q", .san = "c1=Q+"},
             {.uci = "d2f1", .san = "Nf1"},
             {.uci = "d3e1", .san = "Ne1"},
             {.uci = "g2h2", .san = "Rh2"},
             {.uci = "c1c4", .san = "Qxc4"},
             {.uci = "f8b8", .san = "Rb8"},
             {.uci = "d7b5", .san = "Bb5"},
             {.uci = "b8b5", .san = "Rxb5"},
             {.uci = "c4b5", .san = "Qxb5"},
             {.uci = "g3g4", .san = "g4"},
             {.uci = "e1f3", .san = "Nf3+"},
             {.uci = "h1f3", .san = "Bxf3"},
             {.uci = "e4f3", .san = "exf3"},
             {.uci = "g4f5", .san = "gxf5"},
             {.uci = "b5e2", .san = "Qe2"},
             {.uci = "d4d5", .san = "d5"},
             {.uci = "h7g8", .san = "Kg8"},
             {.uci = "h4h5", .san = "h5"},
             {.uci = "g8h7", .san = "Kh7"},
             {.uci = "e3e4", .san = "e4"},
             {.uci = "f6e4", .san = "Nxe4"},
             {.uci = "f2e4", .san = "Nxe4"},
             {.uci = "e2e4", .san = "Qxe4"},
             {.uci = "d5d6", .san = "d6"},
             {.uci = "c7d6", .san = "cxd6"},
             {.uci = "f5f6", .san = "f6"},
             {.uci = "g7f6", .san = "gxf6"},
             {.uci = "h2d2", .san = "Rd2"},
             {.uci = "e4e2", .san = "Qe2"},
             {.uci = "d2e2", .san = "Rxe2"},
             {.uci = "f3e2", .san = "fxe2"},
             {.uci = "g1f2", .san = "Kf2"},
             {.uci = "e2f1q", .san = "exf1=Q+"},
             {.uci = "f2f1", .san = "Kxf1"},
             {.uci = "h7g7", .san = "Kg7"},
             {.uci = "f1e2", .san = "Ke2"},
             {.uci = "g7f7", .san = "Kf7"},
             {.uci = "e2e3", .san = "Ke3"},
             {.uci = "f7e6", .san = "Ke6"},
             {.uci = "e3e4", .san = "Ke4"},
             {.uci = "d6d5", .san = "d5+"},
         }},
    {.desc = "Glucksberg / M. Najdorf / 1929\n\"The Polish immortal\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 44,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "f7f5", .san = "f5"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "d7d5", .san = "d5"},
             {.uci = "e2e3", .san = "e3"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "f1d3", .san = "Bd3"},  {.uci = "f8d6", .san = "Bd6"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "c3e2", .san = "Ne2"},  {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "f3g5", .san = "Ng5"},  {.uci = "d6h2", .san = "Bxh2+"},
             {.uci = "g1h1", .san = "Kh1"},  {.uci = "f6g4", .san = "Ng4"},
             {.uci = "f2f4", .san = "f4"},   {.uci = "d8e8", .san = "Qe8"},
             {.uci = "g2g3", .san = "g3"},   {.uci = "e8h5", .san = "Qh5"},
             {.uci = "h1g2", .san = "Kg2"},  {.uci = "h2g1", .san = "Bg1"},
             {.uci = "e2g1", .san = "Nxg1"}, {.uci = "h5h2", .san = "Qh2+"},
             {.uci = "g2f3", .san = "Kf3"},  {.uci = "e6e5", .san = "e5"},
             {.uci = "d4e5", .san = "dxe5"}, {.uci = "d7e5", .san = "Ndxe5+"},
             {.uci = "f4e5", .san = "fxe5"}, {.uci = "g4e5", .san = "Nxe5+"},
             {.uci = "f3f4", .san = "Kf4"},  {.uci = "e5g6", .san = "Ng6+"},
             {.uci = "f4f3", .san = "Kf3"},  {.uci = "f5f4", .san = "f4"},
             {.uci = "e3f4", .san = "exf4"}, {.uci = "c8g4", .san = "Bg4+"},
             {.uci = "f3g4", .san = "Kxg4"}, {.uci = "g6e5", .san = "Ne5+"},
             {.uci = "f4e5", .san = "fxe5"}, {.uci = "h7h5", .san = "h5#"},
         }},
    {.desc = "D. Byrne / R. J. Fischer / 1956\n\"The game of the century\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 82,
     .moves =
         {
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "g7g6", .san = "g6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "f8g7", .san = "Bg7"},
             {.uci = "d2d4", .san = "d4"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "c1f4", .san = "Bf4"},  {.uci = "d7d5", .san = "d5"},
             {.uci = "d1b3", .san = "Qb3"},  {.uci = "d5c4", .san = "dxc4"},
             {.uci = "b3c4", .san = "Qxc4"}, {.uci = "c7c6", .san = "c6"},
             {.uci = "e2e4", .san = "e4"},   {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "a1d1", .san = "Rd1"},  {.uci = "d7b6", .san = "Nb6"},
             {.uci = "c4c5", .san = "Qc5"},  {.uci = "c8g4", .san = "Bg4"},
             {.uci = "f4g5", .san = "Bg5"},  {.uci = "b6a4", .san = "Na4"},
             {.uci = "c5a3", .san = "Qa3"},  {.uci = "a4c3", .san = "Nxc3"},
             {.uci = "b2c3", .san = "bxc3"}, {.uci = "f6e4", .san = "Nxe4"},
             {.uci = "g5e7", .san = "Bxe7"}, {.uci = "d8b6", .san = "Qb6"},
             {.uci = "f1c4", .san = "Bc4"},  {.uci = "e4c3", .san = "Nxc3"},
             {.uci = "e7c5", .san = "Bc5"},  {.uci = "f8e8", .san = "Rfe8+"},
             {.uci = "e1f1", .san = "Kf1"},  {.uci = "g4e6", .san = "Be6"},
             {.uci = "c5b6", .san = "Bxb6"}, {.uci = "e6c4", .san = "Bxc4+"},
             {.uci = "f1g1", .san = "Kg1"},  {.uci = "c3e2", .san = "Ne2+"},
             {.uci = "g1f1", .san = "Kf1"},  {.uci = "e2d4", .san = "Nxd4+"},
             {.uci = "f1g1", .san = "Kg1"},  {.uci = "d4e2", .san = "Ne2+"},
             {.uci = "g1f1", .san = "Kf1"},  {.uci = "e2c3", .san = "Nc3+"},
             {.uci = "f1g1", .san = "Kg1"},  {.uci = "a7b6", .san = "axb6"},
             {.uci = "a3b4", .san = "Qb4"},  {.uci = "a8a4", .san = "Ra4"},
             {.uci = "b4b6", .san = "Qxb6"}, {.uci = "c3d1", .san = "Nxd1"},
             {.uci = "h2h3", .san = "h3"},   {.uci = "a4a2", .san = "Rxa2"},
             {.uci = "g1h2", .san = "Kh2"},  {.uci = "d1f2", .san = "Nxf2"},
             {.uci = "h1e1", .san = "Re1"},  {.uci = "e8e1", .san = "Rxe1"},
             {.uci = "b6d8", .san = "Qd8+"}, {.uci = "g7f8", .san = "Bf8"},
             {.uci = "f3e1", .san = "Nxe1"}, {.uci = "c4d5", .san = "Bd5"},
             {.uci = "e1f3", .san = "Nf3"},  {.uci = "f2e4", .san = "Ne4"},
             {.uci = "d8b8", .san = "Qb8"},  {.uci = "b7b5", .san = "b5"},
             {.uci = "h3h4", .san = "h4"},   {.uci = "h7h5", .san = "h5"},
             {.uci = "f3e5", .san = "Ne5"},  {.uci = "g8g7", .san = "Kg7"},
             {.uci = "h2g1", .san = "Kg1"},  {.uci = "f8c5", .san = "Bc5+"},
             {.uci = "g1f1", .san = "Kf1"},  {.uci = "e4g3", .san = "Ng3+"},
             {.uci = "f1e1", .san = "Ke1"},  {.uci = "c5b4", .san = "Bb4+"},
             {.uci = "e1d1", .san = "Kd1"},  {.uci = "d5b3", .san = "Bb3+"},
             {.uci = "d1c1", .san = "Kc1"},  {.uci = "g3e2", .san = "Ne2+"},
             {.uci = "c1b1", .san = "Kb1"},  {.uci = "e2c3", .san = "Nc3+"},
             {.uci = "b1c1", .san = "Kc1"},  {.uci = "a2c2", .san = "Rc2#"},
         }},
    {.desc = "G. Kasparov / V. Topalov / 1999\n\"Kasparov's immortal\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 87,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "d7d6", .san = "d6"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "g8f6", .san = "Nf6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "g7g6", .san = "g6"},
             {.uci = "c1e3", .san = "Be3"},   {.uci = "f8g7", .san = "Bg7"},
             {.uci = "d1d2", .san = "Qd2"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "f2f3", .san = "f3"},    {.uci = "b7b5", .san = "b5"},
             {.uci = "g1e2", .san = "Nge2"},  {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "e3h6", .san = "Bh6"},   {.uci = "g7h6", .san = "Bxh6"},
             {.uci = "d2h6", .san = "Qxh6"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "a2a3", .san = "a3"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "e1c1", .san = "O-O-O"}, {.uci = "d8e7", .san = "Qe7"},
             {.uci = "c1b1", .san = "Kb1"},   {.uci = "a7a6", .san = "a6"},
             {.uci = "e2c1", .san = "Nc1"},   {.uci = "e8c8", .san = "O-O-O"},
             {.uci = "c1b3", .san = "Nb3"},   {.uci = "e5d4", .san = "exd4"},
             {.uci = "d1d4", .san = "Rxd4"},  {.uci = "c6c5", .san = "c5"},
             {.uci = "d4d1", .san = "Rd1"},   {.uci = "d7b6", .san = "Nb6"},
             {.uci = "g2g3", .san = "g3"},    {.uci = "c8b8", .san = "Kb8"},
             {.uci = "b3a5", .san = "Na5"},   {.uci = "b7a8", .san = "Ba8"},
             {.uci = "f1h3", .san = "Bh3"},   {.uci = "d6d5", .san = "d5"},
             {.uci = "h6f4", .san = "Qf4+"},  {.uci = "b8a7", .san = "Ka7"},
             {.uci = "h1e1", .san = "Rhe1"},  {.uci = "d5d4", .san = "d4"},
             {.uci = "c3d5", .san = "Nd5"},   {.uci = "b6d5", .san = "Nbxd5"},
             {.uci = "e4d5", .san = "exd5"},  {.uci = "e7d6", .san = "Qd6"},
             {.uci = "d1d4", .san = "Rxd4"},  {.uci = "c5d4", .san = "cxd4"},
             {.uci = "e1e7", .san = "Re7+"},  {.uci = "a7b6", .san = "Kb6"},
             {.uci = "f4d4", .san = "Qxd4+"}, {.uci = "b6a5", .san = "Kxa5"},
             {.uci = "b2b4", .san = "b4+"},   {.uci = "a5a4", .san = "Ka4"},
             {.uci = "d4c3", .san = "Qc3"},   {.uci = "d6d5", .san = "Qxd5"},
             {.uci = "e7a7", .san = "Ra7"},   {.uci = "a8b7", .san = "Bb7"},
             {.uci = "a7b7", .san = "Rxb7"},  {.uci = "d5c4", .san = "Qc4"},
             {.uci = "c3f6", .san = "Qxf6"},  {.uci = "a4a3", .san = "Kxa3"},
             {.uci = "f6a6", .san = "Qxa6+"}, {.uci = "a3b4", .san = "Kxb4"},
             {.uci = "c2c3", .san = "c3+"},   {.uci = "b4c3", .san = "Kxc3"},
             {.uci = "a6a1", .san = "Qa1+"},  {.uci = "c3d2", .san = "Kd2"},
             {.uci = "a1b2", .san = "Qb2+"},  {.uci = "d2d1", .san = "Kd1"},
             {.uci = "h3f1", .san = "Bf1"},   {.uci = "d8d2", .san = "Rd2"},
             {.uci = "b7d7", .san = "Rd7"},   {.uci = "d2d7", .san = "Rxd7"},
             {.uci = "f1c4", .san = "Bxc4"},  {.uci = "b5c4", .san = "bxc4"},
             {.uci = "b2h8", .san = "Qxh8"},  {.uci = "d7d3", .san = "Rd3"},
             {.uci = "h8a8", .san = "Qa8"},   {.uci = "c4c3", .san = "c3"},
             {.uci = "a8a4", .san = "Qa4+"},  {.uci = "d1e1", .san = "Ke1"},
             {.uci = "f3f4", .san = "f4"},    {.uci = "f7f5", .san = "f5"},
             {.uci = "b1c1", .san = "Kc1"},   {.uci = "d3d2", .san = "Rd2"},
             {.uci = "a4a7", .san = "Qa7"},
         }},
    {.desc = "W. Yi / L. B. Bruzon / 2015.07.03\n6th Hainan Danzhou",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 71,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "c7c5", .san = "c5"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "a7a6", .san = "a6"},
             {.uci = "f1e2", .san = "Be2"},   {.uci = "b8c6", .san = "Nc6"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "c5d4", .san = "cxd4"},
             {.uci = "f3d4", .san = "Nxd4"},  {.uci = "d8c7", .san = "Qc7"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c1e3", .san = "Be3"},   {.uci = "f8e7", .san = "Be7"},
             {.uci = "f2f4", .san = "f4"},    {.uci = "d7d6", .san = "d6"},
             {.uci = "g1h1", .san = "Kh1"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "d1e1", .san = "Qe1"},   {.uci = "c6d4", .san = "Nxd4"},
             {.uci = "e3d4", .san = "Bxd4"},  {.uci = "b7b5", .san = "b5"},
             {.uci = "e1g3", .san = "Qg3"},   {.uci = "c8b7", .san = "Bb7"},
             {.uci = "a2a3", .san = "a3"},    {.uci = "a8d8", .san = "Rad8"},
             {.uci = "a1e1", .san = "Rae1"},  {.uci = "d8d7", .san = "Rd7"},
             {.uci = "e2d3", .san = "Bd3"},   {.uci = "c7d8", .san = "Qd8"},
             {.uci = "g3h3", .san = "Qh3"},   {.uci = "g7g6", .san = "g6"},
             {.uci = "f4f5", .san = "f5"},    {.uci = "e6e5", .san = "e5"},
             {.uci = "d4e3", .san = "Be3"},   {.uci = "f8e8", .san = "Re8"},
             {.uci = "f5g6", .san = "fxg6"},  {.uci = "h7g6", .san = "hxg6"},
             {.uci = "c3d5", .san = "Nd5"},   {.uci = "f6d5", .san = "Nxd5"},
             {.uci = "f1f7", .san = "Rxf7"},  {.uci = "g8f7", .san = "Kxf7"},
             {.uci = "h3h7", .san = "Qh7+"},  {.uci = "f7e6", .san = "Ke6"},
             {.uci = "e4d5", .san = "exd5+"}, {.uci = "e6d5", .san = "Kxd5"},
             {.uci = "d3e4", .san = "Be4+"},  {.uci = "d5e4", .san = "Kxe4"},
             {.uci = "h7f7", .san = "Qf7"},   {.uci = "e7f6", .san = "Bf6"},
             {.uci = "e3d2", .san = "Bd2+"},  {.uci = "e4d4", .san = "Kd4"},
             {.uci = "d2e3", .san = "Be3+"},  {.uci = "d4e4", .san = "Ke4"},
             {.uci = "f7b3", .san = "Qb3"},   {.uci = "e4f5", .san = "Kf5"},
             {.uci = "e1f1", .san = "Rf1+"},  {.uci = "f5g4", .san = "Kg4"},
             {.uci = "b3d3", .san = "Qd3"},   {.uci = "b7g2", .san = "Bxg2+"},
             {.uci = "h1g2", .san = "Kxg2"},  {.uci = "d8a8", .san = "Qa8+"},
             {.uci = "g2g1", .san = "Kg1"},   {.uci = "f6g5", .san = "Bg5"},
             {.uci = "d3e2", .san = "Qe2+"},  {.uci = "g4h4", .san = "Kh4"},
             {.uci = "e3f2", .san = "Bf2+"},  {.uci = "h4h3", .san = "Kh3"},
             {.uci = "f2e1", .san = "Be1"},
         }},
    {.desc = "L. Polugaevsky / R. G. Nezhmetdinov / 1958 / Sochi",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 66,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},     {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},     {.uci = "d7d6", .san = "d6"},
             {.uci = "b1c3", .san = "Nc3"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "e2e4", .san = "e4"},     {.uci = "e5d4", .san = "exd4"},
             {.uci = "d1d4", .san = "Qxd4"},   {.uci = "b8c6", .san = "Nc6"},
             {.uci = "d4d2", .san = "Qd2"},    {.uci = "g7g6", .san = "g6"},
             {.uci = "b2b3", .san = "b3"},     {.uci = "f8g7", .san = "Bg7"},
             {.uci = "c1b2", .san = "Bb2"},    {.uci = "e8g8", .san = "O-O"},
             {.uci = "f1d3", .san = "Bd3"},    {.uci = "f6g4", .san = "Ng4"},
             {.uci = "g1e2", .san = "Nge2"},   {.uci = "d8h4", .san = "Qh4"},
             {.uci = "e2g3", .san = "Ng3"},    {.uci = "g4e5", .san = "Nge5"},
             {.uci = "e1g1", .san = "O-O"},    {.uci = "f7f5", .san = "f5"},
             {.uci = "f2f3", .san = "f3"},     {.uci = "g7h6", .san = "Bh6"},
             {.uci = "d2d1", .san = "Qd1"},    {.uci = "f5f4", .san = "f4"},
             {.uci = "g3e2", .san = "Nge2"},   {.uci = "g6g5", .san = "g5"},
             {.uci = "c3d5", .san = "Nd5"},    {.uci = "g5g4", .san = "g4"},
             {.uci = "g2g3", .san = "g3"},     {.uci = "f4g3", .san = "fxg3"},
             {.uci = "h2g3", .san = "hxg3"},   {.uci = "h4h3", .san = "Qh3"},
             {.uci = "f3f4", .san = "f4"},     {.uci = "c8e6", .san = "Be6"},
             {.uci = "d3c2", .san = "Bc2"},    {.uci = "f8f7", .san = "Rf7"},
             {.uci = "g1f2", .san = "Kf2"},    {.uci = "h3h2", .san = "Qh2+"},
             {.uci = "f2e3", .san = "Ke3"},    {.uci = "e6d5", .san = "Bxd5"},
             {.uci = "c4d5", .san = "cxd5"},   {.uci = "c6b4", .san = "Nb4"},
             {.uci = "f1h1", .san = "Rh1"},    {.uci = "f7f4", .san = "Rxf4"},
             {.uci = "h1h2", .san = "Rxh2"},   {.uci = "f4f3", .san = "Rf3+"},
             {.uci = "e3d4", .san = "Kd4"},    {.uci = "h6g7", .san = "Bg7"},
             {.uci = "a2a4", .san = "a4"},     {.uci = "c7c5", .san = "c5+"},
             {.uci = "d5c6", .san = "dxc6"}, /* e.p. */
             {.uci = "b7c6", .san = "bxc6"},   {.uci = "c2d3", .san = "Bd3"},
             {.uci = "e5d3", .san = "Nexd3+"}, {.uci = "d4c4", .san = "Kc4"},
             {.uci = "d6d5", .san = "d5+"},    {.uci = "e4d5", .san = "exd5"},
             {.uci = "c6d5", .san = "cxd5+"},  {.uci = "c4b5", .san = "Kb5"},
             {.uci = "a8b8", .san = "Rb8+"},   {.uci = "b5a5", .san = "Ka5"},
             {.uci = "b4c6", .san = "Nc6+"},
         }},
    {.desc = "M. Botvinnik / J. R. Capablanca / 1938.11.22 / Rotterdam",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 81,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "f8b4", .san = "Bb4"},
             {.uci = "e2e3", .san = "e3"},   {.uci = "d7d5", .san = "d5"},
             {.uci = "a2a3", .san = "a3"},   {.uci = "b4c3", .san = "Bxc3+"},
             {.uci = "b2c3", .san = "bxc3"}, {.uci = "c7c5", .san = "c5"},
             {.uci = "c4d5", .san = "cxd5"}, {.uci = "e6d5", .san = "exd5"},
             {.uci = "f1d3", .san = "Bd3"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "g1e2", .san = "Ne2"},  {.uci = "b7b6", .san = "b6"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "c8a6", .san = "Ba6"},
             {.uci = "d3a6", .san = "Bxa6"}, {.uci = "b8a6", .san = "Nxa6"},
             {.uci = "c1b2", .san = "Bb2"},  {.uci = "d8d7", .san = "Qd7"},
             {.uci = "a3a4", .san = "a4"},   {.uci = "f8e8", .san = "Rfe8"},
             {.uci = "d1d3", .san = "Qd3"},  {.uci = "c5c4", .san = "c4"},
             {.uci = "d3c2", .san = "Qc2"},  {.uci = "a6b8", .san = "Nb8"},
             {.uci = "a1e1", .san = "Rae1"}, {.uci = "b8c6", .san = "Nc6"},
             {.uci = "e2g3", .san = "Ng3"},  {.uci = "c6a5", .san = "Na5"},
             {.uci = "f2f3", .san = "f3"},   {.uci = "a5b3", .san = "Nb3"},
             {.uci = "e3e4", .san = "e4"},   {.uci = "d7a4", .san = "Qxa4"},
             {.uci = "e4e5", .san = "e5"},   {.uci = "f6d7", .san = "Nd7"},
             {.uci = "c2f2", .san = "Qf2"},  {.uci = "g7g6", .san = "g6"},
             {.uci = "f3f4", .san = "f4"},   {.uci = "f7f5", .san = "f5"},
             {.uci = "e5f6", .san = "exf6"}, /* e.p. */
             {.uci = "d7f6", .san = "Nxf6"}, {.uci = "f4f5", .san = "f5"},
             {.uci = "e8e1", .san = "Rxe1"}, {.uci = "f1e1", .san = "Rxe1"},
             {.uci = "a8e8", .san = "Re8"},  {.uci = "e1e6", .san = "Re6"},
             {.uci = "e8e6", .san = "Rxe6"}, {.uci = "f5e6", .san = "fxe6"},
             {.uci = "g8g7", .san = "Kg7"},  {.uci = "f2f4", .san = "Qf4"},
             {.uci = "a4e8", .san = "Qe8"},  {.uci = "f4e5", .san = "Qe5"},
             {.uci = "e8e7", .san = "Qe7"},  {.uci = "b2a3", .san = "Ba3"},
             {.uci = "e7a3", .san = "Qxa3"}, {.uci = "g3h5", .san = "Nh5+"},
             {.uci = "g6h5", .san = "gxh5"}, {.uci = "e5g5", .san = "Qg5+"},
             {.uci = "g7f8", .san = "Kf8"},  {.uci = "g5f6", .san = "Qxf6+"},
             {.uci = "f8g8", .san = "Kg8"},  {.uci = "e6e7", .san = "e7"},
             {.uci = "a3c1", .san = "Qc1+"}, {.uci = "g1f2", .san = "Kf2"},
             {.uci = "c1c2", .san = "Qc2+"}, {.uci = "f2g3", .san = "Kg3"},
             {.uci = "c2d3", .san = "Qd3+"}, {.uci = "g3h4", .san = "Kh4"},
             {.uci = "d3e4", .san = "Qe4+"}, {.uci = "h4h5", .san = "Kxh5"},
             {.uci = "e4e2", .san = "Qe2+"}, {.uci = "h5h4", .san = "Kh4"},
             {.uci = "e2e4", .san = "Qe4+"}, {.uci = "g2g4", .san = "g4"},
             {.uci = "e4e1", .san = "Qe1+"}, {.uci = "h4h5", .san = "Kh5"},
         }},
    {.desc = "V. Bagirov / E. Gufeld / 1973\n\"Mona Lisa\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 64,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},    {.uci = "g7g6", .san = "g6"},
             {.uci = "c2c4", .san = "c4"},    {.uci = "f8g7", .san = "Bg7"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "d7d6", .san = "d6"},
             {.uci = "e2e4", .san = "e4"},    {.uci = "g8f6", .san = "Nf6"},
             {.uci = "f2f3", .san = "f3"},    {.uci = "e8g8", .san = "O-O"},
             {.uci = "c1e3", .san = "Be3"},   {.uci = "b8c6", .san = "Nc6"},
             {.uci = "g1e2", .san = "Nge2"},  {.uci = "a8b8", .san = "Rb8"},
             {.uci = "d1d2", .san = "Qd2"},   {.uci = "a7a6", .san = "a6"},
             {.uci = "e3h6", .san = "Bh6"},   {.uci = "b7b5", .san = "b5"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "h6g7", .san = "Bxg7"},  {.uci = "g8g7", .san = "Kxg7"},
             {.uci = "h4h5", .san = "h5"},    {.uci = "g7h8", .san = "Kh8"},
             {.uci = "c3d5", .san = "Nd5"},   {.uci = "b5c4", .san = "bxc4"},
             {.uci = "h5g6", .san = "hxg6"},  {.uci = "f7g6", .san = "fxg6"},
             {.uci = "d2h6", .san = "Qh6"},   {.uci = "f6h5", .san = "Nh5"},
             {.uci = "g2g4", .san = "g4"},    {.uci = "b8b2", .san = "Rxb2"},
             {.uci = "g4h5", .san = "gxh5"},  {.uci = "g6g5", .san = "g5"},
             {.uci = "h1g1", .san = "Rg1"},   {.uci = "g5g4", .san = "g4"},
             {.uci = "e1c1", .san = "O-O-O"}, {.uci = "b2a2", .san = "Rxa2"},
             {.uci = "e2f4", .san = "Nef4"},  {.uci = "e5f4", .san = "exf4"},
             {.uci = "d5f4", .san = "Nxf4"},  {.uci = "f8f4", .san = "Rxf4"},
             {.uci = "h6f4", .san = "Qxf4"},  {.uci = "c4c3", .san = "c3"},
             {.uci = "f1c4", .san = "Bc4"},   {.uci = "a2a3", .san = "Ra3"},
             {.uci = "f3g4", .san = "fxg4"},  {.uci = "c6b4", .san = "Nb4"},
             {.uci = "c1b1", .san = "Kb1"},   {.uci = "c8e6", .san = "Be6"},
             {.uci = "c4e6", .san = "Bxe6"},  {.uci = "b4d3", .san = "Nd3"},
             {.uci = "f4f7", .san = "Qf7"},   {.uci = "d8b8", .san = "Qb8+"},
             {.uci = "e6b3", .san = "Bb3"},   {.uci = "a3b3", .san = "Rxb3+"},
             {.uci = "b1c2", .san = "Kc2"},   {.uci = "d3b4", .san = "Nb4+"},
             {.uci = "c2b3", .san = "Kxb3"},  {.uci = "b4d5", .san = "Nd5+"},
             {.uci = "b3c2", .san = "Kc2"},   {.uci = "b8b2", .san = "Qb2+"},
             {.uci = "c2d3", .san = "Kd3"},   {.uci = "b2b5", .san = "Qb5+"},
         }},
    {.desc = "A. Beliavsky / J. Nunn / 1985",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 54,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "g7g6", .san = "g6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "f8g7", .san = "Bg7"},
             {.uci = "e2e4", .san = "e4"},   {.uci = "d7d6", .san = "d6"},
             {.uci = "f2f3", .san = "f3"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "c1e3", .san = "Be3"},  {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "d1d2", .san = "Qd2"},  {.uci = "c7c5", .san = "c5"},
             {.uci = "d4d5", .san = "d5"},   {.uci = "d7e5", .san = "Ne5"},
             {.uci = "h2h3", .san = "h3"},   {.uci = "f6h5", .san = "Nh5"},
             {.uci = "e3f2", .san = "Bf2"},  {.uci = "f7f5", .san = "f5"},
             {.uci = "e4f5", .san = "exf5"}, {.uci = "f8f5", .san = "Rxf5"},
             {.uci = "g2g4", .san = "g4"},   {.uci = "f5f3", .san = "Rxf3"},
             {.uci = "g4h5", .san = "gxh5"}, {.uci = "d8f8", .san = "Qf8"},
             {.uci = "c3e4", .san = "Ne4"},  {.uci = "g7h6", .san = "Bh6"},
             {.uci = "d2c2", .san = "Qc2"},  {.uci = "f8f4", .san = "Qf4"},
             {.uci = "g1e2", .san = "Ne2"},  {.uci = "f3f2", .san = "Rxf2"},
             {.uci = "e4f2", .san = "Nxf2"}, {.uci = "e5f3", .san = "Nf3+"},
             {.uci = "e1d1", .san = "Kd1"},  {.uci = "f4h4", .san = "Qh4"},
             {.uci = "f2d3", .san = "Nd3"},  {.uci = "c8f5", .san = "Bf5"},
             {.uci = "e2c1", .san = "Nec1"}, {.uci = "f3d2", .san = "Nd2"},
             {.uci = "h5g6", .san = "hxg6"}, {.uci = "h7g6", .san = "hxg6"},
             {.uci = "f1g2", .san = "Bg2"},  {.uci = "d2c4", .san = "Nxc4"},
             {.uci = "c2f2", .san = "Qf2"},  {.uci = "c4e3", .san = "Ne3+"},
             {.uci = "d1e2", .san = "Ke2"},  {.uci = "h4c4", .san = "Qc4"},
             {.uci = "g2f3", .san = "Bf3"},  {.uci = "a8f8", .san = "Rf8"},
             {.uci = "h1g1", .san = "Rg1"},  {.uci = "e3c2", .san = "Nc2"},
             {.uci = "e2d1", .san = "Kd1"},  {.uci = "f5d3", .san = "Bxd3"},
         }},
    {.desc = "A. Shirov / J. Polgar / 1994 / Buenos Aires",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 58,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},   {.uci = "c7c5", .san = "c5"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "d2d4", .san = "d4"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "f3d4", .san = "Nxd4"}, {.uci = "b8c6", .san = "Nc6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "d7d6", .san = "d6"},
             {.uci = "g2g4", .san = "g4"},   {.uci = "a7a6", .san = "a6"},
             {.uci = "c1e3", .san = "Be3"},  {.uci = "g8e7", .san = "Nge7"},
             {.uci = "d4b3", .san = "Nb3"},  {.uci = "b7b5", .san = "b5"},
             {.uci = "f2f4", .san = "f4"},   {.uci = "c8b7", .san = "Bb7"},
             {.uci = "d1f3", .san = "Qf3"},  {.uci = "g7g5", .san = "g5"},
             {.uci = "f4g5", .san = "fxg5"}, {.uci = "c6e5", .san = "Ne5"},
             {.uci = "f3g2", .san = "Qg2"},  {.uci = "b5b4", .san = "b4"},
             {.uci = "c3e2", .san = "Ne2"},  {.uci = "h7h5", .san = "h5"},
             {.uci = "g4h5", .san = "gxh5"}, {.uci = "e7f5", .san = "Nf5"},
             {.uci = "e3f2", .san = "Bf2"},  {.uci = "d8g5", .san = "Qxg5"},
             {.uci = "b3a5", .san = "Na5"},  {.uci = "f5e3", .san = "Ne3"},
             {.uci = "g2g3", .san = "Qg3"},  {.uci = "g5g3", .san = "Qxg3"},
             {.uci = "e2g3", .san = "Nxg3"}, {.uci = "e3c2", .san = "Nxc2+"},
             {.uci = "e1d1", .san = "Kd1"},  {.uci = "c2a1", .san = "Nxa1"},
             {.uci = "a5b7", .san = "Nxb7"}, {.uci = "b4b3", .san = "b3"},
             {.uci = "a2b3", .san = "axb3"}, {.uci = "a1b3", .san = "Nxb3"},
             {.uci = "d1c2", .san = "Kc2"},  {.uci = "b3c5", .san = "Nc5"},
             {.uci = "b7c5", .san = "Nxc5"}, {.uci = "d6c5", .san = "dxc5"},
             {.uci = "f2e1", .san = "Be1"},  {.uci = "e5f3", .san = "Nf3"},
             {.uci = "e1c3", .san = "Bc3"},  {.uci = "f3d4", .san = "Nd4+"},
             {.uci = "c2d3", .san = "Kd3"},  {.uci = "f8d6", .san = "Bd6"},
             {.uci = "f1g2", .san = "Bg2"},  {.uci = "d6e5", .san = "Be5"},
             {.uci = "d3c4", .san = "Kc4"},  {.uci = "e8e7", .san = "Ke7"},
             {.uci = "h1a1", .san = "Ra1"},  {.uci = "d4c6", .san = "Nc6"},
         }},
    {.desc = "V. Ivanchuk / A. Yusupov / 1991\n\"Ivanchuck's immortal\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 78,
     .moves =
         {
             {.uci = "c2c4", .san = "c4"},    {.uci = "e7e5", .san = "e5"},
             {.uci = "g2g3", .san = "g3"},    {.uci = "d7d6", .san = "d6"},
             {.uci = "f1g2", .san = "Bg2"},   {.uci = "g7g6", .san = "g6"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "b8d7", .san = "Nd7"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "f8g7", .san = "Bg7"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "g8f6", .san = "Ngf6"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "d1c2", .san = "Qc2"},   {.uci = "f8e8", .san = "Re8"},
             {.uci = "f1d1", .san = "Rd1"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "b2b3", .san = "b3"},    {.uci = "d8e7", .san = "Qe7"},
             {.uci = "c1a3", .san = "Ba3"},   {.uci = "e5e4", .san = "e4"},
             {.uci = "f3g5", .san = "Ng5"},   {.uci = "e4e3", .san = "e3"},
             {.uci = "f2f4", .san = "f4"},    {.uci = "d7f8", .san = "Nf8"},
             {.uci = "b3b4", .san = "b4"},    {.uci = "c8f5", .san = "Bf5"},
             {.uci = "c2b3", .san = "Qb3"},   {.uci = "h7h6", .san = "h6"},
             {.uci = "g5f3", .san = "Nf3"},   {.uci = "f6g4", .san = "Ng4"},
             {.uci = "b4b5", .san = "b5"},    {.uci = "g6g5", .san = "g5"},
             {.uci = "b5c6", .san = "bxc6"},  {.uci = "b7c6", .san = "bxc6"},
             {.uci = "f3e5", .san = "Ne5"},   {.uci = "g5f4", .san = "gxf4"},
             {.uci = "e5c6", .san = "Nxc6"},  {.uci = "e7g5", .san = "Qg5"},
             {.uci = "a3d6", .san = "Bxd6"},  {.uci = "f8g6", .san = "Ng6"},
             {.uci = "c3d5", .san = "Nd5"},   {.uci = "g5h5", .san = "Qh5"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "g6h4", .san = "Nxh4"},
             {.uci = "g3h4", .san = "gxh4"},  {.uci = "h5h4", .san = "Qxh4"},
             {.uci = "d5e7", .san = "Nde7+"}, {.uci = "g8h8", .san = "Kh8"},
             {.uci = "e7f5", .san = "Nxf5"},  {.uci = "h4h2", .san = "Qh2+"},
             {.uci = "g1f1", .san = "Kf1"},   {.uci = "e8e6", .san = "Re6"},
             {.uci = "b3b7", .san = "Qb7"},   {.uci = "e6g6", .san = "Rg6"},
             {.uci = "b7a8", .san = "Qxa8+"}, {.uci = "h8h7", .san = "Kh7"},
             {.uci = "a8g8", .san = "Qg8+"},  {.uci = "h7g8", .san = "Kxg8"},
             {.uci = "c6e7", .san = "Nce7+"}, {.uci = "g8h7", .san = "Kh7"},
             {.uci = "e7g6", .san = "Nxg6"},  {.uci = "f7g6", .san = "fxg6"},
             {.uci = "f5g7", .san = "Nxg7"},  {.uci = "g4f2", .san = "Nf2"},
             {.uci = "d6f4", .san = "Bxf4"},  {.uci = "h2f4", .san = "Qxf4"},
             {.uci = "g7e6", .san = "Ne6"},   {.uci = "f4h2", .san = "Qh2"},
             {.uci = "d1b1", .san = "Rdb1"},  {.uci = "f2h3", .san = "Nh3"},
             {.uci = "b1b7", .san = "Rb7+"},  {.uci = "h7h8", .san = "Kh8"},
             {.uci = "b7b8", .san = "Rb8+"},  {.uci = "h2b8", .san = "Qxb8"},
             {.uci = "g2h3", .san = "Bxh3"},  {.uci = "b8g3", .san = "Qg3"},
         }},
    {.desc = "M. Krasenkow / H. Nakamura / 2007 / Barcelona",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 56,
     .moves =
         {
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "g2g3", .san = "g3"},   {.uci = "d7d5", .san = "d5"},
             {.uci = "f1g2", .san = "Bg2"},  {.uci = "f8e7", .san = "Be7"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "b2b3", .san = "b3"},   {.uci = "a7a5", .san = "a5"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "c7c6", .san = "c6"},
             {.uci = "d2d4", .san = "d4"},   {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "d1c2", .san = "Qc2"},  {.uci = "b7b6", .san = "b6"},
             {.uci = "e2e4", .san = "e4"},   {.uci = "c8a6", .san = "Ba6"},
             {.uci = "f3d2", .san = "Nd2"},  {.uci = "c6c5", .san = "c5"},
             {.uci = "e4d5", .san = "exd5"}, {.uci = "c5d4", .san = "cxd4"},
             {.uci = "c3b5", .san = "Nb5"},  {.uci = "e6d5", .san = "exd5"},
             {.uci = "b5d4", .san = "Nxd4"}, {.uci = "a8c8", .san = "Rc8"},
             {.uci = "f1e1", .san = "Re1"},  {.uci = "b6b5", .san = "b5"},
             {.uci = "c1b2", .san = "Bb2"},  {.uci = "f8e8", .san = "Re8"},
             {.uci = "c2d1", .san = "Qd1"},  {.uci = "b5c4", .san = "bxc4"},
             {.uci = "b3c4", .san = "bxc4"}, {.uci = "d8b6", .san = "Qb6"},
             {.uci = "a1b1", .san = "Rb1"},  {.uci = "d5c4", .san = "dxc4"},
             {.uci = "d4c6", .san = "Nc6"},  {.uci = "c8c6", .san = "Rxc6"},
             {.uci = "b2f6", .san = "Bxf6"}, {.uci = "b6f2", .san = "Qxf2+"},
             {.uci = "g1f2", .san = "Kxf2"}, {.uci = "e7c5", .san = "Bc5+"},
             {.uci = "f2f3", .san = "Kf3"},  {.uci = "c6f6", .san = "Rxf6+"},
             {.uci = "f3g4", .san = "Kg4"},  {.uci = "d7e5", .san = "Ne5+"},
             {.uci = "g4g5", .san = "Kg5"},  {.uci = "f6g6", .san = "Rg6+"},
             {.uci = "g5h5", .san = "Kh5"},  {.uci = "f7f6", .san = "f6"},
             {.uci = "e1e5", .san = "Rxe5"}, {.uci = "e8e5", .san = "Rxe5+"},
             {.uci = "h5h4", .san = "Kh4"},  {.uci = "a6c8", .san = "Bc8"},
         }},
    {.desc = "A. Karpov / V. Topalov / 1994\n\"Karpov's immortal\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 77,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},    {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},    {.uci = "c7c5", .san = "c5"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "f3d4", .san = "Nxd4"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "g2g3", .san = "g3"},    {.uci = "b8c6", .san = "Nc6"},
             {.uci = "f1g2", .san = "Bg2"},   {.uci = "f8c5", .san = "Bc5"},
             {.uci = "d4b3", .san = "Nb3"},   {.uci = "c5e7", .san = "Be7"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "d7d6", .san = "d6"},
             {.uci = "c1f4", .san = "Bf4"},   {.uci = "f6h5", .san = "Nh5"},
             {.uci = "e2e3", .san = "e3"},    {.uci = "h5f4", .san = "Nxf4"},
             {.uci = "e3f4", .san = "exf4"},  {.uci = "c8d7", .san = "Bd7"},
             {.uci = "d1d2", .san = "Qd2"},   {.uci = "d8b8", .san = "Qb8"},
             {.uci = "f1e1", .san = "Rfe1"},  {.uci = "g7g6", .san = "g6"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "a7a6", .san = "a6"},
             {.uci = "h4h5", .san = "h5"},    {.uci = "b7b5", .san = "b5"},
             {.uci = "h5g6", .san = "hxg6"},  {.uci = "h7g6", .san = "hxg6"},
             {.uci = "b3c5", .san = "Nc5"},   {.uci = "d6c5", .san = "dxc5"},
             {.uci = "d2d7", .san = "Qxd7"},  {.uci = "f8c8", .san = "Rc8"},
             {.uci = "e1e6", .san = "Rxe6"},  {.uci = "a8a7", .san = "Ra7"},
             {.uci = "e6g6", .san = "Rxg6+"}, {.uci = "f7g6", .san = "fxg6"},
             {.uci = "d7e6", .san = "Qe6+"},  {.uci = "g8g7", .san = "Kg7"},
             {.uci = "g2c6", .san = "Bxc6"},  {.uci = "c8d8", .san = "Rd8"},
             {.uci = "c4b5", .san = "cxb5"},  {.uci = "e7f6", .san = "Bf6"},
             {.uci = "c3e4", .san = "Ne4"},   {.uci = "f6d4", .san = "Bd4"},
             {.uci = "b5a6", .san = "bxa6"},  {.uci = "b8b6", .san = "Qb6"},
             {.uci = "a1d1", .san = "Rd1"},   {.uci = "b6a6", .san = "Qxa6"},
             {.uci = "d1d4", .san = "Rxd4"},  {.uci = "d8d4", .san = "Rxd4"},
             {.uci = "e6f6", .san = "Qf6+"},  {.uci = "g7g8", .san = "Kg8"},
             {.uci = "f6g6", .san = "Qxg6+"}, {.uci = "g8f8", .san = "Kf8"},
             {.uci = "g6e8", .san = "Qe8+"},  {.uci = "f8g7", .san = "Kg7"},
             {.uci = "e8e5", .san = "Qe5+"},  {.uci = "g7g8", .san = "Kg8"},
             {.uci = "e4f6", .san = "Nf6+"},  {.uci = "g8f7", .san = "Kf7"},
             {.uci = "c6e8", .san = "Be8+"},  {.uci = "f7f8", .san = "Kf8"},
             {.uci = "e5c5", .san = "Qxc5+"}, {.uci = "a6d6", .san = "Qd6"},
             {.uci = "c5a7", .san = "Qxa7"},  {.uci = "d6f6", .san = "Qxf6"},
             {.uci = "e8h5", .san = "Bh5"},   {.uci = "d4d2", .san = "Rd2"},
             {.uci = "b2b3", .san = "b3"},    {.uci = "d2b2", .san = "Rb2"},
             {.uci = "g1g2", .san = "Kg2"},
         }},
    {.desc = "M. Carlsen / S. Ernst / 2004\n\"The Magnus effect\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 57,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},    {.uci = "c7c6", .san = "c6"},
             {.uci = "d2d4", .san = "d4"},    {.uci = "d7d5", .san = "d5"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "d5e4", .san = "dxe4"},
             {.uci = "c3e4", .san = "Nxe4"},  {.uci = "c8f5", .san = "Bf5"},
             {.uci = "e4g3", .san = "Ng3"},   {.uci = "f5g6", .san = "Bg6"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "h7h6", .san = "h6"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "b8d7", .san = "Nd7"},
             {.uci = "h4h5", .san = "h5"},    {.uci = "g6h7", .san = "Bh7"},
             {.uci = "f1d3", .san = "Bd3"},   {.uci = "h7d3", .san = "Bxd3"},
             {.uci = "d1d3", .san = "Qxd3"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "c1f4", .san = "Bf4"},   {.uci = "g8f6", .san = "Ngf6"},
             {.uci = "e1c1", .san = "O-O-O"}, {.uci = "f8e7", .san = "Be7"},
             {.uci = "g3e4", .san = "Ne4"},   {.uci = "d8a5", .san = "Qa5"},
             {.uci = "c1b1", .san = "Kb1"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "e4f6", .san = "Nxf6+"}, {.uci = "d7f6", .san = "Nxf6"},
             {.uci = "f3e5", .san = "Ne5"},   {.uci = "a8d8", .san = "Rad8"},
             {.uci = "d3e2", .san = "Qe2"},   {.uci = "c6c5", .san = "c5"},
             {.uci = "e5g6", .san = "Ng6"},   {.uci = "f7g6", .san = "fxg6"},
             {.uci = "e2e6", .san = "Qxe6+"}, {.uci = "g8h8", .san = "Kh8"},
             {.uci = "h5g6", .san = "hxg6"},  {.uci = "f6g8", .san = "Ng8"},
             {.uci = "f4h6", .san = "Bxh6"},  {.uci = "g7h6", .san = "gxh6"},
             {.uci = "h1h6", .san = "Rxh6+"}, {.uci = "g8h6", .san = "Nxh6"},
             {.uci = "e6e7", .san = "Qxe7"},  {.uci = "h6f7", .san = "Nf7"},
             {.uci = "g6f7", .san = "gxf7"},  {.uci = "h8g7", .san = "Kg7"},
             {.uci = "d1d3", .san = "Rd3"},   {.uci = "d8d6", .san = "Rd6"},
             {.uci = "d3g3", .san = "Rg3+"},  {.uci = "d6g6", .san = "Rg6"},
             {.uci = "e7e5", .san = "Qe5+"},  {.uci = "g7f7", .san = "Kxf7"},
             {.uci = "e5f5", .san = "Qf5+"},  {.uci = "g6f6", .san = "Rf6"},
             {.uci = "f5d7", .san = "Qd7#"},
         }},
    {.desc = "N. Short / J. Timman / 1991\n\"The king's walk\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 67,
     .moves =
         {
             {.uci = "e2e4", .san = "e4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "e4e5", .san = "e5"},   {.uci = "f6d5", .san = "Nd5"},
             {.uci = "d2d4", .san = "d4"},   {.uci = "d7d6", .san = "d6"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "g7g6", .san = "g6"},
             {.uci = "f1c4", .san = "Bc4"},  {.uci = "d5b6", .san = "Nb6"},
             {.uci = "c4b3", .san = "Bb3"},  {.uci = "f8g7", .san = "Bg7"},
             {.uci = "d1e2", .san = "Qe2"},  {.uci = "b8c6", .san = "Nc6"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "h2h3", .san = "h3"},   {.uci = "a7a5", .san = "a5"},
             {.uci = "a2a4", .san = "a4"},   {.uci = "d6e5", .san = "dxe5"},
             {.uci = "d4e5", .san = "dxe5"}, {.uci = "c6d4", .san = "Nd4"},
             {.uci = "f3d4", .san = "Nxd4"}, {.uci = "d8d4", .san = "Qxd4"},
             {.uci = "f1e1", .san = "Re1"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "b1d2", .san = "Nd2"},  {.uci = "b6d5", .san = "Nd5"},
             {.uci = "d2f3", .san = "Nf3"},  {.uci = "d4c5", .san = "Qc5"},
             {.uci = "e2e4", .san = "Qe4"},  {.uci = "c5b4", .san = "Qb4"},
             {.uci = "b3c4", .san = "Bc4"},  {.uci = "d5b6", .san = "Nb6"},
             {.uci = "b2b3", .san = "b3"},   {.uci = "b6c4", .san = "Nxc4"},
             {.uci = "b3c4", .san = "bxc4"}, {.uci = "f8e8", .san = "Re8"},
             {.uci = "e1d1", .san = "Rd1"},  {.uci = "b4c5", .san = "Qc5"},
             {.uci = "e4h4", .san = "Qh4"},  {.uci = "b7b6", .san = "b6"},
             {.uci = "c1e3", .san = "Be3"},  {.uci = "c5c6", .san = "Qc6"},
             {.uci = "e3h6", .san = "Bh6"},  {.uci = "g7h8", .san = "Bh8"},
             {.uci = "d1d8", .san = "Rd8"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "a1d1", .san = "Rad1"}, {.uci = "h8g7", .san = "Bg7"},
             {.uci = "d8d7", .san = "R8d7"}, {.uci = "e8f8", .san = "Rf8"},
             {.uci = "h6g7", .san = "Bxg7"}, {.uci = "g8g7", .san = "Kxg7"},
             {.uci = "d1d4", .san = "R1d4"}, {.uci = "a8e8", .san = "Rae8"},
             {.uci = "h4f6", .san = "Qf6+"}, {.uci = "g7g8", .san = "Kg8"},
             {.uci = "h3h4", .san = "h4"},   {.uci = "h7h5", .san = "h5"},
             {.uci = "g1h2", .san = "Kh2"},  {.uci = "e8c8", .san = "Rc8"},
             {.uci = "h2g3", .san = "Kg3"},  {.uci = "c8e8", .san = "Rce8"},
             {.uci = "g3f4", .san = "Kf4"},  {.uci = "b7c8", .san = "Bc8"},
             {.uci = "f4g5", .san = "Kg5"},
         }},
    {.desc = "L. Aronian / V. Anand / 2013\n\"Nerves of steel\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 46,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "d7d5", .san = "d5"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "e7e6", .san = "e6"},
             {.uci = "e2e3", .san = "e3"},   {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "f1d3", .san = "Bd3"},  {.uci = "d5c4", .san = "dxc4"},
             {.uci = "d3c4", .san = "Bxc4"}, {.uci = "b7b5", .san = "b5"},
             {.uci = "c4d3", .san = "Bd3"},  {.uci = "f8d6", .san = "Bd6"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "d1c2", .san = "Qc2"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "a2a3", .san = "a3"},   {.uci = "a8c8", .san = "Rc8"},
             {.uci = "f3g5", .san = "Ng5"},  {.uci = "c6c5", .san = "c5"},
             {.uci = "g5h7", .san = "Nxh7"}, {.uci = "f6g4", .san = "Ng4"},
             {.uci = "f2f4", .san = "f4"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "e3d4", .san = "exd4"}, {.uci = "d6c5", .san = "Bc5"},
             {.uci = "d3e2", .san = "Be2"},  {.uci = "d7e5", .san = "Nde5"},
             {.uci = "e2g4", .san = "Bxg4"}, {.uci = "c5d4", .san = "Bxd4+"},
             {.uci = "g1h1", .san = "Kh1"},  {.uci = "e5g4", .san = "Nxg4"},
             {.uci = "h7f8", .san = "Nxf8"}, {.uci = "f7f5", .san = "f5"},
             {.uci = "f8g6", .san = "Ng6"},  {.uci = "d8f6", .san = "Qf6"},
             {.uci = "h2h3", .san = "h3"},   {.uci = "f6g6", .san = "Qxg6"},
             {.uci = "c2e2", .san = "Qe2"},  {.uci = "g6h5", .san = "Qh5"},
             {.uci = "e2d3", .san = "Qd3"},  {.uci = "d4e3", .san = "Be3"},
         }},
    {.desc = "Jinshi Bai / Ding Liren / 2017.11.04\nChinese Chess League",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 64,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "f8b4", .san = "Bb4"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "c1g5", .san = "Bg5"},  {.uci = "c7c5", .san = "c5"},
             {.uci = "e2e3", .san = "e3"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "d1d4", .san = "Qxd4"}, {.uci = "b8c6", .san = "Nc6"},
             {.uci = "d4d3", .san = "Qd3"},  {.uci = "h7h6", .san = "h6"},
             {.uci = "g5h4", .san = "Bh4"},  {.uci = "d7d5", .san = "d5"},
             {.uci = "a1d1", .san = "Rd1"},  {.uci = "g7g5", .san = "g5"},
             {.uci = "h4g3", .san = "Bg3"},  {.uci = "f6e4", .san = "Ne4"},
             {.uci = "f3d2", .san = "Nd2"},  {.uci = "e4c5", .san = "Nc5"},
             {.uci = "d3c2", .san = "Qc2"},  {.uci = "d5d4", .san = "d4"},
             {.uci = "d2f3", .san = "Nf3"},  {.uci = "e6e5", .san = "e5"},
             {.uci = "f3e5", .san = "Nxe5"}, {.uci = "d4c3", .san = "dxc3"},
             {.uci = "d1d8", .san = "Rxd8"}, {.uci = "c3b2", .san = "cxb2+"},
             {.uci = "e1e2", .san = "Ke2"},  {.uci = "f8d8", .san = "Rxd8"},
             {.uci = "c2b2", .san = "Qxb2"}, {.uci = "c5a4", .san = "Na4"},
             {.uci = "b2c2", .san = "Qc2"},  {.uci = "a4c3", .san = "Nc3+"},
             {.uci = "e2f3", .san = "Kf3"},  {.uci = "d8d4", .san = "Rd4"},
             {.uci = "h2h3", .san = "h3"},   {.uci = "h6h5", .san = "h5"},
             {.uci = "g3h2", .san = "Bh2"},  {.uci = "g5g4", .san = "g4+"},
             {.uci = "f3g3", .san = "Kg3"},  {.uci = "d4d2", .san = "Rd2"},
             {.uci = "c2b3", .san = "Qb3"},  {.uci = "c3e4", .san = "Ne4+"},
             {.uci = "g3h4", .san = "Kh4"},  {.uci = "b4e7", .san = "Be7+"},
             {.uci = "h4h5", .san = "Kxh5"}, {.uci = "g8g7", .san = "Kg7"},
             {.uci = "h2f4", .san = "Bf4"},  {.uci = "c8f5", .san = "Bf5"},
             {.uci = "f4h6", .san = "Bh6+"}, {.uci = "g7h7", .san = "Kh7"},
             {.uci = "b3b7", .san = "Qxb7"}, {.uci = "d2f2", .san = "Rxf2"},
             {.uci = "h6g5", .san = "Bg5"},  {.uci = "a8h8", .san = "Rh8"},
             {.uci = "e5f7", .san = "Nxf7"}, {.uci = "f5g6", .san = "Bg6+"},
             {.uci = "h5g4", .san = "Kxg4"}, {.uci = "c6e5", .san = "Ne5+"},
         }},
    {.desc = "E. Geller / M. Euwe / 1953.08.31",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 52,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},    {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c2c4", .san = "c4"},    {.uci = "e7e6", .san = "e6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "f8b4", .san = "Bb4"},
             {.uci = "e2e3", .san = "e3"},    {.uci = "c7c5", .san = "c5"},
             {.uci = "a2a3", .san = "a3"},    {.uci = "b4c3", .san = "Bxc3+"},
             {.uci = "b2c3", .san = "bxc3"},  {.uci = "b7b6", .san = "b6"},
             {.uci = "f1d3", .san = "Bd3"},   {.uci = "c8b7", .san = "Bb7"},
             {.uci = "f2f3", .san = "f3"},    {.uci = "b8c6", .san = "Nc6"},
             {.uci = "g1e2", .san = "Ne2"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "c6a5", .san = "Na5"},
             {.uci = "e3e4", .san = "e4"},    {.uci = "f6e8", .san = "Ne8"},
             {.uci = "e2g3", .san = "Ng3"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "c3d4", .san = "cxd4"},  {.uci = "a8c8", .san = "Rc8"},
             {.uci = "f3f4", .san = "f4"},    {.uci = "a5c4", .san = "Nxc4"},
             {.uci = "f4f5", .san = "f5"},    {.uci = "f7f6", .san = "f6"},
             {.uci = "f1f4", .san = "Rf4"},   {.uci = "b6b5", .san = "b5"},
             {.uci = "f4h4", .san = "Rh4"},   {.uci = "d8b6", .san = "Qb6"},
             {.uci = "e4e5", .san = "e5"},    {.uci = "c4e5", .san = "Nxe5"},
             {.uci = "f5e6", .san = "fxe6"},  {.uci = "e5d3", .san = "Nxd3"},
             {.uci = "d1d3", .san = "Qxd3"},  {.uci = "b6e6", .san = "Qxe6"},
             {.uci = "d3h7", .san = "Qxh7+"}, {.uci = "g8f7", .san = "Kf7"},
             {.uci = "c1h6", .san = "Bh6"},   {.uci = "f8h8", .san = "Rh8"},
             {.uci = "h7h8", .san = "Qxh8"},  {.uci = "c8c2", .san = "Rc2"},
             {.uci = "a1c1", .san = "Rc1"},   {.uci = "c2g2", .san = "Rxg2+"},
             {.uci = "g1f1", .san = "Kf1"},   {.uci = "e6b3", .san = "Qb3"},
             {.uci = "f1e1", .san = "Ke1"},   {.uci = "b3f3", .san = "Qf3"},
         }},
    {.desc = "S. Levitsky / F. J. Marshall / 1912\n\"The Gold coin game\"",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 46,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "e2e4", .san = "e4"},   {.uci = "d7d5", .san = "d5"},
             {.uci = "b1c3", .san = "Nc3"},  {.uci = "c7c5", .san = "c5"},
             {.uci = "g1f3", .san = "Nf3"},  {.uci = "b8c6", .san = "Nc6"},
             {.uci = "e4d5", .san = "exd5"}, {.uci = "e6d5", .san = "exd5"},
             {.uci = "f1e2", .san = "Be2"},  {.uci = "g8f6", .san = "Nf6"},
             {.uci = "e1g1", .san = "O-O"},  {.uci = "f8e7", .san = "Be7"},
             {.uci = "c1g5", .san = "Bg5"},  {.uci = "e8g8", .san = "O-O"},
             {.uci = "d4c5", .san = "dxc5"}, {.uci = "c8e6", .san = "Be6"},
             {.uci = "f3d4", .san = "Nd4"},  {.uci = "e7c5", .san = "Bxc5"},
             {.uci = "d4e6", .san = "Nxe6"}, {.uci = "f7e6", .san = "fxe6"},
             {.uci = "e2g4", .san = "Bg4"},  {.uci = "d8d6", .san = "Qd6"},
             {.uci = "g4h3", .san = "Bh3"},  {.uci = "a8e8", .san = "Rae8"},
             {.uci = "d1d2", .san = "Qd2"},  {.uci = "c5b4", .san = "Bb4"},
             {.uci = "g5f6", .san = "Bxf6"}, {.uci = "f8f6", .san = "Rxf6"},
             {.uci = "a1d1", .san = "Rad1"}, {.uci = "d6c5", .san = "Qc5"},
             {.uci = "d2e2", .san = "Qe2"},  {.uci = "b4c3", .san = "Bxc3"},
             {.uci = "b2c3", .san = "bxc3"}, {.uci = "c5c3", .san = "Qxc3"},
             {.uci = "d1d5", .san = "Rxd5"}, {.uci = "c6d4", .san = "Nd4"},
             {.uci = "e2h5", .san = "Qh5"},  {.uci = "e8f8", .san = "Ref8"},
             {.uci = "d5e5", .san = "Re5"},  {.uci = "f6h6", .san = "Rh6"},
             {.uci = "h5g5", .san = "Qg5"},  {.uci = "h6h3", .san = "Rxh3"},
             {.uci = "e5c5", .san = "Rc5"},  {.uci = "c3g3", .san = "Qg3"},
         }},
    {.desc = "E. N. Somov-Nasimovitsch / 1928 / Zadachi I Etiudi",
     .fen = "5k2/p1p1p3/2p1p2N/P1R5/P3KP2/8/p6P/b7",
     .movecount = 19,
     .moves =
         {
             {.uci = "c5c2", .san = "Rc2"},   {.uci = "a1b2", .san = "Bb2"},
             {.uci = "c2g2", .san = "Rg2"},   {.uci = "f8e8", .san = "Ke8"},
             {.uci = "g2g8", .san = "Rg8+"},  {.uci = "e8d7", .san = "Kd7"},
             {.uci = "h6f7", .san = "Nf7"},   {.uci = "c6c5", .san = "c5"},
             {.uci = "g8d8", .san = "Rd8+"},  {.uci = "d7c6", .san = "Kc6"},
             {.uci = "d8d2", .san = "Rd2"},   {.uci = "c5c4", .san = "c4"},
             {.uci = "f7d8", .san = "Nd8+"},  {.uci = "c6c5", .san = "Kc5"},
             {.uci = "d8e6", .san = "Nxe6+"}, {.uci = "c5c6", .san = "Kc6"},
             {.uci = "e6d8", .san = "Nd8+"},  {.uci = "c6c5", .san = "Kc5"},
             {.uci = "d2b2", .san = "Rxb2"},
         }},
    {.desc = "J. Hendel / 1859\nPayne's family journal",
     .fen = "8/K1N1p3/2k1N3/4P3/8/8/1P6/8",
     .movecount = 9,
     .moves =
         {
             {.uci = "b2b4", .san = "b4"},
             {.uci = "c6d7", .san = "Kd7"},
             {.uci = "b4b5", .san = "b5"},
             {.uci = "d7c8", .san = "Kc8"},
             {.uci = "b5b6", .san = "b6"},
             {.uci = "c8d7", .san = "Kd7"},
             {.uci = "b6b7", .san = "b7"},
             {.uci = "d7c6", .san = "Kc6"},
             {.uci = "b7b8n", .san = "b8=N#"},
         }},
    {.desc = "J. Hasek / 1929\nLa strategie",
     .fen = "3N3N/1Kpp4/8/1k6/1P3p2/P1P5/8/8",
     .movecount = 9,
     .moves =
         {
             {.uci = "d8c6", .san = "Nc6"},
             {.uci = "d7c6", .san = "dxc6"},
             {.uci = "h8g6", .san = "Ng6"},
             {.uci = "f4f3", .san = "f3"},
             {.uci = "g6e5", .san = "Ne5"},
             {.uci = "f3f2", .san = "f2"},
             {.uci = "e5d3", .san = "Nd3"},
             {.uci = "f2f1q", .san = "f1=Q"},
             {.uci = "d3b2", .san = "Nb2"},
         }},
    {.desc = "K. A. L. Kubbel / 1936\nChess in the USSR",
     .fen = "8/7p/4pK1k/5R2/8/1b2r1P1/8/5B2",
     .movecount = 9,
     .moves =
         {
             {.uci = "f5f2", .san = "Rf2"},
             {.uci = "b3d1", .san = "Bd1"},
             {.uci = "f2h2", .san = "Rh2"},
             {.uci = "d1h5", .san = "Bh5"},
             {.uci = "f1e2", .san = "Be2"},
             {.uci = "e3e2", .san = "Rxe2"},
             {.uci = "g3g4", .san = "g4"},
             {.uci = "e2h2", .san = "Rxh2"},
             {.uci = "g4g5", .san = "g5"},
         }},
    {.desc = "M. B. Newman / 1913\nChess amateur",
     .fen = "4q3/n7/8/7P/k1B1N3/P3N3/8/K7",
     .movecount = 11,
     .moves =
         {
             {.uci = "c4b5", .san = "Bb5+"},
             {.uci = "e8b5", .san = "Qxb5"},
             {.uci = "e4c3", .san = "Nc3+"},
             {.uci = "a4a3", .san = "Kxa3"},
             {.uci = "c3b5", .san = "Nxb5+"},
             {.uci = "a7b5", .san = "Nxb5"},
             {.uci = "h5h6", .san = "h6"},
             {.uci = "b5d6", .san = "Nd6"},
             {.uci = "e3c4", .san = "Nc4+"},
             {.uci = "d6c4", .san = "Nxc4"},
             {.uci = "h6h7", .san = "h7"},
         }},
    {.desc = "V. A. Korolikov / 1935 / Truda",
     .fen = "2B5/pR6/2pP1k2/8/6Kb/4p1P1/5p2/8",
     .movecount = 19,
     .moves =
         {
             {.uci = "d6d7", .san = "d7"},     {.uci = "f6e7", .san = "Ke7"},
             {.uci = "b7b8", .san = "Rb8"},    {.uci = "h4g3", .san = "Bxg3"},
             {.uci = "b8a8", .san = "Ra8"},    {.uci = "f2f1q", .san = "f1=Q"},
             {.uci = "d7d8q", .san = "d8=Q+"}, {.uci = "e7d8", .san = "Kxd8"},
             {.uci = "c8a6", .san = "Ba6+"},   {.uci = "g3b8", .san = "Bb8"},
             {.uci = "a6f1", .san = "Bxf1"},   {.uci = "d8c7", .san = "Kc7"},
             {.uci = "f1a6", .san = "Ba6"},    {.uci = "e3e2", .san = "e2"},
             {.uci = "a6e2", .san = "Bxe2"},   {.uci = "c7b7", .san = "Kb7"},
             {.uci = "e2f3", .san = "Bf3"},    {.uci = "b7a8", .san = "Kxa8"},
             {.uci = "f3c6", .san = "Bxc6#"},
         }},
    {.desc = "Gorgiev / 1929 / 64",
     .fen = "6N1/n7/3kn3/K7/6N1/8/8/4B3",
     .movecount = 11,
     .moves =
         {
             {.uci = "a5b6", .san = "Kb6"},
             {.uci = "a7c8", .san = "Nc8+"},
             {.uci = "b6b7", .san = "Kb7"},
             {.uci = "c8e7", .san = "Ne7"},
             {.uci = "e1g3", .san = "Bg3+"},
             {.uci = "d6d7", .san = "Kd7"},
             {.uci = "g8f6", .san = "N8f6+"},
             {.uci = "d7d8", .san = "Kd8"},
             {.uci = "g3c7", .san = "Bc7+"},
             {.uci = "e6c7", .san = "Nxc7"},
             {.uci = "g4e5", .san = "Ne5"},
         }},
    {.desc = "K. A. L. Kubbel / 1922\nSchachmatny Listok",
     .fen = "1N6/8/K7/3k4/3p3B/p7/2PP4/8",
     .movecount = 11,
     .moves =
         {
             {.uci = "b8c6", .san = "Nc6"},
             {.uci = "d5c6", .san = "Kxc6"},
             {.uci = "h4f6", .san = "Bf6"},
             {.uci = "c6d5", .san = "Kd5"},
             {.uci = "d2d3", .san = "d3"},
             {.uci = "a3a2", .san = "a2"},
             {.uci = "c2c4", .san = "c4+"},
             {.uci = "d5c5", .san = "Kc5"},
             {.uci = "a6b7", .san = "Kb7"},
             {.uci = "a2a1q", .san = "a1=Q"},
             {.uci = "f6e7", .san = "Be7#"},
         }},
    {.desc = "M. Carlsen / G. Kasparov / 2004 / Reykjavik",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 103,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},    {.uci = "d7d5", .san = "d5"},
             {.uci = "c2c4", .san = "c4"},    {.uci = "c7c6", .san = "c6"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "e7e6", .san = "e6"},
             {.uci = "c1g5", .san = "Bg5"},   {.uci = "b8d7", .san = "Nbd7"},
             {.uci = "e2e3", .san = "e3"},    {.uci = "d8a5", .san = "Qa5"},
             {.uci = "f3d2", .san = "Nd2"},   {.uci = "f8b4", .san = "Bb4"},
             {.uci = "d1c2", .san = "Qc2"},   {.uci = "e8g8", .san = "O-O"},
             {.uci = "f1e2", .san = "Be2"},   {.uci = "e6e5", .san = "e5"},
             {.uci = "e1g1", .san = "O-O"},   {.uci = "e5d4", .san = "exd4"},
             {.uci = "d2b3", .san = "Nb3"},   {.uci = "a5b6", .san = "Qb6"},
             {.uci = "e3d4", .san = "exd4"},  {.uci = "d5c4", .san = "dxc4"},
             {.uci = "e2c4", .san = "Bxc4"},  {.uci = "a7a5", .san = "a5"},
             {.uci = "a2a4", .san = "a4"},    {.uci = "b6c7", .san = "Qc7"},
             {.uci = "a1e1", .san = "Rae1"},  {.uci = "h7h6", .san = "h6"},
             {.uci = "g5h4", .san = "Bh4"},   {.uci = "b4d6", .san = "Bd6"},
             {.uci = "h2h3", .san = "h3"},    {.uci = "d7b6", .san = "Nb6"},
             {.uci = "h4f6", .san = "Bxf6"},  {.uci = "b6c4", .san = "Nxc4"},
             {.uci = "c3e4", .san = "Ne4"},   {.uci = "d6h2", .san = "Bh2+"},
             {.uci = "g1h1", .san = "Kh1"},   {.uci = "c4d6", .san = "Nd6"},
             {.uci = "h1h2", .san = "Kxh2"},  {.uci = "d6e4", .san = "Nxe4+"},
             {.uci = "f6e5", .san = "Be5"},   {.uci = "e4d6", .san = "Nd6"},
             {.uci = "c2c5", .san = "Qc5"},   {.uci = "f8d8", .san = "Rd8"},
             {.uci = "d4d5", .san = "d5"},    {.uci = "c7d7", .san = "Qd7"},
             {.uci = "b3d4", .san = "Nd4"},   {.uci = "d6f5", .san = "Nf5"},
             {.uci = "d5c6", .san = "dxc6"},  {.uci = "b7c6", .san = "bxc6"},
             {.uci = "d4c6", .san = "Nxc6"},  {.uci = "d8e8", .san = "Re8"},
             {.uci = "e1d1", .san = "Rd1"},   {.uci = "d7e6", .san = "Qe6"},
             {.uci = "f1e1", .san = "Rfe1"},  {.uci = "c8b7", .san = "Bb7"},
             {.uci = "c6d4", .san = "Nd4"},   {.uci = "f5d4", .san = "Nxd4"},
             {.uci = "c5d4", .san = "Qxd4"},  {.uci = "e6g6", .san = "Qg6"},
             {.uci = "d4g4", .san = "Qg4"},   {.uci = "g6g4", .san = "Qxg4"},
             {.uci = "h3g4", .san = "hxg4"},  {.uci = "b7c6", .san = "Bc6"},
             {.uci = "b2b3", .san = "b3"},    {.uci = "f7f6", .san = "f6"},
             {.uci = "e5c3", .san = "Bc3"},   {.uci = "e8e1", .san = "Rxe1"},
             {.uci = "d1e1", .san = "Rxe1"},  {.uci = "c6d5", .san = "Bd5"},
             {.uci = "e1b1", .san = "Rb1"},   {.uci = "g8f7", .san = "Kf7"},
             {.uci = "h2g3", .san = "Kg3"},   {.uci = "a8b8", .san = "Rb8"},
             {.uci = "b3b4", .san = "b4"},    {.uci = "a5b4", .san = "axb4"},
             {.uci = "c3b4", .san = "Bxb4"},  {.uci = "d5c4", .san = "Bc4"},
             {.uci = "a4a5", .san = "a5"},    {.uci = "c4a6", .san = "Ba6"},
             {.uci = "f2f3", .san = "f3"},    {.uci = "f7g6", .san = "Kg6"},
             {.uci = "g3f4", .san = "Kf4"},   {.uci = "h6h5", .san = "h5"},
             {.uci = "g4h5", .san = "gxh5+"}, {.uci = "g6h5", .san = "Kxh5"},
             {.uci = "b1h1", .san = "Rh1+"},  {.uci = "h5g6", .san = "Kg6"},
             {.uci = "b4c5", .san = "Bc5"},   {.uci = "b8b2", .san = "Rb2"},
             {.uci = "f4g3", .san = "Kg3"},   {.uci = "b2a2", .san = "Ra2"},
             {.uci = "c5b6", .san = "Bb6"},   {.uci = "g6f7", .san = "Kf7"},
             {.uci = "h1c1", .san = "Rc1"},   {.uci = "g7g5", .san = "g5"},
             {.uci = "c1c7", .san = "Rc7+"},  {.uci = "f7g6", .san = "Kg6"},
             {.uci = "c7c6", .san = "Rc6"},   {.uci = "a6f1", .san = "Bf1"},
             {.uci = "b6f2", .san = "Bf2"},
         }},
    {.desc = "M. Carlsen / V. Kramnik / 2016 / Norway",
     .fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR",
     .movecount = 99,
     .moves =
         {
             {.uci = "d2d4", .san = "d4"},    {.uci = "d7d5", .san = "d5"},
             {.uci = "c2c4", .san = "c4"},    {.uci = "e7e6", .san = "e6"},
             {.uci = "b1c3", .san = "Nc3"},   {.uci = "g8f6", .san = "Nf6"},
             {.uci = "c4d5", .san = "cxd5"},  {.uci = "e6d5", .san = "exd5"},
             {.uci = "c1g5", .san = "Bg5"},   {.uci = "c7c6", .san = "c6"},
             {.uci = "e2e3", .san = "e3"},    {.uci = "c8f5", .san = "Bf5"},
             {.uci = "d1f3", .san = "Qf3"},   {.uci = "f5g6", .san = "Bg6"},
             {.uci = "g5f6", .san = "Bxf6"},  {.uci = "d8f6", .san = "Qxf6"},
             {.uci = "f3f6", .san = "Qxf6"},  {.uci = "g7f6", .san = "gxf6"},
             {.uci = "g1f3", .san = "Nf3"},   {.uci = "b8d7", .san = "Nd7"},
             {.uci = "f3h4", .san = "Nh4"},   {.uci = "f8e7", .san = "Be7"},
             {.uci = "c3e2", .san = "Ne2"},   {.uci = "d7b6", .san = "Nb6"},
             {.uci = "e2g3", .san = "Ng3"},   {.uci = "e7b4", .san = "Bb4+"},
             {.uci = "e1d1", .san = "Kd1"},   {.uci = "b6a4", .san = "Na4"},
             {.uci = "g3f5", .san = "Ngf5"},  {.uci = "e8d7", .san = "Kd7"},
             {.uci = "a1b1", .san = "Rb1"},   {.uci = "d7e6", .san = "Ke6"},
             {.uci = "f1d3", .san = "Bd3"},   {.uci = "h8c8", .san = "Rhc8"},
             {.uci = "d1e2", .san = "Ke2"},   {.uci = "b4f8", .san = "Bf8"},
             {.uci = "g2g4", .san = "g4"},    {.uci = "c6c5", .san = "c5"},
             {.uci = "h4g2", .san = "Ng2"},   {.uci = "c5d4", .san = "cxd4"},
             {.uci = "e3d4", .san = "exd4"},  {.uci = "f8d6", .san = "Bd6"},
             {.uci = "h2h4", .san = "h4"},    {.uci = "h7h5", .san = "h5"},
             {.uci = "f5g7", .san = "Ng7+"},  {.uci = "e6e7", .san = "Ke7"},
             {.uci = "g4h5", .san = "gxh5"},  {.uci = "g6d3", .san = "Bxd3+"},
             {.uci = "e2d3", .san = "Kxd3"},  {.uci = "e7d7", .san = "Kd7"},
             {.uci = "g2e3", .san = "Ne3"},   {.uci = "a4b6", .san = "Nb6"},
             {.uci = "e3g4", .san = "Ng4"},   {.uci = "c8h8", .san = "Rh8"},
             {.uci = "h1e1", .san = "Rhe1"},  {.uci = "d6e7", .san = "Be7"},
             {.uci = "g7f5", .san = "Nf5"},   {.uci = "e7d8", .san = "Bd8"},
             {.uci = "h5h6", .san = "h6"},    {.uci = "a8c8", .san = "Rc8"},
             {.uci = "b2b3", .san = "b3"},    {.uci = "c8c6", .san = "Rc6"},
             {.uci = "g4e3", .san = "Nge3"},  {.uci = "d8c7", .san = "Bc7"},
             {.uci = "b1c1", .san = "Rbc1"},  {.uci = "c6c1", .san = "Rxc1"},
             {.uci = "e1c1", .san = "Rxc1"},  {.uci = "c7f4", .san = "Bf4"},
             {.uci = "c1c5", .san = "Rc5"},   {.uci = "d7e6", .san = "Ke6"},
             {.uci = "f5g7", .san = "Ng7+"},  {.uci = "e6d6", .san = "Kd6"},
             {.uci = "e3g4", .san = "Ng4"},   {.uci = "b6d7", .san = "Nd7"},
             {.uci = "c5c2", .san = "Rc2"},   {.uci = "f6f5", .san = "f5"},
             {.uci = "g7f5", .san = "Nxf5+"}, {.uci = "d6e6", .san = "Ke6"},
             {.uci = "f5g7", .san = "Ng7+"},  {.uci = "e6d6", .san = "Kd6"},
             {.uci = "c2e2", .san = "Re2"},   {.uci = "d6c6", .san = "Kc6"},
             {.uci = "e2e8", .san = "Re8"},   {.uci = "h8e8", .san = "Rxe8"},
             {.uci = "g7e8", .san = "Nxe8"},  {.uci = "d7f8", .san = "Nf8"},
             {.uci = "g4e5", .san = "Ne5+"},  {.uci = "f4e5", .san = "Bxe5"},
             {.uci = "d4e5", .san = "dxe5"},  {.uci = "c6d7", .san = "Kd7"},
             {.uci = "e8f6", .san = "Nf6+"},  {.uci = "d7e6", .san = "Ke6"},
             {.uci = "h4h5", .san = "h5"},    {.uci = "e6e5", .san = "Kxe5"},
             {.uci = "f6d7", .san = "Nd7+"},  {.uci = "f8d7", .san = "Nxd7"},
             {.uci = "h6h7", .san = "h7"},    {.uci = "d7c5", .san = "Nc5+"},
             {.uci = "d3e2", .san = "Ke2"},
         }},
};

#endif /* __CHESSGAMES_H__ */
