/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
package org.mozilla.focus.activity

import okhttp3.mockwebserver.MockWebServer
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.mozilla.focus.activity.robots.homeScreen
import org.mozilla.focus.activity.robots.searchScreen
import org.mozilla.focus.helpers.FeatureSettingsHelper
import org.mozilla.focus.helpers.MainActivityFirstrunTestRule
import org.mozilla.focus.helpers.MockWebServerHelper
import org.mozilla.focus.helpers.RetryTestRule
import org.mozilla.focus.helpers.TestAssetHelper.getGenericTabAsset
import org.mozilla.focus.helpers.TestHelper
import org.mozilla.focus.helpers.TestSetup
import org.mozilla.focus.testAnnotations.SmokeTest

/**
 * Verifies main menu items on the homescreen and on a browser page.
 */
class ThreeDotMainMenuTest : TestSetup() {
    private lateinit var webServer: MockWebServer
    private val featureSettingsHelper = FeatureSettingsHelper()

    @get:Rule
    val mActivityTestRule = MainActivityFirstrunTestRule(showFirstRun = false)

    @Rule
    @JvmField
    val retryTestRule = RetryTestRule(3)

    @Before
    override fun setUp() {
        super.setUp()
        webServer = MockWebServer().apply {
            dispatcher = MockWebServerHelper.AndroidAssetDispatcher()
            start()
        }
        featureSettingsHelper.setCfrForTrackingProtectionEnabled(false)
    }

    @After
    fun tearDown() {
        webServer.shutdown()
        featureSettingsHelper.resetAllFeatureFlags()
    }

    @SmokeTest
    @Test
    fun homeScreenMenuItemsTest() {
        homeScreen {
        }.openMainMenu {
            verifyHelpPageLinkExists()
            verifySettingsButtonExists()
        }
    }

    @SmokeTest
    @Test
    fun browserMenuItemsTest() {
        val pageUrl = webServer.getGenericTabAsset(1).url

        searchScreen {
        }.loadPage(pageUrl) {
            verifyPageContent("Tab 1")
        }.openMainMenu {
            verifyShareButtonExists()
            verifyAddToHomeButtonExists()
            verifyFindInPageExists()
            verifyOpenInButtonExists()
            verifyRequestDesktopSiteExists()
            verifySettingsButtonExists()
            verifyReportSiteIssueButtonExists()
        }
    }

    @SmokeTest
    @Test
    fun shareTabTest() {
        val pageUrl = webServer.getGenericTabAsset(1).url

        searchScreen {
        }.loadPage(pageUrl) {
            progressBar.waitUntilGone(TestHelper.waitingTime)
        }.openMainMenu {
        }.openShareScreen {
            verifyShareAppsListOpened()
            TestHelper.mDevice.pressBack()
        }
    }

    @SmokeTest
    @Test
    fun findInPageTest() {
        val pageUrl = webServer.getGenericTabAsset(1).url

        searchScreen {
        }.loadPage(pageUrl) {
            progressBar.waitUntilGone(TestHelper.waitingTime)
        }.openMainMenu {
        }.openFindInPage {
            enterFindInPageQuery("tab")
            verifyFindInPageResult("1/3")
            clickFindInPageNextButton()
            verifyFindInPageResult("2/3")
            clickFindInPageNextButton()
            verifyFindInPageResult("3/3")
            clickFindInPagePrevButton()
            verifyFindInPageResult("2/3")
            clickFindInPagePrevButton()
            verifyFindInPageResult("1/3")
            closeFindInPage()
        }
    }

    @SmokeTest
    @Test
    fun switchDesktopModeTest() {
        val pageUrl = webServer.getGenericTabAsset(1).url

        searchScreen {
        }.loadPage(pageUrl) {
            progressBar.waitUntilGone(TestHelper.waitingTime)
            verifyPageContent("mobile-site")
        }.openMainMenu {
        }.switchDesktopSiteMode {
            progressBar.waitUntilGone(TestHelper.waitingTime)
            verifyPageContent("desktop-site")
        }.openMainMenu {
            verifyRequestDesktopSiteIsEnabled(true)
        }.switchDesktopSiteMode {
            verifyPageContent("mobile-site")
        }.openMainMenu {
            verifyRequestDesktopSiteIsEnabled(false)
        }
    }
}
