﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/codebuild/model/FileSystemType.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace CodeBuild {
namespace Model {
namespace FileSystemTypeMapper {

static const int EFS_HASH = HashingUtils::HashString("EFS");

FileSystemType GetFileSystemTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == EFS_HASH) {
    return FileSystemType::EFS;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<FileSystemType>(hashCode);
  }

  return FileSystemType::NOT_SET;
}

Aws::String GetNameForFileSystemType(FileSystemType enumValue) {
  switch (enumValue) {
    case FileSystemType::NOT_SET:
      return {};
    case FileSystemType::EFS:
      return "EFS";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace FileSystemTypeMapper
}  // namespace Model
}  // namespace CodeBuild
}  // namespace Aws
