/*
 * Copyright (c) 2024, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.jpackage.internal.util.function;

import java.lang.reflect.InvocationTargetException;

public class ExceptionBox extends RuntimeException {

    private static final long serialVersionUID = 1L;

    public static RuntimeException toUnchecked(Exception ex) {
        switch (ex) {
            case RuntimeException rex -> {
                return rex;
            }
            case InvocationTargetException itex -> {
                var t = itex.getCause();
                if (t instanceof Exception cause) {
                    return toUnchecked(cause);
                } else {
                    throw (Error)t;
                }
            }
            case InterruptedException _ -> {
                Thread.currentThread().interrupt();
                return new ExceptionBox(ex);
            }
            default -> {
                return new ExceptionBox(ex);
            }
        }
    }

    public static Exception unbox(Throwable t) {
        switch (t) {
            case ExceptionBox ex -> {
                return unbox(ex.getCause());
            }
            case InvocationTargetException ex -> {
                return unbox(ex.getCause());
            }
            case Exception ex -> {
                return ex;
            }
            case Error err -> {
                throw err;
            }
            default -> {
                // Unreachable
                throw reachedUnreachable();
            }
        }
    }

    public static Error reachedUnreachable() {
        return new AssertionError("Reached unreachable!");
    }

    private ExceptionBox(Exception ex) {
        super(ex);
    }
}
