type token =
  | Param of (string)
  | AUTHOR
  | Version of (string)
  | See of (Types.see_ref)
  | Since of (string)
  | Before of (string)
  | DEPRECATED
  | Raise of (string)
  | RETURN
  | INLINE
  | Custom of (string)
  | Canonical of (string)
  | BEGIN
  | END
  | Title of (int * string option)
  | Style of (Types.style_kind)
  | LIST
  | ENUM
  | Item of (bool)
  | Ref of (Types.ref_kind * string)
  | Special_Ref of (Types.special_ref_kind)
  | Code of (string)
  | Pre_Code of (string)
  | Verb of (string)
  | Target of (string option * string)
  | HTML_Bold of (string)
  | HTML_END_BOLD
  | HTML_Center of (string)
  | HTML_END_CENTER
  | HTML_Left of (string)
  | HTML_END_LEFT
  | HTML_Right of (string)
  | HTML_END_RIGHT
  | HTML_Italic of (string)
  | HTML_END_ITALIC
  | HTML_Title of (string * int)
  | HTML_END_Title of (int)
  | HTML_List of (string)
  | HTML_END_LIST
  | HTML_Enum of (string)
  | HTML_END_ENUM
  | HTML_Item of (string)
  | HTML_END_ITEM
  | MINUS
  | PLUS
  | NEWLINE
  | EOF
  | BLANK
  | Char of (string)
  | DOT
  | Ref_part of (string)

open Parsing;;
let _ = parse_error;;
# 2 "src/octParser.mly"
open Common
open Types
open Errors

(* Convert lexing position into error position *)
let position p =
  { line = p.Lexing.pos_lnum;
    column = p.Lexing.pos_cnum - p.Lexing.pos_bol; }

(* Get the location of the symbol at a given position *)
let rhs_loc n =
  { start = position (Parsing.rhs_start_pos n);
    finish = position (Parsing.rhs_end_pos n); }

(* Useful strings *)

let sempty = ""

let sspace = " "

let snewline = "\n"

let sblank_line = "\n\n"

let sminus = "-"

let splus = "+"

(* Accumulators for text elements *)

type text_item =
    Blank
  | Newline
  | Blank_line
  | String of string
  | Element of text_element

let iminus = String sminus

let iplus = String splus

let skip_blank_or_newline = function
  | Blank :: rest -> rest
  | Newline :: rest -> rest
  | til -> til

let rec skip_whitespace = function
  | Blank :: rest -> skip_whitespace rest
  | Newline :: rest -> skip_whitespace rest
  | Blank_line :: rest -> skip_whitespace rest
  | til -> til

let rec convert acc stracc = function
  | [] ->
        if stracc = [] then acc
        else (Raw (String.concat sempty stracc)) :: acc
  | ti :: rest ->
      let acc, stracc =
        match ti with
        | Blank -> acc, (sspace :: stracc)
        | Newline -> acc, (snewline :: stracc)
        | String s -> acc, (s :: stracc)
        | Blank_line ->
            let acc =
              if stracc = [] then acc
              else (Raw (String.concat sempty stracc)) :: acc
            in
              (Newline :: acc), []
        | Element e ->
            let acc =
              if stracc = [] then acc
              else (Raw (String.concat sempty stracc)) :: acc
            in
              (e :: acc), []
      in
        convert acc stracc rest

let text til =
  let til = skip_whitespace til in
  let til = skip_whitespace (List.rev til) in
    convert [] [] til

let inner til =
  let til = skip_blank_or_newline til in
  let til = skip_blank_or_newline (List.rev til) in
    convert [] [] til

(* Error messages *)

let unclosed opening_name opening_num items closing_name closing_num =
  let error =
    let opening_loc = rhs_loc opening_num in
    let opening = opening_name in
    let closing = closing_name in
    Unclosed { opening_loc; opening; items; closing }
  in
  let loc = rhs_loc closing_num in
    raise (ParserError(loc, error))

let expecting num nonterm =
  let error = Expecting nonterm in
  let loc = rhs_loc num in
    raise (ParserError(loc, error))

(* Utilities for error messages *)

let title_to_string (i, _) =
  let i = string_of_int i in
    "{" ^ i

let style_to_string = function
  | SK_bold -> "{b"
  | SK_italic -> "{i"
  | SK_emphasize -> "{e"
  | SK_center -> "{C"
  | SK_left -> "{L"
  | SK_right -> "{R"
  | SK_superscript -> "{^"
  | SK_subscript -> "{_"
  | SK_custom s -> "{" ^ s

let item_to_string i = if i then "{-" else "{li"

let html_open_to_string t = "<" ^ t ^ ">"
let html_close_to_string t = "</" ^ t ^ ">"

# 184 "src/octParser.ml"
let yytransl_const = [|
  258 (* AUTHOR *);
  263 (* DEPRECATED *);
  265 (* RETURN *);
  266 (* INLINE *);
  269 (* BEGIN *);
  270 (* END *);
  273 (* LIST *);
  274 (* ENUM *);
  283 (* HTML_END_BOLD *);
  285 (* HTML_END_CENTER *);
  287 (* HTML_END_LEFT *);
  289 (* HTML_END_RIGHT *);
  291 (* HTML_END_ITALIC *);
  295 (* HTML_END_LIST *);
  297 (* HTML_END_ENUM *);
  299 (* HTML_END_ITEM *);
  300 (* MINUS *);
  301 (* PLUS *);
  302 (* NEWLINE *);
    0 (* EOF *);
  303 (* BLANK *);
  305 (* DOT *);
    0|]

let yytransl_block = [|
  257 (* Param *);
  259 (* Version *);
  260 (* See *);
  261 (* Since *);
  262 (* Before *);
  264 (* Raise *);
  267 (* Custom *);
  268 (* Canonical *);
  271 (* Title *);
  272 (* Style *);
  275 (* Item *);
  276 (* Ref *);
  277 (* Special_Ref *);
  278 (* Code *);
  279 (* Pre_Code *);
  280 (* Verb *);
  281 (* Target *);
  282 (* HTML_Bold *);
  284 (* HTML_Center *);
  286 (* HTML_Left *);
  288 (* HTML_Right *);
  290 (* HTML_Italic *);
  292 (* HTML_Title *);
  293 (* HTML_END_Title *);
  294 (* HTML_List *);
  296 (* HTML_Enum *);
  298 (* HTML_Item *);
  304 (* Char *);
  306 (* Ref_part *);
    0|]

let yylhs = "\255\255\
\001\000\001\000\002\000\002\000\005\000\005\000\004\000\004\000\
\004\000\004\000\004\000\004\000\006\000\006\000\006\000\008\000\
\008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
\010\000\010\000\011\000\011\000\011\000\012\000\012\000\012\000\
\007\000\007\000\007\000\007\000\009\000\009\000\009\000\013\000\
\013\000\014\000\014\000\014\000\014\000\015\000\015\000\015\000\
\003\000\003\000\003\000\003\000\003\000\016\000\016\000\016\000\
\016\000\016\000\017\000\017\000\017\000\017\000\017\000\017\000\
\020\000\020\000\020\000\020\000\020\000\021\000\021\000\021\000\
\021\000\021\000\026\000\026\000\026\000\026\000\026\000\022\000\
\022\000\022\000\022\000\022\000\023\000\023\000\023\000\023\000\
\023\000\018\000\018\000\018\000\018\000\019\000\019\000\019\000\
\019\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
\024\000\024\000\024\000\027\000\027\000\028\000\028\000\025\000\
\025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
\025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
\025\000\029\000\029\000\030\000\030\000\000\000\000\000"

let yylen = "\002\000\
\001\000\002\000\001\000\003\000\001\000\003\000\002\000\002\000\
\001\000\003\000\003\000\002\000\001\000\001\000\001\000\002\000\
\002\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
\001\000\002\000\001\000\002\000\002\000\002\000\002\000\002\000\
\000\000\001\000\001\000\001\000\001\000\001\000\002\000\001\000\
\002\000\001\000\002\000\002\000\002\000\001\000\001\000\001\000\
\001\000\001\000\002\000\002\000\002\000\001\000\002\000\002\000\
\003\000\002\000\001\000\001\000\002\000\002\000\002\000\002\000\
\001\000\001\000\001\000\001\000\001\000\002\000\002\000\001\000\
\001\000\001\000\002\000\002\000\002\000\003\000\003\000\002\000\
\003\000\003\000\004\000\002\000\002\000\003\000\003\000\004\000\
\002\000\002\000\003\000\003\000\004\000\002\000\003\000\003\000\
\004\000\003\000\003\000\003\000\003\000\005\000\004\000\003\000\
\005\000\004\000\003\000\001\000\004\000\004\000\001\000\001\000\
\001\000\001\000\001\000\001\000\003\000\003\000\003\000\003\000\
\003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
\003\000\003\000\003\000\005\000\004\000\003\000\005\000\004\000\
\003\000\001\000\003\000\003\000\003\000\002\000\002\000"

let yydefred = "\000\000\
\000\000\000\000\000\000\050\000\000\000\000\000\000\000\000\000\
\000\000\108\000\111\000\112\000\113\000\114\000\115\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\065\000\
\066\000\027\000\025\000\069\000\142\000\000\000\049\000\000\000\
\000\000\000\000\000\000\000\000\054\000\067\000\068\000\000\000\
\000\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\013\000\000\000\014\000\000\000\
\000\000\000\000\000\000\000\000\000\000\015\000\000\000\000\000\
\009\000\028\000\026\000\055\000\030\000\029\000\032\000\031\000\
\051\000\000\000\000\000\000\000\056\000\000\000\000\000\074\000\
\052\000\053\000\058\000\063\000\064\000\072\000\073\000\000\000\
\000\000\000\000\099\000\098\000\101\000\100\000\104\000\000\000\
\000\000\116\000\107\000\000\000\123\000\122\000\127\000\126\000\
\129\000\128\000\131\000\130\000\125\000\124\000\121\000\120\000\
\134\000\000\000\000\000\138\000\137\000\000\000\020\000\038\000\
\047\000\048\000\046\000\037\000\016\000\000\000\000\000\000\000\
\000\000\040\000\042\000\017\000\018\000\019\000\021\000\022\000\
\023\000\024\000\000\000\012\000\008\000\007\000\057\000\084\000\
\090\000\000\000\000\000\000\000\070\000\000\000\089\000\094\000\
\000\000\000\000\071\000\000\000\006\000\004\000\110\000\109\000\
\000\000\103\000\000\000\106\000\000\000\000\000\133\000\000\000\
\136\000\000\000\043\000\044\000\045\000\039\000\041\000\011\000\
\010\000\075\000\000\000\092\000\076\000\082\000\091\000\000\000\
\000\000\000\000\077\000\000\000\096\000\087\000\095\000\000\000\
\000\000\119\000\118\000\102\000\117\000\105\000\141\000\140\000\
\132\000\139\000\135\000\078\000\093\000\079\000\083\000\097\000\
\088\000"

let yydgoto = "\003\000\
\029\000\041\000\030\000\071\000\042\000\072\000\031\000\073\000\
\133\000\032\000\033\000\034\000\137\000\138\000\139\000\035\000\
\036\000\089\000\090\000\037\000\189\000\092\000\093\000\038\000\
\039\000\158\000\105\000\106\000\123\000\124\000"

let yysindex = "\131\000\
\187\255\221\254\000\000\000\000\003\255\187\255\187\255\091\255\
\091\255\000\000\000\000\000\000\000\000\000\000\000\000\187\255\
\187\255\187\255\187\255\187\255\187\255\091\255\091\255\000\000\
\000\000\000\000\000\000\000\000\000\000\235\255\000\000\206\003\
\133\255\147\255\242\003\094\004\000\000\000\000\000\000\047\255\
\046\255\000\000\187\255\090\255\096\255\014\255\149\255\133\255\
\147\255\021\255\089\255\086\255\081\255\076\255\070\255\031\255\
\011\255\013\255\187\255\000\255\000\000\187\255\000\000\187\255\
\187\255\187\255\187\255\091\255\187\255\000\000\235\255\016\255\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\206\003\133\255\147\255\000\000\133\003\170\003\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\053\255\
\221\254\099\255\000\000\000\000\000\000\000\000\000\000\187\255\
\018\255\000\000\000\000\020\255\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\187\255\026\255\000\000\000\000\028\255\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\101\255\107\255\130\255\
\138\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\032\255\000\000\000\000\000\000\000\000\000\000\
\000\000\206\003\022\004\147\255\000\000\242\003\000\000\000\000\
\058\004\147\255\000\000\242\003\000\000\000\000\000\000\000\000\
\100\255\000\000\078\255\000\000\079\255\008\255\000\000\244\254\
\000\000\188\255\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\128\004\000\000\000\000\000\000\000\000\206\003\
\022\004\147\255\000\000\128\004\000\000\000\000\000\000\058\004\
\147\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000"

let yyrindex = "\000\000\
\176\002\000\000\000\000\000\000\000\000\110\255\110\255\088\255\
\088\255\000\000\000\000\000\000\000\000\000\000\000\000\108\255\
\098\255\112\255\111\255\094\255\119\255\029\255\029\255\000\000\
\000\000\000\000\000\000\000\000\000\000\141\000\000\000\140\001\
\003\000\052\000\189\001\000\000\000\000\000\000\000\000\001\000\
\158\000\000\000\110\255\000\000\000\000\000\000\081\002\130\002\
\156\002\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\176\002\176\002\000\000\176\002\000\000\176\002\
\176\002\176\002\176\002\176\002\176\002\000\000\161\000\176\002\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\140\001\101\000\150\000\000\000\031\002\031\002\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\110\255\
\104\255\000\000\000\000\104\255\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\122\255\069\255\000\000\000\000\208\255\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\202\002\214\002\227\002\
\176\002\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\176\002\000\000\000\000\000\000\000\000\000\000\
\000\000\140\001\238\001\199\000\000\000\189\001\000\000\000\000\
\238\001\248\000\000\000\189\001\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\140\001\
\238\001\042\001\000\000\000\000\000\000\000\000\000\000\238\001\
\091\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000"

let yygindex = "\000\000\
\000\000\000\000\018\000\000\000\070\000\120\000\034\000\126\000\
\000\000\232\003\234\003\239\003\000\000\035\000\054\000\000\000\
\000\000\113\255\097\255\228\255\225\255\123\255\105\255\226\255\
\229\255\114\000\156\000\102\255\160\000\197\255"

let yytablesize = 1456
let yytable = "\128\000\
\005\000\197\000\035\000\076\000\091\000\094\000\085\000\207\000\
\095\000\198\000\121\000\188\000\125\000\103\000\040\000\149\000\
\205\000\170\000\205\000\172\000\107\000\190\000\043\000\044\000\
\045\000\175\000\209\000\177\000\033\000\122\000\119\000\184\000\
\104\000\051\000\052\000\053\000\054\000\055\000\056\000\104\000\
\216\000\046\000\050\000\129\000\130\000\026\000\027\000\131\000\
\217\000\213\000\208\000\036\000\122\000\151\000\122\000\057\000\
\058\000\157\000\163\000\215\000\098\000\026\000\027\000\026\000\
\027\000\026\000\027\000\120\000\081\000\117\000\033\000\026\000\
\027\000\026\000\027\000\115\000\127\000\026\000\027\000\140\000\
\113\000\141\000\142\000\143\000\144\000\111\000\146\000\033\000\
\109\000\099\000\096\000\204\000\206\000\132\000\097\000\101\000\
\104\000\104\000\167\000\202\000\035\000\145\000\165\000\100\000\
\118\000\150\000\033\000\033\000\116\000\102\000\033\000\114\000\
\168\000\203\000\112\000\110\000\210\000\033\000\210\000\153\000\
\160\000\169\000\033\000\033\000\094\000\186\000\033\000\095\000\
\033\000\195\000\094\000\001\000\002\000\095\000\033\000\195\000\
\026\000\027\000\171\000\174\000\001\000\173\000\033\000\033\000\
\129\000\130\000\074\000\075\000\131\000\036\000\129\000\130\000\
\077\000\078\000\131\000\033\000\176\000\143\000\163\000\178\000\
\002\000\214\000\094\000\212\000\033\000\095\000\166\000\157\000\
\214\000\094\000\182\000\183\000\095\000\129\000\130\000\079\000\
\080\000\131\000\077\000\078\000\185\000\129\000\130\000\026\000\
\027\000\131\000\004\000\179\000\180\000\181\000\147\000\191\000\
\079\000\080\000\074\000\075\000\148\000\199\000\036\000\005\000\
\164\000\006\000\007\000\008\000\009\000\108\000\010\000\011\000\
\012\000\013\000\014\000\015\000\016\000\000\000\017\000\000\000\
\018\000\126\000\019\000\000\000\020\000\000\000\021\000\000\000\
\022\000\000\000\023\000\000\000\211\000\122\000\024\000\025\000\
\026\000\027\000\028\000\059\000\060\000\061\000\062\000\063\000\
\064\000\065\000\066\000\067\000\068\000\069\000\070\000\036\000\
\033\000\033\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
\035\000\035\000\035\000\035\000\035\000\035\000\035\000\059\000\
\035\000\059\000\059\000\059\000\059\000\000\000\059\000\059\000\
\059\000\059\000\059\000\059\000\059\000\035\000\059\000\035\000\
\059\000\035\000\059\000\035\000\059\000\035\000\059\000\035\000\
\059\000\036\000\059\000\000\000\000\000\035\000\059\000\059\000\
\000\000\005\000\059\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\060\000\036\000\060\000\060\000\060\000\060\000\000\000\060\000\
\060\000\060\000\060\000\060\000\060\000\060\000\036\000\060\000\
\036\000\060\000\036\000\060\000\036\000\060\000\036\000\060\000\
\036\000\060\000\036\000\060\000\000\000\000\000\036\000\060\000\
\060\000\000\000\000\000\060\000\035\000\035\000\035\000\035\000\
\035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
\035\000\061\000\035\000\061\000\061\000\061\000\061\000\000\000\
\061\000\061\000\061\000\061\000\061\000\061\000\061\000\035\000\
\061\000\035\000\061\000\035\000\061\000\035\000\061\000\035\000\
\061\000\035\000\061\000\034\000\061\000\000\000\000\000\035\000\
\061\000\061\000\000\000\000\000\061\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\062\000\036\000\062\000\062\000\062\000\062\000\
\000\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
\036\000\062\000\036\000\062\000\036\000\062\000\036\000\062\000\
\036\000\062\000\036\000\062\000\033\000\062\000\000\000\000\000\
\036\000\062\000\062\000\000\000\000\000\062\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\080\000\036\000\080\000\080\000\080\000\
\080\000\000\000\080\000\080\000\080\000\080\000\080\000\080\000\
\080\000\036\000\080\000\036\000\080\000\036\000\080\000\036\000\
\080\000\036\000\080\000\036\000\080\000\035\000\080\000\000\000\
\000\000\036\000\080\000\080\000\000\000\000\000\080\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\085\000\036\000\085\000\085\000\
\085\000\085\000\000\000\085\000\085\000\085\000\085\000\085\000\
\085\000\085\000\036\000\085\000\036\000\085\000\036\000\085\000\
\036\000\085\000\036\000\085\000\036\000\085\000\033\000\085\000\
\000\000\000\000\036\000\085\000\085\000\000\000\000\000\085\000\
\000\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\081\000\036\000\
\081\000\081\000\081\000\081\000\000\000\081\000\081\000\081\000\
\081\000\081\000\081\000\081\000\036\000\081\000\036\000\081\000\
\036\000\081\000\036\000\081\000\036\000\081\000\036\000\081\000\
\034\000\081\000\000\000\000\000\036\000\081\000\081\000\000\000\
\000\000\081\000\036\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\086\000\
\036\000\086\000\086\000\086\000\086\000\000\000\086\000\086\000\
\086\000\086\000\086\000\086\000\086\000\036\000\086\000\036\000\
\086\000\036\000\086\000\036\000\086\000\036\000\086\000\036\000\
\086\000\035\000\086\000\000\000\000\000\036\000\086\000\086\000\
\000\000\000\000\086\000\034\000\034\000\034\000\034\000\034\000\
\034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
\000\000\034\000\000\000\036\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\034\000\000\000\
\034\000\000\000\034\000\000\000\034\000\000\000\034\000\033\000\
\034\000\000\000\000\000\000\000\000\000\000\000\034\000\000\000\
\000\000\000\000\000\000\000\000\033\000\033\000\033\000\033\000\
\033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
\033\000\034\000\033\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\035\000\000\000\033\000\
\000\000\033\000\000\000\033\000\000\000\033\000\000\000\033\000\
\000\000\033\000\036\000\000\000\000\000\000\000\000\000\033\000\
\000\000\000\000\000\000\000\000\000\000\035\000\035\000\035\000\
\035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
\035\000\035\000\000\000\035\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\035\000\000\000\035\000\000\000\035\000\000\000\035\000\000\000\
\035\000\000\000\035\000\000\000\000\000\000\000\000\000\000\000\
\035\000\000\000\000\000\000\000\000\000\000\000\000\000\033\000\
\033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
\033\000\033\000\033\000\000\000\033\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\033\000\000\000\033\000\000\000\033\000\000\000\033\000\
\000\000\033\000\000\000\033\000\000\000\000\000\000\000\000\000\
\000\000\033\000\000\000\000\000\000\000\000\000\000\000\000\000\
\034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
\034\000\034\000\034\000\034\000\034\000\000\000\034\000\000\000\
\000\000\000\000\000\000\034\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
\000\000\034\000\034\000\000\000\000\000\000\000\000\000\000\000\
\000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
\035\000\035\000\035\000\035\000\035\000\035\000\000\000\035\000\
\000\000\000\000\000\000\000\000\035\000\000\000\000\000\000\000\
\000\000\000\000\000\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
\035\000\036\000\035\000\035\000\000\000\000\000\036\000\000\000\
\033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
\033\000\033\000\033\000\033\000\000\000\000\000\000\000\000\000\
\000\000\000\000\036\000\000\000\036\000\036\000\000\000\000\000\
\000\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
\034\000\034\000\034\000\034\000\034\000\034\000\035\000\035\000\
\035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
\035\000\035\000\000\000\036\000\036\000\036\000\036\000\036\000\
\036\000\036\000\036\000\036\000\036\000\036\000\036\000\047\000\
\047\000\048\000\048\000\000\000\000\000\000\000\049\000\049\000\
\000\000\000\000\000\000\000\000\000\000\047\000\047\000\048\000\
\048\000\000\000\000\000\000\000\049\000\049\000\000\000\000\000\
\000\000\000\000\082\000\000\000\083\000\000\000\000\000\000\000\
\000\000\084\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\000\000\134\000\000\000\135\000\000\000\000\000\
\000\000\000\000\136\000\047\000\000\000\048\000\000\000\047\000\
\000\000\048\000\049\000\000\000\000\000\000\000\049\000\000\000\
\000\000\000\000\000\000\000\000\000\000\154\000\154\000\155\000\
\161\000\000\000\000\000\000\000\156\000\162\000\000\000\000\000\
\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\047\000\000\000\048\000\047\000\000\000\048\000\000\000\049\000\
\000\000\000\000\049\000\000\000\000\000\000\000\000\000\000\000\
\000\000\000\000\047\000\000\000\048\000\047\000\000\000\048\000\
\000\000\049\000\000\000\000\000\049\000\000\000\000\000\000\000\
\134\000\000\000\135\000\000\000\000\000\000\000\000\000\136\000\
\000\000\000\000\047\000\000\000\048\000\000\000\000\000\000\000\
\000\000\049\000\000\000\000\000\152\000\192\000\000\000\193\000\
\000\000\000\000\000\000\192\000\194\000\200\000\000\000\000\000\
\000\000\005\000\201\000\006\000\007\000\008\000\009\000\000\000\
\010\000\011\000\012\000\013\000\014\000\015\000\016\000\000\000\
\017\000\000\000\018\000\000\000\019\000\000\000\020\000\000\000\
\021\000\159\000\022\000\000\000\023\000\000\000\000\000\000\000\
\024\000\025\000\026\000\027\000\028\000\000\000\005\000\000\000\
\006\000\007\000\008\000\009\000\000\000\010\000\011\000\012\000\
\013\000\014\000\015\000\016\000\000\000\017\000\000\000\018\000\
\000\000\019\000\000\000\020\000\000\000\021\000\000\000\022\000\
\000\000\023\000\000\000\000\000\000\000\024\000\025\000\026\000\
\027\000\028\000\005\000\000\000\006\000\007\000\008\000\009\000\
\000\000\010\000\011\000\012\000\013\000\014\000\015\000\016\000\
\000\000\017\000\000\000\018\000\000\000\019\000\000\000\020\000\
\000\000\021\000\000\000\022\000\000\000\023\000\000\000\000\000\
\000\000\024\000\025\000\074\000\075\000\028\000\005\000\000\000\
\006\000\007\000\008\000\009\000\000\000\010\000\011\000\012\000\
\013\000\014\000\015\000\016\000\000\000\017\000\000\000\018\000\
\000\000\019\000\000\000\020\000\000\000\021\000\000\000\022\000\
\000\000\023\000\000\000\000\000\000\000\024\000\025\000\026\000\
\027\000\028\000\005\000\000\000\006\000\007\000\008\000\009\000\
\000\000\010\000\011\000\012\000\013\000\014\000\015\000\016\000\
\000\000\017\000\000\000\018\000\000\000\019\000\000\000\020\000\
\000\000\021\000\000\000\022\000\000\000\023\000\000\000\000\000\
\000\000\086\000\187\000\077\000\078\000\088\000\005\000\000\000\
\006\000\007\000\008\000\009\000\000\000\010\000\011\000\012\000\
\013\000\014\000\015\000\016\000\000\000\017\000\000\000\018\000\
\000\000\019\000\000\000\020\000\000\000\021\000\000\000\022\000\
\000\000\023\000\000\000\000\000\000\000\196\000\087\000\077\000\
\078\000\088\000\005\000\000\000\006\000\007\000\008\000\009\000\
\000\000\010\000\011\000\012\000\013\000\014\000\015\000\016\000\
\000\000\017\000\000\000\018\000\000\000\019\000\000\000\020\000\
\000\000\021\000\000\000\022\000\000\000\023\000\000\000\000\000\
\000\000\086\000\087\000\000\000\005\000\088\000\006\000\007\000\
\008\000\009\000\000\000\010\000\011\000\012\000\013\000\014\000\
\015\000\016\000\000\000\017\000\000\000\018\000\000\000\019\000\
\000\000\020\000\000\000\021\000\000\000\022\000\000\000\023\000\
\000\000\000\000\000\000\024\000\025\000\000\000\000\000\028\000"

let yycheck = "\000\001\
\000\000\161\000\000\000\032\000\036\000\036\000\035\000\000\001\
\036\000\161\000\000\001\155\000\000\001\000\001\050\001\000\001\
\171\000\000\001\173\000\000\001\000\001\155\000\020\001\006\000\
\007\000\000\001\039\001\000\001\000\001\042\001\000\001\000\001\
\019\001\016\000\017\000\018\000\019\000\020\000\021\000\019\001\
\200\000\008\000\009\000\044\001\045\001\046\001\047\001\048\001\
\200\000\193\000\043\001\000\000\042\001\082\000\042\001\022\000\
\023\000\086\000\087\000\193\000\043\000\046\001\047\001\046\001\
\047\001\046\001\047\001\037\001\035\000\000\001\042\001\046\001\
\047\001\046\001\047\001\000\001\059\000\046\001\047\001\062\000\
\000\001\064\000\065\000\066\000\067\000\000\001\069\000\000\001\
\000\001\000\001\044\001\014\001\014\001\060\000\049\001\000\001\
\019\001\019\001\000\001\000\001\000\000\068\000\050\001\014\001\
\035\001\072\000\019\001\039\001\033\001\014\001\042\001\031\001\
\014\001\014\001\029\001\027\001\176\000\014\001\178\000\086\000\
\087\000\104\000\019\001\014\001\155\000\154\000\029\001\155\000\
\035\001\158\000\161\000\001\000\002\000\161\000\027\001\164\000\
\046\001\047\001\105\000\122\000\000\000\108\000\031\001\033\001\
\044\001\045\001\046\001\047\001\048\001\000\000\044\001\045\001\
\046\001\047\001\048\001\037\001\123\000\000\000\187\000\126\000\
\000\000\193\000\193\000\192\000\043\001\193\000\097\000\196\000\
\200\000\200\000\137\000\137\000\200\000\044\001\045\001\046\001\
\047\001\048\001\046\001\047\001\147\000\044\001\045\001\046\001\
\047\001\048\001\000\001\134\000\135\000\136\000\071\000\158\000\
\046\001\047\001\046\001\047\001\071\000\164\000\000\000\013\001\
\087\000\015\001\016\001\017\001\018\001\050\000\020\001\021\001\
\022\001\023\001\024\001\025\001\026\001\255\255\028\001\255\255\
\030\001\058\000\032\001\255\255\034\001\255\255\036\001\255\255\
\038\001\255\255\040\001\255\255\041\001\042\001\044\001\045\001\
\046\001\047\001\048\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\000\000\
\041\001\042\001\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\000\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\255\255\020\001\021\001\
\022\001\023\001\024\001\025\001\026\001\027\001\028\001\029\001\
\030\001\031\001\032\001\033\001\034\001\035\001\036\001\037\001\
\038\001\000\000\040\001\255\255\255\255\043\001\044\001\045\001\
\255\255\049\001\048\001\000\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\013\001\014\001\015\001\016\001\017\001\018\001\255\255\020\001\
\021\001\022\001\023\001\024\001\025\001\026\001\027\001\028\001\
\029\001\030\001\031\001\032\001\033\001\034\001\035\001\036\001\
\037\001\038\001\000\000\040\001\255\255\255\255\043\001\044\001\
\045\001\255\255\255\255\048\001\000\001\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\013\001\014\001\015\001\016\001\017\001\018\001\255\255\
\020\001\021\001\022\001\023\001\024\001\025\001\026\001\027\001\
\028\001\029\001\030\001\031\001\032\001\033\001\034\001\035\001\
\036\001\037\001\038\001\000\000\040\001\255\255\255\255\043\001\
\044\001\045\001\255\255\255\255\048\001\000\001\001\001\002\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\013\001\014\001\015\001\016\001\017\001\018\001\
\255\255\020\001\021\001\022\001\023\001\024\001\025\001\026\001\
\027\001\028\001\029\001\030\001\031\001\032\001\033\001\034\001\
\035\001\036\001\037\001\038\001\000\000\040\001\255\255\255\255\
\043\001\044\001\045\001\255\255\255\255\048\001\000\001\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\013\001\014\001\015\001\016\001\017\001\
\018\001\255\255\020\001\021\001\022\001\023\001\024\001\025\001\
\026\001\027\001\028\001\029\001\030\001\031\001\032\001\033\001\
\034\001\035\001\036\001\037\001\038\001\000\000\040\001\255\255\
\255\255\043\001\044\001\045\001\255\255\255\255\048\001\000\001\
\001\001\002\001\003\001\004\001\005\001\006\001\007\001\008\001\
\009\001\010\001\011\001\012\001\013\001\014\001\015\001\016\001\
\017\001\018\001\255\255\020\001\021\001\022\001\023\001\024\001\
\025\001\026\001\027\001\028\001\029\001\030\001\031\001\032\001\
\033\001\034\001\035\001\036\001\037\001\038\001\000\000\040\001\
\255\255\255\255\043\001\044\001\045\001\255\255\255\255\048\001\
\255\255\000\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\013\001\014\001\
\015\001\016\001\017\001\018\001\255\255\020\001\021\001\022\001\
\023\001\024\001\025\001\026\001\027\001\028\001\029\001\030\001\
\031\001\032\001\033\001\034\001\035\001\036\001\037\001\038\001\
\000\000\040\001\255\255\255\255\043\001\044\001\045\001\255\255\
\255\255\048\001\000\001\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\013\001\
\014\001\015\001\016\001\017\001\018\001\255\255\020\001\021\001\
\022\001\023\001\024\001\025\001\026\001\027\001\028\001\029\001\
\030\001\031\001\032\001\033\001\034\001\035\001\036\001\037\001\
\038\001\000\000\040\001\255\255\255\255\043\001\044\001\045\001\
\255\255\255\255\048\001\000\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\255\255\014\001\255\255\000\000\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\027\001\255\255\
\029\001\255\255\031\001\255\255\033\001\255\255\035\001\000\000\
\037\001\255\255\255\255\255\255\255\255\255\255\043\001\255\255\
\255\255\255\255\255\255\255\255\000\001\001\001\002\001\003\001\
\004\001\005\001\006\001\007\001\008\001\009\001\010\001\011\001\
\012\001\000\000\014\001\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\000\000\255\255\027\001\
\255\255\029\001\255\255\031\001\255\255\033\001\255\255\035\001\
\255\255\037\001\000\000\255\255\255\255\255\255\255\255\043\001\
\255\255\255\255\255\255\255\255\255\255\000\001\001\001\002\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\255\255\014\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\027\001\255\255\029\001\255\255\031\001\255\255\033\001\255\255\
\035\001\255\255\037\001\255\255\255\255\255\255\255\255\255\255\
\043\001\255\255\255\255\255\255\255\255\255\255\255\255\001\001\
\002\001\003\001\004\001\005\001\006\001\007\001\008\001\009\001\
\010\001\011\001\012\001\255\255\014\001\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\027\001\255\255\029\001\255\255\031\001\255\255\033\001\
\255\255\035\001\255\255\037\001\255\255\255\255\255\255\255\255\
\255\255\043\001\255\255\255\255\255\255\255\255\255\255\255\255\
\000\001\001\001\002\001\003\001\004\001\005\001\006\001\007\001\
\008\001\009\001\010\001\011\001\012\001\255\255\014\001\255\255\
\255\255\255\255\255\255\019\001\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\039\001\
\255\255\041\001\042\001\255\255\255\255\255\255\255\255\255\255\
\255\255\000\001\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\255\255\014\001\
\255\255\255\255\255\255\255\255\019\001\255\255\255\255\255\255\
\255\255\255\255\255\255\000\001\001\001\002\001\003\001\004\001\
\005\001\006\001\007\001\008\001\009\001\010\001\011\001\012\001\
\039\001\014\001\041\001\042\001\255\255\255\255\019\001\255\255\
\001\001\002\001\003\001\004\001\005\001\006\001\007\001\008\001\
\009\001\010\001\011\001\012\001\255\255\255\255\255\255\255\255\
\255\255\255\255\039\001\255\255\041\001\042\001\255\255\255\255\
\255\255\255\255\001\001\002\001\003\001\004\001\005\001\006\001\
\007\001\008\001\009\001\010\001\011\001\012\001\001\001\002\001\
\003\001\004\001\005\001\006\001\007\001\008\001\009\001\010\001\
\011\001\012\001\255\255\001\001\002\001\003\001\004\001\005\001\
\006\001\007\001\008\001\009\001\010\001\011\001\012\001\008\000\
\009\000\008\000\009\000\255\255\255\255\255\255\008\000\009\000\
\255\255\255\255\255\255\255\255\255\255\022\000\023\000\022\000\
\023\000\255\255\255\255\255\255\022\000\023\000\255\255\255\255\
\255\255\255\255\035\000\255\255\035\000\255\255\255\255\255\255\
\255\255\035\000\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\255\255\060\000\255\255\060\000\255\255\255\255\
\255\255\255\255\060\000\068\000\255\255\068\000\255\255\072\000\
\255\255\072\000\068\000\255\255\255\255\255\255\072\000\255\255\
\255\255\255\255\255\255\255\255\255\255\086\000\087\000\086\000\
\087\000\255\255\255\255\255\255\086\000\087\000\255\255\255\255\
\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
\105\000\255\255\105\000\108\000\255\255\108\000\255\255\105\000\
\255\255\255\255\108\000\255\255\255\255\255\255\255\255\255\255\
\255\255\255\255\123\000\255\255\123\000\126\000\255\255\126\000\
\255\255\123\000\255\255\255\255\126\000\255\255\255\255\255\255\
\137\000\255\255\137\000\255\255\255\255\255\255\255\255\137\000\
\255\255\255\255\147\000\255\255\147\000\255\255\255\255\255\255\
\255\255\147\000\255\255\255\255\000\001\158\000\255\255\158\000\
\255\255\255\255\255\255\164\000\158\000\164\000\255\255\255\255\
\255\255\013\001\164\000\015\001\016\001\017\001\018\001\255\255\
\020\001\021\001\022\001\023\001\024\001\025\001\026\001\255\255\
\028\001\255\255\030\001\255\255\032\001\255\255\034\001\255\255\
\036\001\000\001\038\001\255\255\040\001\255\255\255\255\255\255\
\044\001\045\001\046\001\047\001\048\001\255\255\013\001\255\255\
\015\001\016\001\017\001\018\001\255\255\020\001\021\001\022\001\
\023\001\024\001\025\001\026\001\255\255\028\001\255\255\030\001\
\255\255\032\001\255\255\034\001\255\255\036\001\255\255\038\001\
\255\255\040\001\255\255\255\255\255\255\044\001\045\001\046\001\
\047\001\048\001\013\001\255\255\015\001\016\001\017\001\018\001\
\255\255\020\001\021\001\022\001\023\001\024\001\025\001\026\001\
\255\255\028\001\255\255\030\001\255\255\032\001\255\255\034\001\
\255\255\036\001\255\255\038\001\255\255\040\001\255\255\255\255\
\255\255\044\001\045\001\046\001\047\001\048\001\013\001\255\255\
\015\001\016\001\017\001\018\001\255\255\020\001\021\001\022\001\
\023\001\024\001\025\001\026\001\255\255\028\001\255\255\030\001\
\255\255\032\001\255\255\034\001\255\255\036\001\255\255\038\001\
\255\255\040\001\255\255\255\255\255\255\044\001\045\001\046\001\
\047\001\048\001\013\001\255\255\015\001\016\001\017\001\018\001\
\255\255\020\001\021\001\022\001\023\001\024\001\025\001\026\001\
\255\255\028\001\255\255\030\001\255\255\032\001\255\255\034\001\
\255\255\036\001\255\255\038\001\255\255\040\001\255\255\255\255\
\255\255\044\001\045\001\046\001\047\001\048\001\013\001\255\255\
\015\001\016\001\017\001\018\001\255\255\020\001\021\001\022\001\
\023\001\024\001\025\001\026\001\255\255\028\001\255\255\030\001\
\255\255\032\001\255\255\034\001\255\255\036\001\255\255\038\001\
\255\255\040\001\255\255\255\255\255\255\044\001\045\001\046\001\
\047\001\048\001\013\001\255\255\015\001\016\001\017\001\018\001\
\255\255\020\001\021\001\022\001\023\001\024\001\025\001\026\001\
\255\255\028\001\255\255\030\001\255\255\032\001\255\255\034\001\
\255\255\036\001\255\255\038\001\255\255\040\001\255\255\255\255\
\255\255\044\001\045\001\255\255\013\001\048\001\015\001\016\001\
\017\001\018\001\255\255\020\001\021\001\022\001\023\001\024\001\
\025\001\026\001\255\255\028\001\255\255\030\001\255\255\032\001\
\255\255\034\001\255\255\036\001\255\255\038\001\255\255\040\001\
\255\255\255\255\255\255\044\001\045\001\255\255\255\255\048\001"

let yynames_const = "\
  AUTHOR\000\
  DEPRECATED\000\
  RETURN\000\
  INLINE\000\
  BEGIN\000\
  END\000\
  LIST\000\
  ENUM\000\
  HTML_END_BOLD\000\
  HTML_END_CENTER\000\
  HTML_END_LEFT\000\
  HTML_END_RIGHT\000\
  HTML_END_ITALIC\000\
  HTML_END_LIST\000\
  HTML_END_ENUM\000\
  HTML_END_ITEM\000\
  MINUS\000\
  PLUS\000\
  NEWLINE\000\
  EOF\000\
  BLANK\000\
  DOT\000\
  "

let yynames_block = "\
  Param\000\
  Version\000\
  See\000\
  Since\000\
  Before\000\
  Raise\000\
  Custom\000\
  Canonical\000\
  Title\000\
  Style\000\
  Item\000\
  Ref\000\
  Special_Ref\000\
  Code\000\
  Pre_Code\000\
  Verb\000\
  Target\000\
  HTML_Bold\000\
  HTML_Center\000\
  HTML_Left\000\
  HTML_Right\000\
  HTML_Italic\000\
  HTML_Title\000\
  HTML_END_Title\000\
  HTML_List\000\
  HTML_Enum\000\
  HTML_Item\000\
  Char\000\
  Ref_part\000\
  "

let yyact = [|
  (fun _ -> failwith "parser")
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 205 "src/octParser.mly"
                    ( (text _1, []) )
# 817 "src/octParser.ml"
               : Types.t))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'tags) in
    Obj.repr(
# 206 "src/octParser.mly"
                    ( (text _1, List.rev _2) )
# 825 "src/octParser.ml"
               : Types.t))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'reference_part) in
    Obj.repr(
# 211 "src/octParser.mly"
                 ( [_1] )
# 832 "src/octParser.ml"
               : (string option * string) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : (string option * string) list) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'reference_part) in
    Obj.repr(
# 212 "src/octParser.mly"
                                     ( _3 :: _1 )
# 840 "src/octParser.ml"
               : (string option * string) list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 216 "src/octParser.mly"
           ( (None, _1) )
# 847 "src/octParser.ml"
               : 'reference_part))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 217 "src/octParser.mly"
                          ( (Some _1, _3) )
# 855 "src/octParser.ml"
               : 'reference_part))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'simple_tag) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 223 "src/octParser.mly"
                                   ( [_1] )
# 863 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'simple_tag) in
    Obj.repr(
# 224 "src/octParser.mly"
                                   ( expecting 2 "tag" )
# 870 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text_tag) in
    Obj.repr(
# 225 "src/octParser.mly"
                                   ( [_1] )
# 877 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'tags) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'simple_tag) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 226 "src/octParser.mly"
                                   ( _2 :: _1 )
# 886 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'tags) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'simple_tag) in
    Obj.repr(
# 227 "src/octParser.mly"
                                   ( expecting 3 "tag" )
# 894 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'tags) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_tag) in
    Obj.repr(
# 228 "src/octParser.mly"
                                   ( _2 :: _1 )
# 902 "src/octParser.ml"
               : 'tags))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 232 "src/octParser.mly"
                    ( Version _1 )
# 909 "src/octParser.ml"
               : 'simple_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 233 "src/octParser.mly"
                    ( Since _1 )
# 916 "src/octParser.ml"
               : 'simple_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 234 "src/octParser.mly"
                    ( Canonical _1 )
# 923 "src/octParser.ml"
               : 'simple_tag))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'string) in
    Obj.repr(
# 238 "src/octParser.mly"
                         ( Author (String.concat sempty _2) )
# 930 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : Types.see_ref) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 239 "src/octParser.mly"
                         ( See(_1, (text _2)) )
# 938 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 240 "src/octParser.mly"
                         ( Before(_1, (text _2)) )
# 946 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 241 "src/octParser.mly"
                         ( Deprecated (text _2) )
# 953 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 242 "src/octParser.mly"
                         ( Param(_1, (text _2)) )
# 961 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 243 "src/octParser.mly"
                         ( Raised_exception(_1, (text _2)) )
# 969 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 244 "src/octParser.mly"
                         ( Return_value (text _2) )
# 976 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 245 "src/octParser.mly"
                         ( Inline )
# 983 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text) in
    Obj.repr(
# 246 "src/octParser.mly"
                         ( Custom(_1, (text _2)) )
# 991 "src/octParser.ml"
               : 'text_tag))
; (fun __caml_parser_env ->
    Obj.repr(
# 252 "src/octParser.mly"
                    ( () )
# 997 "src/octParser.ml"
               : 'blanks))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    Obj.repr(
# 253 "src/octParser.mly"
                    ( () )
# 1004 "src/octParser.ml"
               : 'blanks))
; (fun __caml_parser_env ->
    Obj.repr(
# 257 "src/octParser.mly"
                    ( () )
# 1010 "src/octParser.ml"
               : 'newline))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    Obj.repr(
# 258 "src/octParser.mly"
                    ( () )
# 1017 "src/octParser.ml"
               : 'newline))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    Obj.repr(
# 259 "src/octParser.mly"
                    ( () )
# 1024 "src/octParser.ml"
               : 'newline))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    Obj.repr(
# 263 "src/octParser.mly"
                    ( () )
# 1031 "src/octParser.ml"
               : 'blank_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blank_line) in
    Obj.repr(
# 264 "src/octParser.mly"
                     ( () )
# 1038 "src/octParser.ml"
               : 'blank_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blank_line) in
    Obj.repr(
# 265 "src/octParser.mly"
                     ( () )
# 1045 "src/octParser.ml"
               : 'blank_line))
; (fun __caml_parser_env ->
    Obj.repr(
# 269 "src/octParser.mly"
                        ( [] )
# 1051 "src/octParser.ml"
               : 'whitespace))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'blanks) in
    Obj.repr(
# 270 "src/octParser.mly"
                        ( [Blank] )
# 1058 "src/octParser.ml"
               : 'whitespace))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'newline) in
    Obj.repr(
# 271 "src/octParser.mly"
                        ( [Newline] )
# 1065 "src/octParser.ml"
               : 'whitespace))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 272 "src/octParser.mly"
                        ( [Blank_line] )
# 1072 "src/octParser.ml"
               : 'whitespace))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 278 "src/octParser.mly"
                                                  ( [] )
# 1079 "src/octParser.ml"
               : 'string))
; (fun __caml_parser_env ->
    Obj.repr(
# 279 "src/octParser.mly"
                                                  ( expecting 1 "string" )
# 1085 "src/octParser.ml"
               : 'string))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'string_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 280 "src/octParser.mly"
                                                  ( List.rev _1 )
# 1093 "src/octParser.ml"
               : 'string))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'string_item) in
    Obj.repr(
# 284 "src/octParser.mly"
                                   ( [snd _1] )
# 1100 "src/octParser.ml"
               : 'string_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'string_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'string_item) in
    Obj.repr(
# 285 "src/octParser.mly"
                                   ( (snd _2) :: (fst _2) :: _1 )
# 1108 "src/octParser.ml"
               : 'string_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'string_char) in
    Obj.repr(
# 289 "src/octParser.mly"
                              ( (sempty, _1) )
# 1115 "src/octParser.ml"
               : 'string_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'string_char) in
    Obj.repr(
# 290 "src/octParser.mly"
                              ( (sspace, _2) )
# 1123 "src/octParser.ml"
               : 'string_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'string_char) in
    Obj.repr(
# 291 "src/octParser.mly"
                              ( (snewline, _2) )
# 1131 "src/octParser.ml"
               : 'string_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blank_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'string_char) in
    Obj.repr(
# 292 "src/octParser.mly"
                              ( (sblank_line, _2) )
# 1139 "src/octParser.ml"
               : 'string_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 296 "src/octParser.mly"
                              ( _1 )
# 1146 "src/octParser.ml"
               : 'string_char))
; (fun __caml_parser_env ->
    Obj.repr(
# 297 "src/octParser.mly"
                              ( sminus )
# 1152 "src/octParser.ml"
               : 'string_char))
; (fun __caml_parser_env ->
    Obj.repr(
# 298 "src/octParser.mly"
                              ( splus )
# 1158 "src/octParser.ml"
               : 'string_char))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 303 "src/octParser.mly"
                                             ( _1 )
# 1165 "src/octParser.ml"
               : 'text))
; (fun __caml_parser_env ->
    Obj.repr(
# 304 "src/octParser.mly"
                                             ( expecting 1 "text" )
# 1171 "src/octParser.ml"
               : 'text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 305 "src/octParser.mly"
                                             ( List.rev_append _1 _2 )
# 1179 "src/octParser.ml"
               : 'text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body_with_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list_final) in
    Obj.repr(
# 306 "src/octParser.mly"
                                             ( List.rev_append _1 [Element (List _2)] )
# 1187 "src/octParser.ml"
               : 'text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body_with_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum_final) in
    Obj.repr(
# 307 "src/octParser.mly"
                                             ( List.rev_append _1 [Element (Enum _2)] )
# 1195 "src/octParser.ml"
               : 'text))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 311 "src/octParser.mly"
                                             ( [_1] )
# 1202 "src/octParser.ml"
               : 'text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 312 "src/octParser.mly"
                                             ( [_2; Blank] )
# 1210 "src/octParser.ml"
               : 'text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 313 "src/octParser.mly"
                                             ( _2 :: _1 )
# 1218 "src/octParser.ml"
               : 'text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 314 "src/octParser.mly"
                                             ( _3 :: Blank :: _1 )
# 1227 "src/octParser.ml"
               : 'text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body_with_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item_after_line) in
    Obj.repr(
# 315 "src/octParser.mly"
                                             ( List.rev_append _2 _1 )
# 1235 "src/octParser.ml"
               : 'text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'newline) in
    Obj.repr(
# 319 "src/octParser.mly"
                                             ( [Newline] )
# 1242 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 320 "src/octParser.mly"
                                             ( [Blank_line] )
# 1249 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'newline) in
    Obj.repr(
# 321 "src/octParser.mly"
                                             ( Newline :: _1 )
# 1257 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 322 "src/octParser.mly"
                                              ( Blank_line :: _1 )
# 1265 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body_with_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list) in
    Obj.repr(
# 323 "src/octParser.mly"
                                             ( (Element (List _2)) :: _1 )
# 1273 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'text_body_with_line) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum) in
    Obj.repr(
# 324 "src/octParser.mly"
                                             ( (Element (Enum _2)) :: _1 )
# 1281 "src/octParser.ml"
               : 'text_body_with_line))
; (fun __caml_parser_env ->
    Obj.repr(
# 328 "src/octParser.mly"
                                     ( iminus )
# 1287 "src/octParser.ml"
               : 'text_item))
; (fun __caml_parser_env ->
    Obj.repr(
# 329 "src/octParser.mly"
                                     ( iplus )
# 1293 "src/octParser.ml"
               : 'text_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text_element) in
    Obj.repr(
# 330 "src/octParser.mly"
                                     ( Element _1 )
# 1300 "src/octParser.ml"
               : 'text_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'html_text_element) in
    Obj.repr(
# 331 "src/octParser.mly"
                                     ( Element _1 )
# 1307 "src/octParser.ml"
               : 'text_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 332 "src/octParser.mly"
                                     ( String _1 )
# 1314 "src/octParser.ml"
               : 'text_item))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 336 "src/octParser.mly"
                             ( [iminus; _2] )
# 1321 "src/octParser.ml"
               : 'text_item_after_line))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 337 "src/octParser.mly"
                             ( [iplus; _2] )
# 1328 "src/octParser.ml"
               : 'text_item_after_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'text_element) in
    Obj.repr(
# 338 "src/octParser.mly"
                             ( [Element _1] )
# 1335 "src/octParser.ml"
               : 'text_item_after_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'html_text_element) in
    Obj.repr(
# 339 "src/octParser.mly"
                             ( [Element _1] )
# 1342 "src/octParser.ml"
               : 'text_item_after_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 340 "src/octParser.mly"
                             ( [String _1] )
# 1349 "src/octParser.ml"
               : 'text_item_after_line))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 346 "src/octParser.mly"
                                                    ( [_2; Blank] )
# 1357 "src/octParser.ml"
               : 'shortcut_text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item_after_line) in
    Obj.repr(
# 347 "src/octParser.mly"
                                                    ( List.rev_append _2 [Newline] )
# 1365 "src/octParser.ml"
               : 'shortcut_text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 1 : 'shortcut_text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 348 "src/octParser.mly"
                                                    ( _2 :: _1 )
# 1373 "src/octParser.ml"
               : 'shortcut_text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'blanks) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'text_item) in
    Obj.repr(
# 349 "src/octParser.mly"
                                                    ( _3 :: Blank :: _1 )
# 1382 "src/octParser.ml"
               : 'shortcut_text_body))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'text_item_after_line) in
    Obj.repr(
# 350 "src/octParser.mly"
                                                    ( List.rev_append _3 (Newline :: _1) )
# 1391 "src/octParser.ml"
               : 'shortcut_text_body))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 356 "src/octParser.mly"
                                                    ( [[]] )
# 1398 "src/octParser.ml"
               : 'shortcut_list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 357 "src/octParser.mly"
                                                    ( [inner (List.rev _2)] )
# 1406 "src/octParser.ml"
               : 'shortcut_list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list) in
    Obj.repr(
# 358 "src/octParser.mly"
                                                    ( [] :: _3 )
# 1414 "src/octParser.ml"
               : 'shortcut_list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list) in
    Obj.repr(
# 359 "src/octParser.mly"
                                                    ( (inner (List.rev _2)) :: _4 )
# 1423 "src/octParser.ml"
               : 'shortcut_list))
; (fun __caml_parser_env ->
    Obj.repr(
# 360 "src/octParser.mly"
                                                    ( expecting 2 "list item" )
# 1429 "src/octParser.ml"
               : 'shortcut_list))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 364 "src/octParser.mly"
                                                    ( [[]] )
# 1436 "src/octParser.ml"
               : 'shortcut_enum))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'blank_line) in
    Obj.repr(
# 365 "src/octParser.mly"
                                                    ( [inner (List.rev _2)] )
# 1444 "src/octParser.ml"
               : 'shortcut_enum))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum) in
    Obj.repr(
# 366 "src/octParser.mly"
                                                    ( [] :: _3 )
# 1452 "src/octParser.ml"
               : 'shortcut_enum))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum) in
    Obj.repr(
# 367 "src/octParser.mly"
                                                    ( (inner (List.rev _2)) :: _4 )
# 1461 "src/octParser.ml"
               : 'shortcut_enum))
; (fun __caml_parser_env ->
    Obj.repr(
# 368 "src/octParser.mly"
                                                    ( expecting 2 "enumerated list item" )
# 1467 "src/octParser.ml"
               : 'shortcut_enum))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 374 "src/octParser.mly"
                                                        ( [[]] )
# 1474 "src/octParser.ml"
               : 'shortcut_list_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 375 "src/octParser.mly"
                                                        ( [inner (List.rev _2)] )
# 1482 "src/octParser.ml"
               : 'shortcut_list_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list_final) in
    Obj.repr(
# 376 "src/octParser.mly"
                                                        ( [] :: _3 )
# 1490 "src/octParser.ml"
               : 'shortcut_list_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_list_final) in
    Obj.repr(
# 377 "src/octParser.mly"
                                                        ( (inner (List.rev _2)) :: _4 )
# 1499 "src/octParser.ml"
               : 'shortcut_list_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 381 "src/octParser.mly"
                                                        ( [[]] )
# 1506 "src/octParser.ml"
               : 'shortcut_enum_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'whitespace) in
    Obj.repr(
# 382 "src/octParser.mly"
                                                        ( [inner (List.rev _2)] )
# 1514 "src/octParser.ml"
               : 'shortcut_enum_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum_final) in
    Obj.repr(
# 383 "src/octParser.mly"
                                                        ( [] :: _3 )
# 1522 "src/octParser.ml"
               : 'shortcut_enum_final))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'shortcut_text_body) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'newline) in
    let _4 = (Parsing.peek_val __caml_parser_env 0 : 'shortcut_enum_final) in
    Obj.repr(
# 384 "src/octParser.mly"
                                                        ( (inner (List.rev _2)) :: _4 )
# 1531 "src/octParser.ml"
               : 'shortcut_enum_final))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : int * string option) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 391 "src/octParser.mly"
    ( let n, l = _1 in
        Title (n, l, (inner _2)) )
# 1540 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : int * string option) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 394 "src/octParser.mly"
    ( unclosed (title_to_string _1) 1 "text" "}" 3 )
# 1548 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Types.style_kind) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 396 "src/octParser.mly"
    ( Style(_1, (inner _2)) )
# 1556 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : Types.style_kind) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 398 "src/octParser.mly"
    ( unclosed (style_to_string _1) 1 "text" "}" 3 )
# 1564 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'list) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 400 "src/octParser.mly"
    ( List (List.rev _3) )
# 1573 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'list) in
    Obj.repr(
# 402 "src/octParser.mly"
    ( unclosed "{ul" 1 "list item" "}" 4 )
# 1581 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 404 "src/octParser.mly"
    ( expecting 3 "list item" )
# 1588 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'list) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 406 "src/octParser.mly"
    ( Enum (List.rev _3) )
# 1597 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'list) in
    Obj.repr(
# 408 "src/octParser.mly"
    ( unclosed "{ol" 1 "list item" "}" 4 )
# 1605 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 410 "src/octParser.mly"
    ( expecting 3 "enumerated list item" )
# 1612 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Types.ref_kind * string) in
    Obj.repr(
# 412 "src/octParser.mly"
    ( let k, n = _1 in
        Ref (k, n, None) )
# 1620 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Types.ref_kind * string) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 415 "src/octParser.mly"
    ( let k, n = _2 in
        Ref (k, n, Some (inner _3)) )
# 1629 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _2 = (Parsing.peek_val __caml_parser_env 2 : Types.ref_kind * string) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 418 "src/octParser.mly"
    ( unclosed "{" 1 "text" "}" 3 )
# 1637 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : Types.special_ref_kind) in
    Obj.repr(
# 420 "src/octParser.mly"
    ( Special_ref _1 )
# 1644 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 422 "src/octParser.mly"
    ( Code _1 )
# 1651 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 424 "src/octParser.mly"
    ( PreCode _1 )
# 1658 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string) in
    Obj.repr(
# 426 "src/octParser.mly"
    ( Verbatim _1 )
# 1665 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : string option * string) in
    Obj.repr(
# 428 "src/octParser.mly"
    ( let t, s = _1 in
        Target (t, s) )
# 1673 "src/octParser.ml"
               : 'text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'item) in
    Obj.repr(
# 435 "src/octParser.mly"
                         ( [ _1 ] )
# 1680 "src/octParser.ml"
               : 'list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'list) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'item) in
    Obj.repr(
# 436 "src/octParser.mly"
                         ( _3 :: _1 )
# 1689 "src/octParser.ml"
               : 'list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : bool) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 440 "src/octParser.mly"
                      ( inner _2 )
# 1697 "src/octParser.ml"
               : 'item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : bool) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 441 "src/octParser.mly"
                      ( unclosed (item_to_string _1) 1 "text" "}" 3 )
# 1705 "src/octParser.ml"
               : 'item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * int) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : int) in
    Obj.repr(
# 448 "src/octParser.mly"
    ( let _, n = _1 in
      if n <> _3 then raise Parse_error;
      Title(n, None, (inner _2)) )
# 1716 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string * int) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 452 "src/octParser.mly"
    ( let tag, _ = _1 in
      unclosed (html_open_to_string tag) 1
        "text" (html_close_to_string tag) 3 )
# 1726 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 456 "src/octParser.mly"
    ( Style(SK_bold, (inner _2)) )
# 1734 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 458 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1743 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 461 "src/octParser.mly"
    ( Style(SK_italic, (inner _2)) )
# 1751 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 463 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1760 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 466 "src/octParser.mly"
    ( Style(SK_center, (inner _2)) )
# 1768 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 468 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1777 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 471 "src/octParser.mly"
    ( Style(SK_left, (inner _2)) )
# 1785 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 473 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1794 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 476 "src/octParser.mly"
    ( Style(SK_right, (inner _2)) )
# 1802 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 478 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1811 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'html_list) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 481 "src/octParser.mly"
    ( List (List.rev _3) )
# 1821 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'html_list) in
    Obj.repr(
# 483 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
         "HTML list item" (html_close_to_string _1) 4 )
# 1831 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 486 "src/octParser.mly"
    ( expecting 2 "HTML list item" )
# 1839 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 4 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 3 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 2 : 'html_list) in
    let _4 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 488 "src/octParser.mly"
    ( Enum (List.rev _3) )
# 1849 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 3 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 2 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 1 : 'html_list) in
    Obj.repr(
# 490 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "HTML list item" (html_close_to_string _1) 4 )
# 1859 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    Obj.repr(
# 493 "src/octParser.mly"
    ( expecting 3 "HTML list item" )
# 1867 "src/octParser.ml"
               : 'html_text_element))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 0 : 'html_item) in
    Obj.repr(
# 499 "src/octParser.mly"
                                     ( [ _1 ] )
# 1874 "src/octParser.ml"
               : 'html_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : 'html_list) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'whitespace) in
    let _3 = (Parsing.peek_val __caml_parser_env 0 : 'html_item) in
    Obj.repr(
# 500 "src/octParser.mly"
                                     ( _3 :: _1 )
# 1883 "src/octParser.ml"
               : 'html_list))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 505 "src/octParser.mly"
    ( inner _2 )
# 1891 "src/octParser.ml"
               : 'html_item))
; (fun __caml_parser_env ->
    let _1 = (Parsing.peek_val __caml_parser_env 2 : string) in
    let _2 = (Parsing.peek_val __caml_parser_env 1 : 'text) in
    Obj.repr(
# 507 "src/octParser.mly"
    ( unclosed (html_open_to_string _1) 1
        "text" (html_close_to_string _1) 3 )
# 1900 "src/octParser.ml"
               : 'html_item))
(* Entry main *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
(* Entry reference_parts *)
; (fun __caml_parser_env -> raise (Parsing.YYexit (Parsing.peek_val __caml_parser_env 0)))
|]
let yytables =
  { Parsing.actions=yyact;
    Parsing.transl_const=yytransl_const;
    Parsing.transl_block=yytransl_block;
    Parsing.lhs=yylhs;
    Parsing.len=yylen;
    Parsing.defred=yydefred;
    Parsing.dgoto=yydgoto;
    Parsing.sindex=yysindex;
    Parsing.rindex=yyrindex;
    Parsing.gindex=yygindex;
    Parsing.tablesize=yytablesize;
    Parsing.table=yytable;
    Parsing.check=yycheck;
    Parsing.error_function=parse_error;
    Parsing.names_const=yynames_const;
    Parsing.names_block=yynames_block }
let main (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 1 lexfun lexbuf : Types.t)
let reference_parts (lexfun : Lexing.lexbuf -> token) (lexbuf : Lexing.lexbuf) =
   (Parsing.yyparse yytables 2 lexfun lexbuf : (string option * string) list)
;;
