/*
 *                 (C) Copyright 2005 Nilo J. Gonzalez
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser Gereral Public Licence as published by the Free
 * Software Foundation; either version 2 of the Licence, or (at your opinion) any
 * later version.
 * 
 * This library is distributed in the hope that it will be usefull, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of merchantability or fitness for a
 * particular purpose. See the GNU Lesser General Public Licence for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public Licence along
 * with this library; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, Ma 02111-1307 USA.
 *
 * http://www.gnu.org/licenses/lgpl.html (English)
 * http://gugs.sindominio.net/gnu-gpl/lgpl-es.html (Espa�ol)
 *
 *
 * Original author: Nilo J. Gonzalez
 */

/**
 * Esta clase implementa los colores por defecto del NimRODLookAndFeel.
 * @author Nilo J. Gonzalez
 */

package com.nilo.plaf.nimrod;

import java.awt.Color;
import java.awt.Font;
import javax.swing.plaf.ColorUIResource;
import javax.swing.plaf.FontUIResource;
import javax.swing.plaf.metal.DefaultMetalTheme;

/**
 * Define un <I>tema</I> de color para el NimRODLookAndFeel. En realidad, valen
 * para cualquier Look&Feel que herede de MetalLookAndFeel.<BR>
 * Se usa asi:
 * 
 * <PRE>
 * NimRODLookAndFeel NimRODLF = new NimRODLookAndFeel();
 * NimRODLF.setCurrentTheme(new NimRODTheme());
 * UIManager.setLookAndFeel(NimRODLF);
 * </PRE>
 * 
 * Con esto se pone un color gris oscuro. Tambien define temas partiendo de un
 * color base, modificando los valores primarios.
 * 
 * <PRE>
 * NimRODLookAndFeel NimRODLF = new NimRODLookAndFeel();
 * NimRODLF.setCurrentTheme( new NimRODTheme( &lt;I&gt;unColor&lt;/I&gt;));
 * UIManager.setLookAndFeel(NimRODLF);
 * </PRE>
 * 
 * o partiendo de dos colores base, uno para los valores primarios y otro para
 * los secundarios.
 * 
 * <PRE>
 * NimRODLookAndFeel NimRODLF = new NimRODLookAndFeel();
 * NimRODLF.setCurrentTheme( new NimRODTheme( &lt;I&gt;unColorPrimario&lt;/I&gt;, &lt;I&gt;unColorSecundario&lt;/I&gt;));
 * UIManager.setLookAndFeel(NimRODLF);
 * </PRE>
 * 
 * Para entender como va el temita de los colores, puede ayudar mucho consultar
 * esta pagina: <a
 * href='http://java.sun.com/products/jlf/ed1/dg/higg.htm'>http://java.sun.com/products/jlf/ed1/dg/higg.htm</a>
 */
public class NimRODTheme extends DefaultMetalTheme {
	public static final int DEFAULT_MENU_OPACITY = 195;
	public static final int DEFAULT_FRAME_OPACITY = 180;

	// primarios
	private ColorUIResource primary1 = new ColorUIResource(229, 189, 0);

	private ColorUIResource primary2 = new ColorUIResource(239, 199, 0);

	private ColorUIResource primary3 = new ColorUIResource(249, 209, 0);

	// secondarios
	private ColorUIResource secondary1 = new ColorUIResource(217, 215, 173);

	private ColorUIResource secondary2 = new ColorUIResource(227, 225, 183);

	private ColorUIResource secondary3 = new ColorUIResource(237, 235, 193);

	private ColorUIResource black = new ColorUIResource(0, 0, 0);

	private ColorUIResource white = new ColorUIResource(255, 255, 255);

	// la fuente
	private FontUIResource font = new FontUIResource("SansSerif", Font.PLAIN,
			12);
	private FontUIResource boldFont = new FontUIResource("SansSerif",
			Font.BOLD, 12);

	// la opacidadMenu de los menus
	private int opacidadMenu = DEFAULT_MENU_OPACITY;

	// la opacidadMenu de los InternalFrames
	private int opacidadFrame = DEFAULT_FRAME_OPACITY;

	public NimRODTheme() {
		super();
	}

	/**
	 * Este constructor recibe por parametro el color que se desea utilizar como
	 * color principal de "fondo". Es el color que se usara como fondo de los
	 * botones, dialogos, menus... El resto de los colores de fondo se calculan
	 * oscureciendo este en diversa medida.
	 * 
	 * @param base
	 *            Color el color de fondo.
	 */
	public NimRODTheme(Color base) {
		super();

		setPrimary(base);
	}

	/**
	 * Este constructor recibe por parametro los colores que se desea utilizar.
	 * Base es el color que se usara como fondo de los botones, dialogos,
	 * menus... y prim es el color que se usara para los objetos seleccionados.
	 * En palabras de Sun, Prim es el color que da "personalidad" al tema... El
	 * resto de los colores se calculan oscureciendo estos en diversa medida.
	 * 
	 * @param prim
	 *            Color el color a usar en las selecciones.
	 * @param base
	 *            Color el color de fondo.
	 */
	public NimRODTheme(Color prim, Color sec) {
		super();

		setPrimary(prim);
		setSecondary(sec);
	}

	public String getName() {
		return "NimROD Theme";
	}

	protected ColorUIResource getPrimary1() {
		return primary1;
	}

	protected ColorUIResource getPrimary2() {
		return primary2;
	}

	protected ColorUIResource getPrimary3() {
		return primary3;
	}

	protected ColorUIResource getSecondary1() {
		return secondary1;
	}

	protected ColorUIResource getSecondary2() {
		return secondary2;
	}

	protected ColorUIResource getSecondary3() {
		return secondary3;
	}

	protected ColorUIResource getBlack() {
		return black;
	}

	protected ColorUIResource getWhite() {
		return white;
	}

	public void setPrimary(Color selection) {
		int r = selection.getRed();
		int g = selection.getGreen();
		int b = selection.getBlue();

		primary1 = new ColorUIResource(new Color((r > 20 ? r - 20 : 0),
				(g > 20 ? g - 20 : 0), (b > 20 ? b - 20 : 0)));
		primary2 = new ColorUIResource(new Color((r > 10 ? r - 10 : 0),
				(g > 10 ? g - 10 : 0), (b > 10 ? b - 10 : 0)));
		primary3 = new ColorUIResource(selection);
	}

	public void setSecondary(Color background) {
		int r = background.getRed();
		int g = background.getGreen();
		int b = background.getBlue();

		secondary1 = new ColorUIResource(new Color((r > 20 ? r - 20 : 0),
				(g > 20 ? g - 20 : 0), (b > 20 ? b - 20 : 0)));
		secondary2 = new ColorUIResource(new Color((r > 10 ? r - 10 : 0),
				(g > 10 ? g - 10 : 0), (b > 10 ? b - 10 : 0)));
		secondary3 = new ColorUIResource(background);
	}

	public void setPrimary1(Color col) {
		primary1 = new ColorUIResource(col);
	}

	public void setPrimary2(Color col) {
		primary2 = new ColorUIResource(col);
	}

	public void setPrimary3(Color col) {
		primary3 = new ColorUIResource(col);
	}

	public void setSecondary1(Color col) {
		secondary1 = new ColorUIResource(col);
	}

	public void setSecondary2(Color col) {
		secondary2 = new ColorUIResource(col);
	}

	public void setSecondary3(Color col) {
		secondary3 = new ColorUIResource(col);
	}

	public void setBlack(Color col) {
		black = new ColorUIResource(col);
	}

	public void setWhite(Color col) {
		white = new ColorUIResource(col);
	}

	public void setOpacity(int val) {
		setMenuOpacity(val);
	}

	public int getOpacity() {
		return getMenuOpacity();
	}

	public void setMenuOpacity(int val) {
		opacidadMenu = val;
	}

	public int getMenuOpacity() {
		return opacidadMenu;
	}

	public void setFrameOpacity(int val) {
		opacidadFrame = val;
	}

	public int getFrameOpacity() {
		return opacidadFrame;
	}

	public FontUIResource getControlTextFont() {
		return font;
	}

	public FontUIResource getMenuTextFont() {
		return font;
	}

	public FontUIResource getSubTextFont() {
		return font;
	}

	public FontUIResource getSystemTextFont() {
		return boldFont;
	}

	public FontUIResource getUserTextFont() {
		return font;
	}

	public FontUIResource getWindowTitleFont() {
		return boldFont;
	}

	public String toString() {
		StringBuffer cad = new StringBuffer();

		cad.append("nimrodlf.p1=" + encode(primary1) + "\n");
		cad.append("nimrodlf.p2=" + encode(primary2) + "\n");
		cad.append("nimrodlf.p3=" + encode(primary3) + "\n");
		cad.append("nimrodlf.s1=" + encode(secondary1) + "\n");
		cad.append("nimrodlf.s2=" + encode(secondary2) + "\n");
		cad.append("nimrodlf.s3=" + encode(secondary3) + "\n");

		cad.append("nimrodlf.w=" + encode(white) + "\n");
		cad.append("nimrodlf.b=" + encode(black) + "\n");
		cad.append("nimrodlf.menuOpacity=" + opacidadMenu + "\n");
		cad.append("nimrodlf.frameOpacity=" + opacidadFrame + "\n");

		return cad.toString();
	}

	protected String encode(Color col) {
		String r = Integer.toHexString(col.getRed()).toUpperCase();
		String g = Integer.toHexString(col.getGreen()).toUpperCase();
		String b = Integer.toHexString(col.getBlue()).toUpperCase();

		return "#" + (r.length() == 1 ? "0" + r : r)
				+ (g.length() == 1 ? "0" + g : g)
				+ (b.length() == 1 ? "0" + b : b);
	}
}
