/*
 *         Copyright (C) 2008 Fadi Al-katout (cutout) <cutout33@gmail.com>
 * 
 *  Real Media Converter is free software; you can redistribute it and/or modify   
 * it under the terms of the GNU General  Public Licence as published by the 
 * Free Software Foundation;  either version 2 of the Licence, 
 * or(at your option) any later version.
 * 
 * Real Media Converter is distributed in  the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;  without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A  PARTICULAR PURPOSE. 
 * See the GNU General  Public Licence for more details.
 * 
 * You should have received a copy of  the GNU General Public Licence
 * along with Real Media Converter; 
 * if not, write to  the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor,  Boston, MA  02110-1301, USA
 */

package net.dudesblog.rmc.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;

import net.dudesblog.rmc.util.constants.Constants;

import org.apache.log4j.PropertyConfigurator;

public final class FileManager {

	public static void initializeTempFolder() {
		if (!Constants.FileConstants.RMC_USER_FOLDER.exists()) {
			Constants.FileConstants.RMC_USER_FOLDER.mkdir();
		}
		if (!Constants.FileConstants.RMC_TEMP_FOLDER.exists()) {
			Constants.FileConstants.RMC_TEMP_FOLDER.mkdir();
		} else {
			File[] files = Constants.FileConstants.RMC_TEMP_FOLDER.listFiles();
			for (File file : files) {
				file.delete();
			}
		}
		initLog4jOutputPath();
	}

	private static void initLog4jOutputPath() {
		String dynamicLog = Constants.FileConstants.RMC_USER_FOLDER
				.getAbsolutePath();
		Properties properties = new Properties();
		try {
			properties.load(new FileInputStream("log4j.properties"));
		} catch (Exception e) {
			e.printStackTrace();
		}
		properties.put("log.dir", dynamicLog); // overwrite "log.dir"
		PropertyConfigurator.configure(properties);

	}

	public static File addToTmpFolder(String fileName) {
		return new File(Constants.FileConstants.RMC_TEMP_FOLDER, fileName);
	}

	public static void copyFile(File fromFile, File toFile) throws IOException {
		String fromFileName = fromFile.getAbsolutePath();
		if (!fromFile.exists())
			throw new IOException("FileCopy: " + "no such source file: "
					+ fromFileName);
		if (!fromFile.isFile())
			throw new IOException("FileCopy: " + "can't copy directory: "
					+ fromFileName);
		if (!fromFile.canRead())
			throw new IOException("FileCopy: " + "source file is unreadable: "
					+ fromFileName);
		if (toFile.isDirectory())
			toFile = new File(toFile, fromFile.getName());
		String parent = toFile.getParent();
		if (parent == null)
			parent = System.getProperty("user.dir");
		File dir = new File(parent);
		if (!dir.exists())
			throw new IOException("FileCopy: "
					+ "destination directory doesn't exist: " + parent);
		if (dir.isFile())
			throw new IOException("FileCopy: "
					+ "destination is not a directory: " + parent);
		if (!dir.canWrite())
			throw new IOException("FileCopy: "
					+ "destination directory is unwriteable: " + parent);
		FileInputStream from = null;
		FileOutputStream to = null;
		try {
			from = new FileInputStream(fromFile);
			to = new FileOutputStream(toFile);
			byte[] buffer = new byte[4096];
			int bytesRead;
			while ((bytesRead = from.read(buffer)) != -1)
				to.write(buffer, 0, bytesRead); // write
		} finally {
			if (from != null)
				try {
					from.close();
				} catch (IOException e) {
					;
				}
			if (to != null)
				try {
					to.close();
				} catch (IOException e) {
					;
				}
		}
	}

	public static boolean isValidFile(File file) throws IOException {
		if (!file.exists())
			throw new IOException("File Validation : "
					+ "no such source file: " + file.getAbsolutePath());
		if (!file.isFile())
			throw new IOException("File Validation : "
					+ "can't copy directory: " + file.getAbsolutePath());
		if (!file.canRead())
			throw new IOException("File Validation : "
					+ "source file is unreadable: " + file.getAbsolutePath());
		if (!file.getName().endsWith(".ra") && !file.getName().endsWith(".rm")
				&& !file.getName().endsWith(".ram")
				&& !file.getName().endsWith(".wav")
				&& !file.getName().endsWith(".RA")
				&& !file.getName().endsWith(".RM")
				&& !file.getName().endsWith(".RAM")
				&& !file.getName().endsWith(".WAV"))
			throw new IOException("File Validation : "
					+ "source file is not a real media file : "
					+ file.getAbsolutePath());
		return true;
	}

}
