"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const ansi = require("ansi-colors");
const execa = require("execa");
const fs_extra_1 = require("fs-extra");
const Listr = require("listr");
const path = require("path");
const che_1 = require("../../api/che");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
function createNamespaceTask(flags) {
    return {
        title: `Create Namespace (${flags.chenamespace})`,
        task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const exist = yield che.cheNamespaceExist(flags.chenamespace);
            if (exist) {
                task.title = `${task.title}...It already exists.`;
            }
            else if (util_1.isKubernetesPlatformFamily(flags.platform)) {
                yield execa(`kubectl create namespace ${flags.chenamespace}`, { shell: true });
                task.title = `${task.title}...done.`;
            }
            else if (util_1.isOpenshiftPlatformFamily(flags.platform)) {
                yield execa(`oc new-project ${flags.chenamespace}`, { shell: true });
                task.title = `${task.title}...done.`;
            }
        })
    };
}
exports.createNamespaceTask = createNamespaceTask;
function copyOperatorResources(flags, cacheDir) {
    return {
        title: 'Copying operator resources',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            ctx.resourcesPath = yield copyCheOperatorResources(flags.templates, cacheDir);
            task.title = `${task.title}...done.`;
        })
    };
}
exports.copyOperatorResources = copyOperatorResources;
function copyCheOperatorResources(templatesDir, cacheDir) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const srcDir = path.join(templatesDir, '/codeready-workspaces-operator/');
        const destDir = path.join(cacheDir, '/templates/codeready-workspaces-operator/');
        yield fs_extra_1.remove(destDir);
        yield fs_extra_1.mkdirp(destDir);
        yield fs_extra_1.copy(srcDir, destDir);
        return destDir;
    });
}
function createEclipseCheCluster(flags, kube) {
    return {
        title: `Create the Custom Resource of type ${constants_1.CHE_CLUSTER_CRD} in the namespace ${flags.chenamespace}`,
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCluster = yield kube.getCheCluster(flags.chenamespace);
            if (cheCluster) {
                task.title = `${task.title}...It already exists.`;
            }
            else {
                // CodeReady Workspaces operator supports only Multi-User Che
                ctx.isCheDeployed = true;
                ctx.isPostgresDeployed = true;
                ctx.isKeycloakDeployed = true;
                // plugin and devfile registry will be deployed only when external ones are not configured
                ctx.isPluginRegistryDeployed = !flags['plugin-registry-url'];
                ctx.isDevfileRegistryDeployed = !flags['devfile-registry-url'];
                const yamlFilePath = flags['che-operator-cr-yaml'] === '' ? ctx.resourcesPath + 'crds/org_v1_che_cr.yaml' : flags['che-operator-cr-yaml'];
                const cr = yield kube.createCheClusterFromFile(yamlFilePath, flags, ctx, flags['che-operator-cr-yaml'] === '');
                ctx.cr = cr;
                ctx.isKeycloakReady = ctx.isKeycloakReady || cr.spec.auth.externalIdentityProvider;
                ctx.isPostgresReady = ctx.isPostgresReady || cr.spec.database.externalDb;
                ctx.isDevfileRegistryReady = ctx.isDevfileRegistryReady || cr.spec.server.externalDevfileRegistry;
                ctx.isPluginRegistryReady = ctx.isPluginRegistryReady || cr.spec.server.externalPluginRegistry;
                if (cr.spec.server.customCheProperties && cr.spec.server.customCheProperties.CHE_MULTIUSER === 'false') {
                    flags.multiuser = false;
                }
                if (cr.spec.auth && cr.spec.auth.updateAdminPassword) {
                    ctx.highlightedMessages.push('CodeReady Workspaces admin credentials are: "admin:admin". You will be asked to change default Che admin password on the first login.');
                }
                task.title = `${task.title}...done.`;
            }
        })
    };
}
exports.createEclipseCheCluster = createEclipseCheCluster;
function retrieveCheCaCertificateTask(flags) {
    return {
        title: 'Retrieving Che self-signed CA certificate',
        // It makes sense to retrieve CA certificate only if self-signed certificate is used.
        enabled: () => flags.tls && flags.installer !== 'helm',
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const cheCaCert = yield che.retrieveCheCaCert(flags.chenamespace);
            if (cheCaCert) {
                const targetFile = yield che.saveCheCaCert(cheCaCert);
                task.title = `${task.title}... is exported to ${targetFile}`;
                ctx.highlightedMessages.push(getMessageImportCaCertIntoBrowser(targetFile));
            }
            else {
                task.title = `${task.title}... commonly trusted certificate is used.`;
            }
        })
    };
}
exports.retrieveCheCaCertificateTask = retrieveCheCaCertificateTask;
function getMessageImportCaCertIntoBrowser(caCertFileLocation) {
    const message = `❗${ansi.yellow('[MANUAL ACTION REQUIRED]')} Please add Che self-signed CA certificate into your browser: ${caCertFileLocation}.\n` +
        `Documentation how to add a CA certificate into a browser: ${constants_1.DOCS_LINK_IMPORT_CA_CERT_INTO_BROWSER}`;
    return message;
}
exports.getMessageImportCaCertIntoBrowser = getMessageImportCaCertIntoBrowser;
function getRetrieveKeycloakCredentialsTask(flags) {
    return {
        title: 'Retrieving Keycloak admin credentials',
        enabled: (ctx) => ctx.cr && !ctx.cr.spec.auth.externalIdentityProvider && flags.multiuser && (flags.installer !== 'operator' || flags.installer !== 'olm'),
        task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const che = new che_1.CheHelper(flags);
            const [login, password] = yield che.retrieveKeycloakAdminCredentials(flags.chenamespace);
            if (login && password) {
                ctx.highlightedMessages.push(`Autogenerated Keycloak credentials are: "${login}:${password}".`);
                task.title = `${task.title}... ${login}:${password}`;
            }
            else {
                task.title = `${task.title}... Failed.`;
            }
        })
    };
}
exports.getRetrieveKeycloakCredentialsTask = getRetrieveKeycloakCredentialsTask;
/**
 * Prints important to user messages which are stored in ctx.highlightedMessages
 * Typically this task is the last task of a command.
 */
function getPrintHighlightedMessagesTask() {
    return {
        title: 'Show important messages',
        enabled: ctx => ctx.highlightedMessages && ctx.highlightedMessages.length > 0,
        task: (ctx) => {
            const printMessageTasks = new Listr([], ctx.listrOptions);
            for (const message of ctx.highlightedMessages) {
                printMessageTasks.add({
                    title: message,
                    task: () => { }
                });
            }
            return printMessageTasks;
        }
    };
}
exports.getPrintHighlightedMessagesTask = getPrintHighlightedMessagesTask;
//# sourceMappingURL=common-tasks.js.map