/*
 * Copyright (c) 2001, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.MethodExitRequest.addClassFilter_rt;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * MethodExitRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.MethodExitRequest.addClassFilter(ReferenceType)</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to those
 *    whose location is in the given reference type or
 *    any of its subtypes.
 * The cases to check include entering methods in four types,
 * two in super-class filter_rt001aTestClass10 and its sub-class filter_rt001aTestClass11, and
 * two in super-class filter_rt001aTestClass20 and its sub-class filter_rt001aTestClass21.
 * A filter used restricts events only to methods in first pair.
 *
 * The test works as follows.
 * - The debugger resumes the debuggee and waits for the BreakpointEvent.
 * - The debuggee creates a thread, thread1, invoking methods
 *   in the super-class filter_rt001aTestClass10 and its sub-class filter_rt001aTestClass11 and
 *   in the super-class filter_rt001aTestClass20 and its sub-class filter_rt001aTestClass21
 *   and invokes the methodForCommunication to be suspended and
 *   to inform the debugger with the event.
 * - Upon getting the BreakpointEvent, the debugger
 *   - gets ReferenceTypes to use to filter MethodExitEvents,
 *   - sets up MethodExitRequest for the events,
 *   - restricts the events to those in filter_rt001aTestClass11,
 *   - and resumes the debuggee and waits for the events.
 * - The debuggee starts the thread1.
 * - Upon getting the events, the debugger performs checks required.
 *
 * In third phase, at the end of the test, the debuggee        changes
 * the value of the "instruction" which the debugger and debuggee
 * use to inform each other of needed actions,  and both end.
 */

public class filter_rt001 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        int result = run(argv,System.out);
        if (result != 0) {
            throw new RuntimeException("TEST FAILED with result " + result);
        }
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.MethodExitRequest.addClassFilter_rt.filter_rt001a";
        return new filter_rt001().runThis(argv, out);
    }

    private String testedClassName =
      "nsk.jdi.MethodExitRequest.addClassFilter_rt.filter_rt001aTestClass11";

    protected void testRun() {
        EventRequest  eventRequest1      = null;
        String        property1          = "MethodExitRequest1";
        ReferenceType testClassReference = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {
                case 0:
                testClassReference = (ReferenceType) debuggee.classByName(testedClassName);

                eventRequest1 = setting21MethodExitRequest(null, testClassReference,
                                                   EventRequest.SUSPEND_NONE, property1);

                display("......waiting for MethodExitEvent in expected thread");
                Event newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1}, waitTime, false);

                if ( !(newEvent instanceof MethodExitEvent)) {
                    setFailedStatus("ERROR: new event is not MethodExitEvent");
                } else {

                    String property = (String) newEvent.request().getProperty("number");
                    display("       got new MethodExitEvent with propety 'number' == " + property);

                    if ( !property.equals(property1) ) {
                        setFailedStatus("ERROR: property is not : " + property1);
                    }

                    ReferenceType refType = ((MethodExitEvent)newEvent).location().declaringType();
                    if (!refType.equals(testClassReference)) {
                        setFailedStatus("Received unexpected declaring type of the event: " + refType.name() +
                            "\n\texpected one: " + testClassReference.name());
                    }
               }
               break;

               default:
               throw new Failure("** default case 2 **");
           }
        }
        return;
    }

    private MethodExitRequest setting21MethodExitRequest ( ThreadReference thread,
                                                            ReferenceType    testedClass,
                                                            int              suspendPolicy,
                                                            String           property       ) {
        try {
            display("......setting up MethodExitRequest:");
            display("       thread: " + thread + "; class: " + testedClass +  "; property: " + property);

            MethodExitRequest
            menr = eventRManager.createMethodExitRequest();
            menr.putProperty("number", property);
            if (thread != null)
                menr.addThreadFilter(thread);
            menr.addClassFilter(testedClass);
            menr.setSuspendPolicy(suspendPolicy);

            display("      a MethodExitRequest has been set up");
            return menr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up MethodExitRequest **");
        }
    }
}
