/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_3_3d.c                                                */
/*                                                                          */
/* description:  piecewise cubic Lagrange elements in 3d                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS3_3D  20

static const REAL bary3_3d[N_BAS3_3D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
						   {0.0, 1.0, 0.0, 0.0},
						   {0.0, 0.0, 1.0, 0.0},
						   {0.0, 0.0, 0.0, 1.0},
					    {2.0/3.0,1.0/3.0,    0.0,    0.0},
					    {1.0/3.0,2.0/3.0,    0.0,    0.0},
					    {2.0/3.0,    0.0,1.0/3.0,    0.0},
					    {1.0/3.0,    0.0,2.0/3.0,    0.0},
					    {2.0/3.0,    0.0,    0.0,1.0/3.0},
					    {1.0/3.0,    0.0,    0.0,2.0/3.0},
					    {    0.0,2.0/3.0,1.0/3.0,    0.0},
					    {    0.0,1.0/3.0,2.0/3.0,    0.0},
					    {    0.0,2.0/3.0,    0.0,1.0/3.0},
					    {    0.0,1.0/3.0,    0.0,2.0/3.0},
					    {    0.0,    0.0,2.0/3.0,1.0/3.0},
					    {    0.0,    0.0,1.0/3.0,2.0/3.0},
					    {    0.0,1.0/3.0,1.0/3.0,1.0/3.0},
					    {1.0/3.0,    0.0,1.0/3.0,1.0/3.0},
					    {1.0/3.0,1.0/3.0,    0.0,1.0/3.0},
					    {1.0/3.0,1.0/3.0,1.0/3.0,    0.0}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v0_3d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[0] - 1.0)*lambda[0] + 1.0)*lambda[0]);
}

static const REAL *grd_phi3v0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[0] - 9.0)*lambda[0] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v0_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[0] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v1_3d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[1] - 1.0)*lambda[1] + 1.0)*lambda[1]);
}

static const REAL *grd_phi3v1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (13.5*lambda[1] - 9.0)*lambda[1] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v1_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[1] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v2_3d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[2] - 1.0)*lambda[2] + 1.0)*lambda[2]);
}

static const REAL *grd_phi3v2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0, 0.0, 0.0};
  grd[2] = (13.5*lambda[2] - 9.0)*lambda[2] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v2_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 27.0*lambda[2] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 3                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v3_3d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[3] - 1.0)*lambda[3] + 1.0)*lambda[3]);
}

static const REAL *grd_phi3v3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0, 0.0, 0.0};
  grd[3] = (13.5*lambda[3] - 9.0)*lambda[3] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v3_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[3][3] = 27.0*lambda[3] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 0 inbetween vertex 0 and 1                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e0_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[1]);
}

static const REAL *grd_phi3e0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[1];
  grd[1] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e0_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[1];
  D2[0][1] = D2[1][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 0 inbetween vertex 0 and 1                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e1_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[0]);
}

static const REAL *grd_phi3e1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[1] - 4.5)*lambda[1];
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e1_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[0];
  D2[0][1] = D2[1][0] = 27.0*lambda[1] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 1 inbetween vertex 0 and 2                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e2_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[2]);
}

static const REAL *grd_phi3e2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[2];
  grd[2] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e2_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[2];
  D2[0][2] = D2[2][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 1 inbetween vertex 0 and 2                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e3_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[2] - 4.5)*lambda[2]*lambda[0]);
}

static const REAL *grd_phi3e3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[2] - 4.5)*lambda[2];
  grd[2] = (27.0*lambda[2] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e3_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 27.0*lambda[0];
  D2[0][2] = D2[2][0] = 27.0*lambda[2] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 2 inbetween vertex 0 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e4_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[3]);
}

static const REAL *grd_phi3e4_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[3];
  grd[3] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e4_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[3];
  D2[0][3] = D2[3][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 0 inbetween vertex 0 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e5_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[3] - 4.5)*lambda[3]*lambda[0]);
}

static const REAL *grd_phi3e5_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[3] - 4.5)*lambda[3];
  grd[3] = (27.0*lambda[3] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e5_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[3][3] = 27.0*lambda[0];
  D2[0][3] = D2[3][0] = 27.0*lambda[3] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 3 inbetween vertex 1 and 2                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e6_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[2]);
}

static const REAL *grd_phi3e6_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[2];
  grd[2] = (13.5*lambda[1] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e6_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[2];
  D2[1][2] = D2[2][1] = 27.0*lambda[1] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 3 inbetween vertex 1 and 2                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e7_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[2] - 4.5)*lambda[2]*lambda[1]);
}

static const REAL *grd_phi3e7_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (13.5*lambda[2] - 4.5)*lambda[2];
  grd[2] = (27.0*lambda[2] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e7_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 27.0*lambda[1];
  D2[1][2] = D2[2][1] = 27.0*lambda[2] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 4 inbetween vertex 1 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e8_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[3]);
}

static const REAL *grd_phi3e8_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[3];
  grd[3] = (13.5*lambda[1] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e8_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[3];
  D2[1][3] = D2[3][1] = 27.0*lambda[1] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 4 inbetween vertex 1 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e9_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[3] - 4.5)*lambda[3]*lambda[1]);
}

static const REAL *grd_phi3e9_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (13.5*lambda[3] - 4.5)*lambda[3];
  grd[3] = (27.0*lambda[3] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e9_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[3][3] = 27.0*lambda[1];
  D2[1][3] = D2[3][1] = 27.0*lambda[3] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 5 inbetween vertex 2 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e10_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[2] - 4.5)*lambda[2]*lambda[3]);
}

static const REAL *grd_phi3e10_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[2] = (27.0*lambda[2] - 4.5)*lambda[3];
  grd[3] = (13.5*lambda[2] - 4.5)*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e10_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 27.0*lambda[3];
  D2[2][3] = D2[3][2] = 27.0*lambda[2] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 5 inbetween vertex 2 and 3                      */
/*--------------------------------------------------------------------------*/

static REAL phi3e11_3d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[3] - 4.5)*lambda[3]*lambda[2]);
}

static const REAL *grd_phi3e11_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[2] = (13.5*lambda[3] - 4.5)*lambda[3];
  grd[3] = (27.0*lambda[3] - 4.5)*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e11_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[3][3] = 27.0*lambda[2];
  D2[2][3] = D2[3][2] = 27.0*lambda[3] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at barycenter of face 0                                   */
/*--------------------------------------------------------------------------*/

static REAL phi3c0_3d(const REAL lambda[N_LAMBDA])
{
  return(27.0*lambda[1]*lambda[2]*lambda[3]);
}

static const REAL *grd_phi3c0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = 27.0*lambda[2]*lambda[3];
  grd[2] = 27.0*lambda[1]*lambda[3];
  grd[3] = 27.0*lambda[1]*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3c0_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][2] = D2[2][1] = 27.0*lambda[3];
  D2[1][3] = D2[3][1] = 27.0*lambda[2];
  D2[2][3] = D2[3][2] = 27.0*lambda[1];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at barycenter of face 1                                   */
/*--------------------------------------------------------------------------*/

static REAL phi3c1_3d(const REAL lambda[N_LAMBDA])
{
  return(27.0*lambda[0]*lambda[2]*lambda[3]);
}

static const REAL *grd_phi3c1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 27.0*lambda[2]*lambda[3];
  grd[2] = 27.0*lambda[0]*lambda[3];
  grd[3] = 27.0*lambda[0]*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3c1_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][2] = D2[2][0] = 27.0*lambda[3];
  D2[0][3] = D2[3][0] = 27.0*lambda[2];
  D2[2][3] = D2[3][2] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at barycenter of face 2                                   */
/*--------------------------------------------------------------------------*/

static REAL phi3c2_3d(const REAL lambda[N_LAMBDA])
{
  return(27.0*lambda[0]*lambda[1]*lambda[3]);
}

static const REAL *grd_phi3c2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA];
  grd[0] = 27.0*lambda[1]*lambda[3];
  grd[1] = 27.0*lambda[0]*lambda[3];
  grd[3] = 27.0*lambda[0]*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3c2_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 27.0*lambda[3];
  D2[0][3] = D2[3][0] = 27.0*lambda[1];
  D2[1][3] = D2[3][1] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at barycenter of face 3                                   */
/*--------------------------------------------------------------------------*/

static REAL phi3c3_3d(const REAL lambda[N_LAMBDA])
{
  return(27.0*lambda[0]*lambda[1]*lambda[2]);
}

static const REAL *grd_phi3c3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA];
  grd[0] = 27.0*lambda[1]*lambda[2];
  grd[1] = 27.0*lambda[0]*lambda[2];
  grd[2] = 27.0*lambda[0]*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3c3_3d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 27.0*lambda[2];
  D2[0][2] = D2[2][0] = 27.0*lambda[1];
  D2[1][2] = D2[2][1] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *get_dof_indices3_3d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  index_vec[N_BAS3_3D];
  DOF         *rvec = idof ? idof : index_vec;
  int         i, j, n0;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = dof[i][n0];
  n0 = admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      rvec[j++] = dof[N_VERTICES_3D+i][n0];
      rvec[j++] = dof[N_VERTICES_3D+i][n0+1];
    }
    else
    {
      rvec[j++] = dof[N_VERTICES_3D+i][n0+1];
      rvec[j++] = dof[N_VERTICES_3D+i][n0];
    }
  }
  n0 = admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = dof[N_VERTICES_3D+N_EDGES_3D+i][n0];

  return((const DOF *) rvec);
}

static const S_CHAR *get_bound3_3d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound3_3d");
  static S_CHAR  bound_vec[N_BAS3_3D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i, j;

  DEBUG_TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = el_info->vertex_bound[i];
  for (j = N_VERTICES_3D, i = 0; i < N_EDGES_3D; j += 2, i++)
    rvec[j] = rvec[j+1] = el_info->edge_bound[i];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = el_info->face_bound[i];
  
  return((const S_CHAR *) rvec);
}

static const int *get_int_vec3_3d(const EL *el, const DOF_INT_VEC *vec,
				int *ivec)
{
  FUNCNAME("get_int_vec3_3d");
  int         i, j, n0;
  static int  local_vec[N_BAS3_3D];
  int         *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF         **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = v[dof[N_VERTICES_3D+N_EDGES_3D+i][n0]];

  return((const int *) rvec);
}

static const REAL *get_real_vec3_3d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_vec3_3d");
  int            i, j, n0;
  static REAL    local_vec[N_BAS3_3D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = v[dof[N_VERTICES_3D+N_EDGES_3D+i][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec3_3d(const EL *el,
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rvec)
{
  FUNCNAME("get_real_d_vec3_3d");
  int            i, j, k, n0;
  static REAL_D  local_vec[N_BAS3_3D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_3D+i][n0]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_3D+i][n0+1]][k];
      }
      j += 2;
    }
    else
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_3D+i][n0+1]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_3D+i][n0]][k];
      }
      j += 2;
    }
  }
  n0 = vec->fe_space->admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++, j++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[j][k] = v[dof[N_VERTICES_3D+N_EDGES_3D+i][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec3_3d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec3_3d");
  int            i, j, n0;
  static U_CHAR  local_vec[N_BAS3_3D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = v[dof[N_VERTICES_3D+N_EDGES_3D+i][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec3_3d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec3_3d");
  int            i, j, n0;
  static S_CHAR  local_vec[N_BAS3_3D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_3D; i < N_EDGES_3D; i++)
  {
    if (dof[vertex_of_edge_3d[i][0]][0] < dof[vertex_of_edge_3d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_3D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[FACE];
  for (i = 0; i < N_FACES_3D; i++)
    rvec[j++] = v[dof[N_VERTICES_3D+N_EDGES_3D+i][n0]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,3,3,20)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,3,3,20)


/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void real_refine_inter3_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter3_3d");
  EL              *el;
  REAL            *v = nil;
  const DOF       *cd;
  DOF              pd[N_BAS3_3D], cdi;
  int              i, lr_set, node0, n0;
  U_CHAR           typ;
  const DOF     *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;
  typ = list->el_info.el_type;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  v[cd[3]] = (0.0625*(-v[pd[0]] - v[pd[1]]) + 0.5625*(v[pd[4]] + v[pd[5]]));
  v[cd[8]] = (0.3125*(v[pd[0]] - v[pd[5]]) + 0.0625*v[pd[1]]
	      + 0.937500*v[pd[4]]);
  v[cd[9]] = v[pd[4]];
  v[cd[12]] = (0.0625*(v[pd[0]] + v[pd[1]] - v[pd[4]] - v[pd[5]])
	       + 0.25*(-v[pd[6]] - v[pd[10]])
	       + 0.5*(v[pd[7]] + v[pd[11]] + v[pd[19]]));
  v[cd[13]] = v[pd[19]];
  v[cd[14]] = (0.0625*(v[pd[0]] + v[pd[1]] - v[pd[4]] - v[pd[5]])
	       + 0.25*(-v[pd[8]] - v[pd[12]]) 
	       + 0.5*(v[pd[9]] + v[pd[13]] + v[pd[18]]));
  v[cd[15]] = (v[pd[18]]);
  v[cd[16]] = (0.0625*(v[pd[0]] + v[pd[1]] - v[pd[4]] - v[pd[5]])
	       + 0.125*(-v[pd[6]]-v[pd[8]]-v[pd[10]]-v[pd[12]])
	       + 0.5*(v[pd[16]] + v[pd[17]]) 
	       + 0.25*(v[pd[18]] + v[pd[19]]));
  v[cd[17]] = (0.0625*(-v[pd[0]] + v[pd[1]]) 
	       + 0.1875*(v[pd[4]] - v[pd[5]]) + 0.375*v[pd[8]]
	       - 0.125*v[pd[12]] + 0.75*v[pd[18]]);
  v[cd[18]] = (0.0625*(-v[pd[0]] + v[pd[1]]) 
	       + 0.1875*(v[pd[4]] - v[pd[5]]) + 0.375*v[pd[6]]
	       - 0.125*v[pd[10]] + 0.75*v[pd[19]]);

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  if (typ == 0)
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    v[cd[8]] = (0.0625*v[pd[0]] + 0.3125*(v[pd[1]] - v[pd[4]])
		+ 0.9375*v[pd[5]]);
    v[cd[9]] = v[pd[5]];
    v[cd[17]] = (0.0625*(v[pd[0]] - v[pd[1]]) 
		 + 0.1875*(-v[pd[4]] + v[pd[5]])
		 - 0.125*v[pd[6]] + 0.375*v[pd[10]] + 0.75*v[pd[19]]);
    v[cd[18]] = (0.0625*(v[pd[0]] - v[pd[1]]) 
		 + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[8]] 
		 + 0.375*v[pd[12]] + 0.75*v[pd[18]]);
  }
  else
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    v[cd[8]] = (0.0625*v[pd[0]] + 0.3125*(v[pd[1]] - v[pd[4]])
		+ 0.9375*v[pd[5]]);
    v[cd[9]] = v[pd[5]];
    v[cd[17]] = (0.0625*(v[pd[0]] - v[pd[1]]) 
		 + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[8]]
		 + 0.375*v[pd[12]] + 0.75*v[pd[18]]);
    v[cd[18]] = (0.0625*(v[pd[0]] - v[pd[1]]) + 
		 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[6]]
		 + 0.375*v[pd[10]] + 0.75*v[pd[19]]);
  }

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  node0 = drv->fe_space->mesh->node[FACE];
  n0 = admin->n0_dof[FACE];
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    typ = list[i].el_info.el_type;
    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[0], admin, nil);

    switch(lr_set)
    {
    case 1:
      v[cd[12]] = (0.0625*(v[pd[0]]+v[pd[1]]-v[pd[4]]-v[pd[5]])
		   + 0.25*(-v[pd[6]] - v[pd[10]]) 
		   + 0.5*(v[pd[7]] + v[pd[11]] + v[pd[19]]));
      v[cd[13]] = v[pd[19]];
      v[cd[16]] = (0.0625*(v[pd[0]]+v[pd[1]]-v[pd[4]]-v[pd[5]])
		   + 0.125*(-v[pd[6]]-v[pd[8]]-v[pd[10]]-v[pd[12]])
		   + 0.5*(v[pd[16]] + v[pd[17]]) 
		   + 0.25*(v[pd[18]] + v[pd[19]]));
      v[cd[18]] = (0.0625*(-v[pd[0]] + v[pd[1]])
		   + 0.1875*(v[pd[4]] - v[pd[5]]) + 0.375*v[pd[6]]
		   - 0.125*v[pd[10]] + 0.75*v[pd[19]]);
      break;
    case 2:
      v[cd[14]] = (0.0625*(v[pd[0]]+v[pd[1]]-v[pd[4]]-v[pd[5]])
		   + 0.25*(-v[pd[8]] - v[pd[12]]) 
		   + 0.5*(v[pd[9]] + v[pd[13]] + v[pd[18]]));
      v[cd[15]] = v[pd[18]];
      v[cd[16]] = (0.0625*(v[pd[0]]+v[pd[1]]-v[pd[4]]-v[pd[5]])
		   + 0.125*(-v[pd[6]]-v[pd[8]]-v[pd[10]]-v[pd[12]])
		   + 0.5*(v[pd[16]] + v[pd[17]])
		   + 0.25*(v[pd[18]] + v[pd[19]]));
      v[cd[17]] = (0.0625*(-v[pd[0]] + v[pd[1]])
		   + 0.1875*(v[pd[4]] - v[pd[5]]) + 0.375*v[pd[8]]
		   - 0.125*v[pd[12]] + 0.75*v[pd[18]]);
      break;
   case 3:
      v[cd[16]] = (0.0625*(v[pd[0]]+v[pd[1]]-v[pd[4]]-v[pd[5]])
		   + 0.125*(-v[pd[6]]-v[pd[8]]-v[pd[10]]-v[pd[12]])
		   + 0.5*(v[pd[16]] + v[pd[17]])
		   + 0.25*(v[pd[18]] + v[pd[19]]));
      break;
    }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[1], admin, nil);

    if (typ == 0)
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	v[cdi] = (0.0625*(v[pd[0]] - v[pd[1]])
		  + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[6]]
		  + 0.375*v[pd[10]] + 0.75*v[pd[19]]);
	break;
      case 2:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	v[cdi] = (0.0625*(v[pd[0]] - v[pd[1]])
		  + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[8]]
		  + 0.375*v[pd[12]] + 0.75*v[pd[18]]);
	break;
      }
    }
    else
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	v[cdi] = (0.0625*(v[pd[0]] - v[pd[1]])
		  + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[6]]
		  + 0.375*v[pd[10]] + 0.75*v[pd[19]]);
	break;
      case 2:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	v[cdi] = (0.0625*(v[pd[0]] - v[pd[1]])
		  + 0.1875*(-v[pd[4]] + v[pd[5]]) - 0.125*v[pd[8]]
		  + 0.375*v[pd[12]] + 0.75*v[pd[18]]);
	break;
      }
    }
  }

  return;
}

static void real_coarse_inter3_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter3_3d");
  EL        *el;
  REAL      *v = nil;
  int       i, lr_set;
  int       node_e, node_f, n0_e, n0_f;
  DOF       **pds, **cds;
  DOF       pd_o[N_BAS3_3D], cd, pd;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  node_e = drv->fe_space->mesh->node[EDGE];        
  n0_e = admin->n0_dof[EDGE];
  node_f = drv->fe_space->mesh->node[FACE];        
  n0_f = admin->n0_dof[FACE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  pds = el->dof;
  cds = el->child[0]->dof;
  get_dof_indices(el, admin, pd_o);

  pd = (pds[0][0] < pds[1][0]) ? pds[node_e][n0_e] : pds[node_e][n0_e+1];
  cd = cds[0][0] < cds[3][0] ? cds[node_e+2][n0_e+1] : cds[node_e+2][n0_e];
  v[pd] = v[cd];

  pd = el->dof[node_f+2][n0_f];
  cd = cds[2][0] < cds[3][0] ? cds[node_e+5][n0_e+1] : cds[node_e+5][n0_e];
  v[pd] = v[cd];

  pd = el->dof[node_f+3][n0_f]; 
  cd = cds[1][0] < cds[3][0] ? cds[node_e+4][n0_e+1] : cds[node_e+4][n0_e];
  v[pd] = v[cd];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cds = el->child[1]->dof;

  pd = (pds[0][0] < pds[1][0]) ? pds[node_e][n0_e+1] : pds[node_e][n0_e];
  cd = cds[0][0] < cds[3][0] ? cds[node_e+2][n0_e+1] : cds[node_e+2][n0_e];
  v[pd] = v[cd];

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;

    pds = el->dof;
    cds = el->child[0]->dof;

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

    switch(lr_set)
    {
    case 1:
      pd = el->dof[node_f+3][n0_f];
      cd = cds[1][0] < cds[3][0] ? 
	cds[node_e+4][n0_e+1] : cds[node_e+4][n0_e];
      v[pd] = v[cd];
      break;
    case 2:
      pd = el->dof[node_f+2][n0_f];
      cd = cds[2][0] < cds[3][0] ?
	cds[node_e+5][n0_e+1] : cds[node_e+5][n0_e];
      v[pd] = v[cd];
      break;
    }
  }
  return;
}

static void real_coarse_restr3_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr3_3d");
  EL                *el;
  REAL              *v = nil;
  const DOF         *cd;
  DOF                pd[N_BAS3_3D], cdi;
  int                node0, n0, i, lr_set;
  U_CHAR             typ;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN   *admin;

  if (n < 1) return;
  el = list->el_info.el;
  typ = list->el_info.el_type;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  v[pd[0]] += (0.0625*(-v[cd[3]] + v[cd[12]] + v[cd[14]]
		       + v[cd[16]] - v[cd[17]] - v[cd[18]]) 
	       + 0.3125*v[cd[8]]);
  v[pd[1]] += (0.0625*(-v[cd[3]] + v[cd[8]] + v[cd[12]] + v[cd[14]]
		       + v[cd[16]] + v[cd[17]] + v[cd[18]]));
  v[pd[4]] = (v[cd[9]] + 0.5625*v[cd[3]] + 0.9375*v[cd[8]]
	      + 0.0625*(-v[cd[12]] - v[cd[14]] - v[cd[16]])
	      + 0.1875*(v[cd[17]] + v[cd[18]]));
  v[pd[5]] = (0.5625*v[cd[3]] - 0.3125*v[cd[8]]
	      + 0.0625*(-v[cd[12]] - v[cd[14]] - v[cd[16]])
	      + 0.1875*(-v[cd[17]] - v[cd[18]]));
  v[pd[6]] += (-0.25*v[cd[12]] - 0.125*v[cd[16]] + 0.375*v[cd[18]]);
  v[pd[7]] += 0.5*v[cd[12]];
  v[pd[8]] += (-0.25*v[cd[14]] - 0.125*v[cd[16]] + 0.375*v[cd[17]]);
  v[pd[9]] += 0.5*v[cd[14]];
  v[pd[10]] += (-0.25*v[cd[12]] + 0.125*(-v[cd[16]] - v[cd[18]]));
  v[pd[11]] += 0.5*v[cd[12]];
  v[pd[12]] += (-0.25*v[cd[14]] + 0.125*(-v[cd[16]] - v[cd[17]]));
  v[pd[13]] += 0.5*v[cd[14]];
  v[pd[16]] += 0.5*v[cd[16]];
  v[pd[17]] += 0.5*v[cd[16]];
  v[pd[18]] = (v[cd[15]] + 0.5*v[cd[14]] + 0.25*v[cd[16]]
		 + 0.75*v[cd[17]]);
  v[pd[19]] = (v[cd[13]] + 0.5*v[cd[12]] + 0.25*v[cd[16]]
		 + 0.75*v[cd[18]]);

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  if (typ == 0)
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    v[pd[0]] += 0.0625*(v[cd[8]] + v[cd[17]] + v[cd[18]]);
    v[pd[1]] += 0.3125*v[cd[8]] + 0.0625*(-v[cd[17]] - v[cd[18]]);
    v[pd[4]] += -0.3125*v[cd[8]] + 0.1875*(-v[cd[17]] - v[cd[18]]);
    v[pd[5]] += (v[cd[9]] + 0.9375*v[cd[8]] + 0.1875*(v[cd[17]] + v[cd[18]]));
    v[pd[6]] += -0.125*v[cd[17]];
    v[pd[8]] += -0.125*v[cd[18]];
    v[pd[10]] += 0.375*v[cd[17]];
    v[pd[12]] += 0.375*v[cd[18]];
    v[pd[18]] += 0.75*v[cd[18]];
    v[pd[19]] += 0.750000*v[cd[17]];
  }
  else
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 1|2                                                   */
/*--------------------------------------------------------------------------*/

    v[pd[0]] += 0.0625*(v[cd[8]] + v[cd[17]] + v[cd[18]]);
    v[pd[1]] += 0.3125*v[cd[8]] + 0.0625*(-v[cd[17]] - v[cd[18]]);
    v[pd[4]] += -0.3125*v[cd[8]] + 0.1875*(-v[cd[17]] - v[cd[18]]);
    v[pd[5]] += (v[cd[9]] + 0.9375*v[cd[8]] + 0.1875*(v[cd[17]] + v[cd[18]]));
    v[pd[6]] += -0.125*v[cd[18]];
    v[pd[8]] += -0.125*v[cd[17]];
    v[pd[10]] += 0.375*v[cd[18]];
    v[pd[12]] += 0.375*v[cd[17]];
    v[pd[18]] += 0.75*v[cd[17]];
    v[pd[19]] += 0.75*v[cd[18]];
  }

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  node0 = drv->fe_space->mesh->node[FACE];
  n0 = admin->n0_dof[FACE];

  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    typ = list[i].el_info.el_type;

    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[0], admin, nil);
    cdi = cd[16];

    switch(lr_set)
    {
    case 1:
      v[pd[0]] += 0.0625*(v[cd[12]] + v[cdi] - v[cd[18]]);
      v[pd[1]] += 0.0625*(v[cd[12]] + v[cdi] + v[cd[18]]);
      v[pd[4]] += (0.0625*(-v[cd[12]] - v[cdi]) + 0.1875*v[cd[18]]);
      v[pd[5]] += (0.0625*(-v[cd[12]] - v[cdi]) - 0.1875*v[cd[18]]);
      v[pd[6]] += (-0.25*v[cd[12]] - 0.125*v[cdi] + 0.375*v[cd[18]]);
      v[pd[7]] += 0.5*v[cd[12]];
      v[pd[8]] += -0.125*v[cdi];
      v[pd[10]] += -0.25*v[cd[12]] + 0.125*(-v[cdi] - v[cd[18]]);
      v[pd[11]] += 0.5*v[cd[12]];
      v[pd[12]] += -0.125*v[cdi];
      v[pd[16]] += 0.5*v[cdi];
      v[pd[17]] += 0.5*v[cdi];
      v[pd[18]] += 0.25*v[cdi];
      v[pd[19]] = (v[cd[13]] + 0.5*v[cd[12]] + 0.25*v[cdi] + 0.75*v[cd[18]]);
      break;
    case 2:
      v[pd[0]] += 0.0625*(v[cd[14]] + v[cdi] - v[cd[17]]);
      v[pd[1]] += 0.0625*(v[cd[14]] + v[cdi] + v[cd[17]]);
      v[pd[4]] += 0.0625*(-v[cd[14]] - v[cdi]) + 0.1875*v[cd[17]];
      v[pd[5]] += 0.0625*(-v[cd[14]] - v[cdi]) - 0.1875*v[cd[17]];
      v[pd[6]] += -0.125*v[cdi];
      v[pd[8]] += -0.25*v[cd[14]] - 0.125*v[cdi] + 0.375*v[cd[17]];
      v[pd[9]] += 0.5*v[cd[14]];
      v[pd[10]] += -0.125*v[cdi];
      v[pd[12]] += -0.25*v[cd[14]] + 0.125*(-v[cdi] - v[cd[17]]);
      v[pd[13]] += 0.5*v[cd[14]];
      v[pd[16]] += 0.5*v[cdi];
      v[pd[17]] += 0.5*v[cdi];
      v[pd[18]] = (v[cd[15]] + 0.5*v[cd[14]] + 0.25*v[cdi] + 0.75*v[cd[17]]);
      v[pd[19]] += 0.25*v[cdi];
      break;
    case 3:
      v[pd[0]] += 0.0625*v[cdi];
      v[pd[1]] += 0.0625*v[cdi];
      v[pd[4]] += -0.0625*v[cdi];
      v[pd[5]] += -0.0625*v[cdi];
      v[pd[6]] += -0.125*v[cdi];
      v[pd[8]] += -0.125*v[cdi];
      v[pd[10]] += -0.125*v[cdi];
      v[pd[12]] += -0.125*v[cdi];
      v[pd[16]] += 0.5*v[cdi];
      v[pd[17]] += 0.5*v[cdi];
      v[pd[18]] += 0.25*v[cdi];
      v[pd[19]] += 0.25*v[cdi];
      break;
    }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[1], admin, nil);

    if (typ == 0)
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	v[pd[0]] += 0.0625*v[cdi];
	v[pd[1]] += -0.0625*v[cdi];
	v[pd[4]] += -0.1875*v[cdi];
	v[pd[5]] += 0.1875*v[cdi];
	v[pd[6]] += -0.125*v[cdi];
	v[pd[10]] += 0.375*v[cdi];
	v[pd[19]] += 0.75*v[cdi];
	break;
      case 2:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	v[pd[0]] += 0.0625*v[cdi];
	v[pd[1]] += -0.0625*v[cdi];
	v[pd[4]] += -0.1875*v[cdi];
	v[pd[5]] += 0.1875*v[cdi];
	v[pd[8]] += -0.125*v[cdi];
	v[pd[12]] += 0.375*v[cdi];
	v[pd[18]] += 0.75*v[cdi];
	break;
      }
    }
    else
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	v[pd[0]] += 0.0625*v[cdi];
	v[pd[1]] += -0.0625*v[cdi];
	v[pd[4]] += -0.1875*v[cdi];
	v[pd[5]] += 0.1875*v[cdi];
	v[pd[6]] += -0.125*v[cdi];
	v[pd[10]] += 0.375*v[cdi];
	v[pd[19]] += 0.75*v[cdi];
	break;
      case 2:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	v[pd[0]] += 0.0625*v[cdi];
	v[pd[1]] += -0.0625*v[cdi];
	v[pd[4]] += -0.1875*v[cdi];
	v[pd[5]] += 0.1875*v[cdi];
	v[pd[8]] += -0.125*v[cdi];
	v[pd[12]] += 0.375*v[cdi];
	v[pd[18]] += 0.75*v[cdi];
	break;
      }
    }
  }
  return;
}


static void real_d_refine_inter3_3d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_refine_inter3_3d");
  EL                *el;
  REAL_D            *v = nil;
  const DOF         *cd;
  DOF                pd[N_BAS3_3D], cdi;
  int                node0, n0, i, lr_set, k;
  U_CHAR             typ;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN   *admin;

  if (n < 1) return;
  el = list->el_info.el;
  typ = list->el_info.el_type;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[3]][k] = (0.0625*(-v[pd[0]][k] - v[pd[1]][k])
		   + 0.5625*(v[pd[4]][k] + v[pd[5]][k]));
    v[cd[8]][k] = (0.3125*(v[pd[0]][k] - v[pd[5]][k]) + 0.0625*v[pd[1]][k]
		   + 0.937500*v[pd[4]][k]);
    v[cd[9]][k] = v[pd[4]][k];
    v[cd[12]][k] = (0.0625*(v[pd[0]][k] + v[pd[1]][k]
			    - v[pd[4]][k] - v[pd[5]][k])
		    + 0.25*(-v[pd[6]][k] - v[pd[10]][k])
		    + 0.5*(v[pd[7]][k] + v[pd[11]][k] + v[pd[19]][k]));
    v[cd[13]][k] = v[pd[19]][k];
    v[cd[14]][k] = (0.0625*(v[pd[0]][k] + v[pd[1]][k] - v[pd[4]][k]
			    - v[pd[5]][k]) 
		    + 0.25*(-v[pd[8]][k] - v[pd[12]][k]) 
		    + 0.5*(v[pd[9]][k] + v[pd[13]][k] + v[pd[18]][k]));
    v[cd[15]][k] = (v[pd[18]][k]);
    v[cd[16]][k] = (0.0625*(v[pd[0]][k] + v[pd[1]][k]
			    - v[pd[4]][k] - v[pd[5]][k])
		    + 0.125*(-v[pd[6]][k] - v[pd[8]][k]
			     - v[pd[10]][k] - v[pd[12]][k])
		    + 0.5*(v[pd[16]][k] + v[pd[17]][k]) 
		    + 0.25*(v[pd[18]][k] + v[pd[19]][k]));
    v[cd[17]][k] = (0.0625*(-v[pd[0]][k] + v[pd[1]][k]) 
		    + 0.1875*(v[pd[4]][k] - v[pd[5]][k]) + 0.375*v[pd[8]][k]
		    - 0.125*v[pd[12]][k] + 0.75*v[pd[18]][k]);
    v[cd[18]][k] = (0.0625*(-v[pd[0]][k] + v[pd[1]][k]) 
		    + 0.1875*(v[pd[4]][k] - v[pd[5]][k]) + 0.375*v[pd[6]][k]
		    - 0.125*v[pd[10]][k] + 0.75*v[pd[19]][k]);
  }
/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  if (typ == 0)
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    for (k = 0; k < DIM_OF_WORLD; k++)
    {
      v[cd[8]][k] = (0.0625*v[pd[0]][k] + 0.3125*(v[pd[1]][k] - v[pd[4]][k])
		     + 0.9375*v[pd[5]][k]);
      v[cd[9]][k] = v[pd[5]][k];
      v[cd[17]][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k]) 
		      + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		      - 0.125*v[pd[6]][k] + 0.375*v[pd[10]][k]
		      + 0.75*v[pd[19]][k]);
      v[cd[18]][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k]) 
		      + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		      - 0.125*v[pd[8]][k] 
		      + 0.375*v[pd[12]][k] + 0.75*v[pd[18]][k]);
    }
  }
  else
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    for (k = 0; k < DIM_OF_WORLD; k++)
    {
      v[cd[8]][k] = (0.0625*v[pd[0]][k] + 0.3125*(v[pd[1]][k] - v[pd[4]][k])
		     + 0.9375*v[pd[5]][k]);
      v[cd[9]][k] = v[pd[5]][k];
      v[cd[17]][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k]) 
		      + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		      - 0.125*v[pd[8]][k] + 0.375*v[pd[12]][k]
		      + 0.75*v[pd[18]][k]);
      v[cd[18]][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k]) + 
		      0.1875*(-v[pd[4]][k] + v[pd[5]][k]) - 0.125*v[pd[6]][k]
		      + 0.375*v[pd[10]][k] + 0.75*v[pd[19]][k]);
    }
  }

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  node0 = drdv->fe_space->mesh->node[FACE];
  n0 = admin->n0_dof[FACE];
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    typ = list[i].el_info.el_type;

    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set,"no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[0], admin, nil);

    switch(lr_set)
    {
    case 1:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[cd[12]][k] = (0.0625*(v[pd[0]][k]+v[pd[1]][k]
				-v[pd[4]][k]-v[pd[5]][k])
			+ 0.25*(-v[pd[6]][k] - v[pd[10]][k]) 
			+ 0.5*(v[pd[7]][k] + v[pd[11]][k] + v[pd[19]][k]));
	v[cd[13]][k] = v[pd[19]][k];
	v[cd[16]][k] = (0.0625*(v[pd[0]][k]+v[pd[1]][k]
				-v[pd[4]][k]-v[pd[5]][k])
			+ 0.125*(-v[pd[6]][k]-v[pd[8]][k]
				 -v[pd[10]][k]-v[pd[12]][k])
			+ 0.5*(v[pd[16]][k] + v[pd[17]][k]) 
			+ 0.25*(v[pd[18]][k] + v[pd[19]][k]));
	v[cd[18]][k] = (0.0625*(-v[pd[0]][k] + v[pd[1]][k])
			+ 0.1875*(v[pd[4]][k] - v[pd[5]][k])
			+ 0.375*v[pd[6]][k] - 0.125*v[pd[10]][k]
			+ 0.75*v[pd[19]][k]);
      }
      break;
    case 2:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[cd[14]][k] = (0.0625*(v[pd[0]][k]+v[pd[1]][k]
				-v[pd[4]][k]-v[pd[5]][k])
			+ 0.25*(-v[pd[8]][k] - v[pd[12]][k]) 
			+ 0.5*(v[pd[9]][k] + v[pd[13]][k] + v[pd[18]][k]));
	v[cd[15]][k] = v[pd[18]][k];
	v[cd[16]][k] = (0.0625*(v[pd[0]][k]+v[pd[1]][k]
				-v[pd[4]][k]-v[pd[5]][k])
			+ 0.125*(-v[pd[6]][k]-v[pd[8]][k]
				 -v[pd[10]][k]-v[pd[12]][k])
			+ 0.5*(v[pd[16]][k] + v[pd[17]][k])
			+ 0.25*(v[pd[18]][k] + v[pd[19]][k]));
	v[cd[17]][k] = (0.0625*(-v[pd[0]][k] + v[pd[1]][k])
			+ 0.1875*(v[pd[4]][k] - v[pd[5]][k])
			+ 0.375*v[pd[8]][k] - 0.125*v[pd[12]][k]
			+ 0.75*v[pd[18]][k]);
      }
      break;
    case 3:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {    
	v[cd[16]][k] = (0.0625*(v[pd[0]][k]+v[pd[1]][k]
				-v[pd[4]][k]-v[pd[5]][k])
			+ 0.125*(-v[pd[6]][k]-v[pd[8]][k]
				 -v[pd[10]][k]-v[pd[12]][k])
			+ 0.5*(v[pd[16]][k] + v[pd[17]][k])
			+ 0.25*(v[pd[18]][k] + v[pd[19]][k]));
      }
      break;
    }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[1], admin, nil);

    if (typ == 0)
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[cdi][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k])
		       + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		       - 0.125*v[pd[6]][k] + 0.375*v[pd[10]][k]
		       + 0.75*v[pd[19]][k]);
	}
	break;
      case 2:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[cdi][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k])
		       + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		       - 0.125*v[pd[8]][k]
		       + 0.375*v[pd[12]][k] + 0.75*v[pd[18]][k]);
	}
	break;
      }
    }
    else
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[cdi][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k])
		       + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		       - 0.125*v[pd[6]][k] + 0.375*v[pd[10]][k]
		       + 0.75*v[pd[19]][k]);
	}
	break;
      case 2:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[cdi][k] = (0.0625*(v[pd[0]][k] - v[pd[1]][k])
		       + 0.1875*(-v[pd[4]][k] + v[pd[5]][k])
		       - 0.125*v[pd[8]][k] + 0.375*v[pd[12]][k]
		       + 0.75*v[pd[18]][k]);
	}
	break;
      }
    }
  }

  return;
}

static void real_d_coarse_inter3_3d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter3_3d");
  EL        *el;
  REAL_D    *v = nil;
  int       i, lr_set, k;
  int       node_e, node_f, n0_e, n0_f;
  DOF       **pds, **cds;
  DOF       pd_o[N_BAS3_3D], cd, pd;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  node_e = drdv->fe_space->mesh->node[EDGE];        
  n0_e = admin->n0_dof[EDGE];
  node_f = drdv->fe_space->mesh->node[FACE];        
  n0_f = admin->n0_dof[FACE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  pds = el->dof;
  cds = el->child[0]->dof;
  get_dof_indices(el, admin, pd_o);

  pd = (pds[0][0] < pds[1][0]) ? pds[node_e][n0_e] : pds[node_e][n0_e+1];
  cd = cds[0][0] < cds[3][0] ? cds[node_e+2][n0_e+1] : cds[node_e+2][n0_e];
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd][k] = v[cd][k];

  pd = el->dof[node_f+2][n0_f];
  cd = cds[2][0] < cds[3][0] ? cds[node_e+5][n0_e+1] : cds[node_e+5][n0_e];
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd][k] = v[cd][k];

  pd = el->dof[node_f+3][n0_f]; 
  cd = cds[1][0] < cds[3][0] ? cds[node_e+4][n0_e+1] : cds[node_e+4][n0_e];
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd][k] = v[cd][k];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cds = el->child[1]->dof;

  pd = (pds[0][0] < pds[1][0]) ? pds[node_e][n0_e+1] : pds[node_e][n0_e];
  cd = cds[0][0] < cds[3][0] ? cds[node_e+2][n0_e+1] : cds[node_e+2][n0_e];
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd][k] = v[cd][k];


/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;

    pds = el->dof;
    cds = el->child[0]->dof;

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

    switch(lr_set)
    {
    case 1:
      pd = el->dof[node_f+3][n0_f];
      cd = cds[1][0] < cds[3][0] ? 
	cds[node_e+4][n0_e+1] : cds[node_e+4][n0_e];
      for (k = 0; k < DIM_OF_WORLD; k++)
	v[pd][k] = v[cd][k];
      break;
    case 2:
      pd = el->dof[node_f+2][n0_f];
      cd = cds[2][0] < cds[3][0] ?
	cds[node_e+5][n0_e+1] : cds[node_e+5][n0_e];
      for (k = 0; k < DIM_OF_WORLD; k++)
	v[pd][k] = v[cd][k];
      break;
    }
  }
  return;
}

static void real_d_coarse_restr3_3d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_d_coarse_restr3_3d");
  EL                *el;
  REAL_D            *v = nil;
  const DOF         *cd;
  DOF                pd[N_BAS3_3D], cdi;
  int                node0, n0, i, k, lr_set;
  U_CHAR             typ;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN   *admin;

  if (n < 1) return;
  el = list->el_info.el;
  typ = list->el_info.el_type;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (0.0625*(-v[cd[3]][k] + v[cd[12]][k] + v[cd[14]][k]
			    + v[cd[16]][k] - v[cd[17]][k] - v[cd[18]][k]) 
		    + 0.3125*v[cd[8]][k]);
    v[pd[1]][k] += (0.0625*(-v[cd[3]][k] + v[cd[8]][k] + v[cd[12]][k]
			    + v[cd[14]][k] + v[cd[16]][k] + v[cd[17]][k]
			    + v[cd[18]][k]));
    v[pd[4]][k] = (v[cd[9]][k] + 0.5625*v[cd[3]][k] + 0.9375*v[cd[8]][k]
		   + 0.0625*(-v[cd[12]][k] - v[cd[14]][k] - v[cd[16]][k])
		   + 0.1875*(v[cd[17]][k] + v[cd[18]][k]));
    v[pd[5]][k] = (0.5625*v[cd[3]][k] - 0.3125*v[cd[8]][k]
		   + 0.0625*(-v[cd[12]][k] - v[cd[14]][k] - v[cd[16]][k])
		   + 0.1875*(-v[cd[17]][k] - v[cd[18]][k]));
    v[pd[6]][k] += (-0.25*v[cd[12]][k] - 0.125*v[cd[16]][k]
		    + 0.375*v[cd[18]][k]);
    v[pd[7]][k] += 0.5*v[cd[12]][k];
    v[pd[8]][k] += (-0.25*v[cd[14]][k] - 0.125*v[cd[16]][k]
		    + 0.375*v[cd[17]][k]);
    v[pd[9]][k] += 0.5*v[cd[14]][k];
    v[pd[10]][k] += (-0.25*v[cd[12]][k]
		     + 0.125*(-v[cd[16]][k] - v[cd[18]][k]));
    v[pd[11]][k] += 0.5*v[cd[12]][k];
    v[pd[12]][k] += (-0.25*v[cd[14]][k]
		     + 0.125*(-v[cd[16]][k] - v[cd[17]][k]));
    v[pd[13]][k] += 0.5*v[cd[14]][k];
    v[pd[16]][k] += 0.5*v[cd[16]][k];
    v[pd[17]][k] += 0.5*v[cd[16]][k];
    v[pd[18]][k] = (v[cd[15]][k] + 0.5*v[cd[14]][k] + 0.25*v[cd[16]][k]
		    + 0.75*v[cd[17]][k]);
    v[pd[19]][k] = (v[cd[13]][k] + 0.5*v[cd[12]][k] + 0.25*v[cd[16]][k]
		    + 0.75*v[cd[18]][k]);
  }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  if (typ == 0)
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 0                                                     */
/*--------------------------------------------------------------------------*/

    for (k = 0; k < DIM_OF_WORLD; k++)
    {   
      v[pd[0]][k] += 0.0625*(v[cd[8]][k] + v[cd[17]][k] + v[cd[18]][k]);
      v[pd[1]][k] += (0.3125*v[cd[8]][k] 
		      + 0.0625*(-v[cd[17]][k] - v[cd[18]][k]));
      v[pd[4]][k] += (-0.3125*v[cd[8]][k]
		      + 0.1875*(-v[cd[17]][k] - v[cd[18]][k]));
      v[pd[5]][k] += (v[cd[9]][k] + 0.9375*v[cd[8]][k]
		      + 0.1875*(v[cd[17]][k] + v[cd[18]][k]));
      v[pd[6]][k] += -0.125*v[cd[17]][k];
      v[pd[8]][k] += -0.125*v[cd[18]][k];
      v[pd[10]][k] += 0.375*v[cd[17]][k];
      v[pd[12]][k] += 0.375*v[cd[18]][k];
      v[pd[18]][k] += 0.75*v[cd[18]][k];
      v[pd[19]][k] += 0.750000*v[cd[17]][k];
    }
  }
  else
  {
/*--------------------------------------------------------------------------*/
/*  parent of el_type 1|2                                                   */
/*--------------------------------------------------------------------------*/

    for (k = 0; k < DIM_OF_WORLD; k++)
    {
      v[pd[0]][k] += 0.0625*(v[cd[8]][k] + v[cd[17]][k] + v[cd[18]][k]);
      v[pd[1]][k] += (0.3125*v[cd[8]][k]
		      + 0.0625*(-v[cd[17]][k] - v[cd[18]][k]));
      v[pd[4]][k] += (-0.3125*v[cd[8]][k]
		      + 0.1875*(-v[cd[17]][k] - v[cd[18]][k]));
      v[pd[5]][k] += (v[cd[9]][k] + 0.9375*v[cd[8]][k]
		      + 0.1875*(v[cd[17]][k] + v[cd[18]][k]));
      v[pd[6]][k] += -0.125*v[cd[18]][k];
      v[pd[8]][k] += -0.125*v[cd[17]][k];
      v[pd[10]][k] += 0.375*v[cd[18]][k];
      v[pd[12]][k] += 0.375*v[cd[17]][k];
      v[pd[18]][k] += 0.75*v[cd[17]][k];
      v[pd[19]][k] += 0.75*v[cd[18]][k];
    }
  }

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  node0 = drdv->fe_space->mesh->node[FACE];
  n0 = admin->n0_dof[FACE];

  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    typ = list[i].el_info.el_type;

    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[0], admin, nil);
    cdi = cd[16];

    switch(lr_set)
    {
    case 1:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[pd[0]][k] += 0.0625*(v[cd[12]][k] + v[cdi][k] - v[cd[18]][k]);
	v[pd[1]][k] += 0.0625*(v[cd[12]][k] + v[cdi][k] + v[cd[18]][k]);
	v[pd[4]][k] += (0.0625*(-v[cd[12]][k] - v[cdi][k])
			+ 0.1875*v[cd[18]][k]);
	v[pd[5]][k] += (0.0625*(-v[cd[12]][k] - v[cdi][k])
			- 0.1875*v[cd[18]][k]);
	v[pd[6]][k] += (-0.25*v[cd[12]][k] - 0.125*v[cdi][k]
			+ 0.375*v[cd[18]][k]);
	v[pd[7]][k] += 0.5*v[cd[12]][k];
	v[pd[8]][k] += -0.125*v[cdi][k];
	v[pd[10]][k] += -0.25*v[cd[12]][k] + 0.125*(-v[cdi][k]-v[cd[18]][k]);
	v[pd[11]][k] += 0.5*v[cd[12]][k];
	v[pd[12]][k] += -0.125*v[cdi][k];
	v[pd[16]][k] += 0.5*v[cdi][k];
	v[pd[17]][k] += 0.5*v[cdi][k];
	v[pd[18]][k] += 0.25*v[cdi][k];
	v[pd[19]][k] = (v[cd[13]][k] + 0.5*v[cd[12]][k] + 0.25*v[cdi][k]
			+ 0.75*v[cd[18]][k]);
      }
      break;
    case 2:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[pd[0]][k] += 0.0625*(v[cd[14]][k] + v[cdi][k] - v[cd[17]][k]);
	v[pd[1]][k] += 0.0625*(v[cd[14]][k] + v[cdi][k] + v[cd[17]][k]);
	v[pd[4]][k] += (0.0625*(-v[cd[14]][k] - v[cdi][k])
			+ 0.1875*v[cd[17]][k]);
	v[pd[5]][k] += (0.0625*(-v[cd[14]][k] - v[cdi][k])
			- 0.1875*v[cd[17]][k]);
	v[pd[6]][k] += -0.125*v[cdi][k];
	v[pd[8]][k] += (-0.25*v[cd[14]][k] - 0.125*v[cdi][k]
			+ 0.375*v[cd[17]][k]);
	v[pd[9]][k] += 0.5*v[cd[14]][k];
	v[pd[10]][k] += -0.125*v[cdi][k];
	v[pd[12]][k] += -0.25*v[cd[14]][k] + 0.125*(-v[cdi][k]-v[cd[17]][k]);
	v[pd[13]][k] += 0.5*v[cd[14]][k];
	v[pd[16]][k] += 0.5*v[cdi][k];
	v[pd[17]][k] += 0.5*v[cdi][k];
	v[pd[18]][k] = (v[cd[15]][k] + 0.5*v[cd[14]][k]
			+ 0.25*v[cdi][k] + 0.75*v[cd[17]][k]);
	v[pd[19]][k] += 0.25*v[cdi][k];
      }
      break;
    case 3:
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[pd[0]][k] += 0.0625*v[cdi][k];
	v[pd[1]][k] += 0.0625*v[cdi][k];
	v[pd[4]][k] += -0.0625*v[cdi][k];
	v[pd[5]][k] += -0.0625*v[cdi][k];
	v[pd[6]][k] += -0.125*v[cdi][k];
	v[pd[8]][k] += -0.125*v[cdi][k];
	v[pd[10]][k] += -0.125*v[cdi][k];
	v[pd[12]][k] += -0.125*v[cdi][k];
	v[pd[16]][k] += 0.5*v[cdi][k];
	v[pd[17]][k] += 0.5*v[cdi][k];
	v[pd[18]][k] += 0.25*v[cdi][k];
	v[pd[19]][k] += 0.25*v[cdi][k];
      }
      break;
    }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[1], admin, nil);

    if (typ == 0)
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[pd[0]][k] += 0.0625*v[cdi][k];
	  v[pd[1]][k] += -0.0625*v[cdi][k];
	  v[pd[4]][k] += -0.1875*v[cdi][k];
	  v[pd[5]][k] += 0.1875*v[cdi][k];
	  v[pd[6]][k] += -0.125*v[cdi][k];
	  v[pd[10]][k] += 0.375*v[cdi][k];
	  v[pd[19]][k] += 0.75*v[cdi][k];
	}
	break;
      case 2:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[pd[0]][k] += 0.0625*v[cdi][k];
	  v[pd[1]][k] += -0.0625*v[cdi][k];
	  v[pd[4]][k] += -0.1875*v[cdi][k];
	  v[pd[5]][k] += 0.1875*v[cdi][k];
	  v[pd[8]][k] += -0.125*v[cdi][k];
	  v[pd[12]][k] += 0.375*v[cdi][k];
	  v[pd[18]][k] += 0.75*v[cdi][k];
	}
	break;
      }
    }
    else
    {
      switch(lr_set)
      {
      case 1:
	cdi = el->child[1]->dof[node0+2][n0];      
	DEBUG_TEST_EXIT(cdi == cd[18], "cdi != cd[18]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[pd[0]][k] += 0.0625*v[cdi][k];
	  v[pd[1]][k] += -0.0625*v[cdi][k];
	  v[pd[4]][k] += -0.1875*v[cdi][k];
	  v[pd[5]][k] += 0.1875*v[cdi][k];
	  v[pd[6]][k] += -0.125*v[cdi][k];
	  v[pd[10]][k] += 0.375*v[cdi][k];
	  v[pd[19]][k] += 0.75*v[cdi][k];
	}
	break;
      case 2:
	cdi = el->child[1]->dof[node0+1][n0];      
	DEBUG_TEST_EXIT(cdi == cd[17], "cdi != cd[17]\n");
	for (k = 0; k < DIM_OF_WORLD; k++)
	{
	  v[pd[0]][k] += 0.0625*v[cdi][k];
	  v[pd[1]][k] += -0.0625*v[cdi][k];
	  v[pd[4]][k] += -0.1875*v[cdi][k];
	  v[pd[5]][k] += 0.1875*v[cdi][k];
	  v[pd[8]][k] += -0.125*v[cdi][k];
	  v[pd[12]][k] += 0.375*v[cdi][k];
	  v[pd[18]][k] += 0.75*v[cdi][k];
	}
	break;
      }
    }
  }
  return;
}

static BAS_FCT *phi3_3d[N_BAS3_3D] ={phi3v0_3d, phi3v1_3d,phi3v2_3d, phi3v3_3d,
				     phi3e0_3d, phi3e1_3d,phi3e2_3d, phi3e3_3d,
				     phi3e4_3d, phi3e5_3d,phi3e6_3d, phi3e7_3d,
				     phi3e8_3d,phi3e9_3d,phi3e10_3d,phi3e11_3d,
				     phi3c0_3d,phi3c1_3d,phi3c2_3d,phi3c3_3d};

static GRD_BAS_FCT *grd_phi3_3d[N_BAS3_3D] = {grd_phi3v0_3d,  grd_phi3v1_3d,
					      grd_phi3v2_3d,  grd_phi3v3_3d,
					      grd_phi3e0_3d,  grd_phi3e1_3d,
					      grd_phi3e2_3d,  grd_phi3e3_3d,
					      grd_phi3e4_3d,  grd_phi3e5_3d,
					      grd_phi3e6_3d,  grd_phi3e7_3d,
					      grd_phi3e8_3d,  grd_phi3e9_3d,
					      grd_phi3e10_3d, grd_phi3e11_3d,
					      grd_phi3c0_3d,  grd_phi3c1_3d,
					      grd_phi3c2_3d,  grd_phi3c3_3d};

static D2_BAS_FCT   *D2_phi3_3d[N_BAS3_3D]  = {D2_phi3v0_3d,  D2_phi3v1_3d,
					       D2_phi3v2_3d,  D2_phi3v3_3d,
					       D2_phi3e0_3d,  D2_phi3e1_3d,
					       D2_phi3e2_3d,  D2_phi3e3_3d,
					       D2_phi3e4_3d,  D2_phi3e5_3d,
					       D2_phi3e6_3d,  D2_phi3e7_3d,
					       D2_phi3e8_3d,  D2_phi3e9_3d,
					       D2_phi3e10_3d, D2_phi3e11_3d,
					       D2_phi3c0_3d,  D2_phi3c1_3d,
					       D2_phi3c2_3d,  D2_phi3c3_3d};

static const BAS_FCTS lagrange3_3d = {"lagrange3_3d", 3, N_BAS3_3D, 3,
				      {1,0,2,1},/* VERTEX,CENTER,EDGE,FACE */
				      nil,
				      phi3_3d, grd_phi3_3d, D2_phi3_3d, 
				      get_dof_indices3_3d, 
				      get_bound3_3d,
				      interpol3_3d,
				      interpol_d3_3d,
				      get_int_vec3_3d,
				      get_real_vec3_3d,
				      get_real_d_vec3_3d,
				      get_uchar_vec3_3d,
				      get_schar_vec3_3d,
				      real_refine_inter3_3d,
				      real_coarse_inter3_3d,
				      real_coarse_restr3_3d,
				      real_d_refine_inter3_3d,
				      real_d_coarse_inter3_3d,
				      real_d_coarse_restr3_3d,
				      bary3_3d,};
