#include "libpdftex.h"
#include "image.h"

image_entry *image_ptr, *image_tab = 0;
integer image_max;

integer new_image_entry()
{
    ENTRY_ROOM(image, 256);
    image_ptr->ref_count = 0;
    return image_ptr++ - image_tab;
}

void img_free() 
{
    image_entry *p;
    if (image_tab == 0)
        return;
    for (p = image_tab; p < image_ptr; p++) {
        if (p->ref_count > 0) {
            WARN("pending image (%i references)" AND p->ref_count);
            while (p->ref_count > 0)
                deleteimageref(p - image_tab);
        }
    }
    free(image_tab);
}

void addimageref(integer img)
{
    IMG_REFCOUNT(img)++;
}

void deleteimageref(integer img)
{
    if (IMG_REFCOUNT(img) == 0) {
        switch (IMG_TYPE(img)) {
        case IMAGE_TYPE_PNG:
            xfclose(PNG_PTR(img)->io_ptr, filename);
            png_destroy_read_struct(&(PNG_PTR(img)), &(PNG_INFO(img)), NULL);
            break;
        case IMAGE_TYPE_JPG:
            xfclose(JPG_INFO(img)->file, filename);
            break;
        case IMAGE_TYPE_PDF:
            epdf_delete(img);
            break;
        default:
            FAIL("unknown type of image");
        }
        XFREE(IMG_NAME(img));
        return;
    }
    IMG_REFCOUNT(img)--;
}

integer imagewidth(integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        return PNG_INFO(img)->width;
    case IMAGE_TYPE_JPG:
        return JPG_INFO(img)->width;
    case IMAGE_TYPE_PDF:
        return PDF_INFO(img)->bbox[2] - PDF_INFO(img)->bbox[0];
    default:
        FAIL("unknown type of image");
    }
}

integer imageheight(integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        return PNG_INFO(img)->height;
    case IMAGE_TYPE_JPG:
        return JPG_INFO(img)->height;
    case IMAGE_TYPE_PDF:
        return PDF_INFO(img)->bbox[3] - PDF_INFO(img)->bbox[1];
    default:
        FAIL("unknown type of image");
    }
}

integer imagexres(integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        if (PNG_INFO(img)->valid & PNG_INFO_pHYs)
            return 0.0254*png_get_x_pixels_per_meter(PNG_PTR(img), 
                                                     PNG_INFO(img)) + 0.5;
        return 0;
    case IMAGE_TYPE_JPG:
        return JPG_INFO(img)->x_res;
    case IMAGE_TYPE_PDF:
        return 0;
    default:
        FAIL("unknown type of image");
    }
}

integer imageyres(integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        if (PNG_INFO(img)->valid & PNG_INFO_pHYs)
            return 0.0254*png_get_y_pixels_per_meter(PNG_PTR(img),
                                                     PNG_INFO(img)) + 0.5;
        return 0;
    case IMAGE_TYPE_JPG:
        return JPG_INFO(img)->y_res;
    case IMAGE_TYPE_PDF:
        return 0;
    default:
        FAIL("unknown type of image");
    }
}

boolean iscolorimage(integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        return PNG_INFO(img)->color_type != PNG_COLOR_TYPE_GRAY;
    case IMAGE_TYPE_JPG:
        return JPG_INFO(img)->color_space != JPG_GRAY;
    case IMAGE_TYPE_PDF:
        FAIL("`imagecolor' should not be called for PDF image");
    default:
        FAIL("unknown type of image");
    }
}

boolean ispdfimage(integer img)
{
    return IMG_TYPE(img) == IMAGE_TYPE_PDF;
}

integer readimg()
{
    integer img = new_image_entry();
    filename = makecstring(makenamestring());
    flushstring();
    IMG_NAME(img) = kpse_find_file(filename, kpse_tex_ps_header_format, true);
    if (IMG_NAME(img) == 0)
        FAIL("cannot open image file");
    if (strcasecmp(strchr(filename, 0) - 4, ".pdf") == 0) {
        PDF_INFO(img) = XTALLOC(1, pdf_image_struct);
        read_pdf_info(img);
        IMG_TYPE(img) = IMAGE_TYPE_PDF;
    }
    else if (strcasecmp(strchr(filename, 0) - 4, ".png") == 0) {
        read_png_info(img);
        IMG_TYPE(img) = IMAGE_TYPE_PNG;
    }
    else if (strcasecmp(strchr(filename, 0) - 4, ".jpg") == 0) {
        JPG_INFO(img) = XTALLOC(1, JPG_IMAGE_INFO);
        switch (read_jpg_info(img)) {
        case 0:
            break;
        case 4:
            FAIL("unsupported type of compression");
        default: 
            FAIL("reading JPG image failed");
        }
        IMG_TYPE(img) = IMAGE_TYPE_JPG;
    }
    else FAIL("unknown type of image");
    return img;
}

void writeimg(integer n, integer img)
{
    switch (IMG_TYPE(img)) {
    case IMAGE_TYPE_PNG:
        write_png(n, img);
        break;
    case IMAGE_TYPE_JPG:
        write_jpg(n, img);
        break;
    case IMAGE_TYPE_PDF:
        write_epdf(n, img);
        break;
    default:
        FAIL("unknown type of image");
    }
}
