#include "libpdftex.h"

typedef signed char     TTF_CHAR;
typedef unsigned char   TTF_BYTE;
typedef signed short    TTF_SHORT;
typedef unsigned short  TTF_USHORT;
typedef signed long     TTF_LONG;
typedef unsigned long   TTF_ULONG;
typedef unsigned long   TTF_FIXED;
typedef unsigned short  TTF_FUNIT;
typedef signed short    TTF_FWORD;
typedef unsigned short  TTF_UFWORD;
typedef unsigned short  TTF_F2DOT14;

#define TTF_CHAR_SIZE    1
#define TTF_BYTE_SIZE    1
#define TTF_SHORT_SIZE   2
#define TTF_USHORT_SIZE  2
#define TTF_LONG_SIZE    4
#define TTF_ULONG_SIZE   4
#define TTF_FIXED_SIZE   4
#define TTF_FWORD_SIZE   2
#define TTF_UFWORD_SIZE  2
#define TTF_F2DOT14_SIZE 2

#define ARG_1_AND_2_ARE_WORDS       (1<<0)
#define ARGS_ARE_XY_VALUES          (1<<1)
#define ROUND_XY_TO_GRID            (1<<2)
#define WE_HAVE_A_SCALE             (1<<3)
#define RESERVED                    (1<<4)
#define MORE_COMPONENTS             (1<<5)
#define WE_HAVE_AN_X_AND_Y_SCALE    (1<<6)
#define WE_HAVE_A_TWO_BY_TWO        (1<<7)
#define WE_HAVE_INSTRUCTIONS        (1<<8)
#define USE_MY_METRICS              (1<<9)

#define GET_NUM(t)      ((t)getnum(t##_SIZE))
#define SKIP(n)         getnum(n)

#define GET_CHAR()      GET_NUM(TTF_CHAR)
#define GET_BYTE()      GET_NUM(TTF_BYTE)
#define GET_SHORT()     GET_NUM(TTF_SHORT)
#define GET_USHORT()    GET_NUM(TTF_USHORT)
#define GET_LONG()      GET_NUM(TTF_LONG)
#define GET_ULONG()     GET_NUM(TTF_ULONG)
#define GET_FIXED()     GET_NUM(TTF_FIXED)
#define GET_FUNIT()     GET_NUM(TTF_FUNIT)
#define GET_FWORD()     GET_NUM(TTF_FWORD)
#define GET_UFWORD()    GET_NUM(TTF_UFWORD)
#define GET_F2DOT14()   GET_NUM(TTF_F2DOT14)

#define PUT_NUM(t,n)    ((t)putnum(t##_SIZE, n))

#define PUT_CHAR(n)     PUT_NUM(TTF_CHAR, n)
#define PUT_BYTE(n)     PUT_NUM(TTF_BYTE, n)
#define PUT_SHORT(n)    PUT_NUM(TTF_SHORT, n)
#define PUT_USHORT(n)   PUT_NUM(TTF_USHORT, n)
#define PUT_LONG(n)     PUT_NUM(TTF_LONG, n)
#define PUT_ULONG(n)    PUT_NUM(TTF_ULONG, n)
#define PUT_FIXED(n)    PUT_NUM(TTF_FIXED, n)
#define PUT_FUNIT(n)    PUT_NUM(TTF_FUNIT, n)
#define PUT_FWORD(n)    PUT_NUM(TTF_FWORD, n)
#define PUT_UFWORD(n)   PUT_NUM(TTF_UFWORD, n)
#define PUT_F2DOT14(n)  PUT_NUM(TTF_F2DOT14, n)

#define COPY_BYTE()     PUT_BYTE(GET_BYTE())
#define COPY_CHAR()     PUT_CHAR(GET_CHAR())
#define COPY_USHORT()   PUT_USHORT(GET_USHORT())
#define COPY_SHORT()    PUT_SHORT(GET_SHORT())
#define COPY_ULONG()    PUT_ULONG(GET_ULONG())
#define COPY_LONG()     PUT_LONG(GET_LONG())
#define COPY_FIXED()    PUT_FIXED(GET_FIXED())
#define COPY_FUNIT()    PUT_FUNIT(GET_FUNIT())
#define COPY_FWORD()    PUT_FWORD(GET_FWORD())
#define COPY_UFWORD()   PUT_UFWORD(GET_UFWORD())
#define COPY_F2DOT14()  PUT_F2DOT14(GET_F2DOT14())

#define MAC_GLYPH_MAX   258
#define TABDIR_OFF      12

typedef struct {
    char tag[4];
    TTF_ULONG checksum;
    TTF_ULONG offset;
    TTF_ULONG length;
} tabdir_entry;

typedef struct {
    TTF_LONG offset;
    TTF_LONG newoffset;
    TTF_UFWORD advWidth;
    TTF_FWORD lsb;
    char *name;             /* name of glyph */
    TTF_SHORT newindex;      /* new index of glyph in output file */
    TTF_USHORT name_index;   /* index of name as read from font file */
} glyph_entry;

typedef struct {
    char *name;             /* name of glyph */
    short newindex;         /* new index of glyph in output file */
} ttfenc_entry;

typedef struct {
    TTF_USHORT platform_id;
    TTF_USHORT encoding_id;
    TTF_USHORT language_id;
    TTF_USHORT name_id;
    TTF_USHORT length;
    TTF_USHORT offset;
} name_record;

static TTF_USHORT ntabs;
static TTF_ULONG checksum;
static TTF_USHORT upem;
static TTF_FIXED post_format;
static TTF_SHORT loca_format;
static TTF_ULONG ttf_offset;
static TTF_USHORT glyphs_count;
static TTF_USHORT new_glyphs_count;
static TTF_USHORT nhmtxs;
static TTF_USHORT max_glyph_index;
static TTF_USHORT new_ntabs;

/* 
 * reindexing glyphs is a bit unclear: `glyph_tab' contains glyphs in
 * original order in font file, `ttfenc_tab' is the new encoding vector and
 * `glyph_index' is the new order of glyphs. So n-th glyph in new order is
 * located at `glyph_tab[glyph_index[n]]'. The field `newindex' of entries in
 * both `glyph_tab' and `ttfenc_tab' contains the index of the corresponding
 * glyph...
 *
 */

static glyph_entry *glyph_tab;
static ttfenc_entry *ttfenc_tab;
static short *glyph_index;

static name_record *names_tab;
static int names_record_num;
static char *names_storage;
static int names_storage_size;

static tabdir_entry *dir_tab;
static char *glyph_names_buf;
static int chkcount;
static FILE *ttf_file;
integer ttf_length;

#define INFILE ttf_file
#define OUTFILE pdffile

#define TTF_OPEN()      truetypebopenin(ttf_file)
#define TTF_CLOSE()     xfclose(ttf_file)
#define TTF_GETCHAR()   xgetc(ttf_file)
#define TTF_EOF()       feof(ttf_file)

char *mac_glyph_names[MAC_GLYPH_MAX] = {
notdef, ".null", "CR", "space", "exclam", "quotedbl", "numbersign", "dollar",
"percent", "ampersand", "quotesingle", "parenleft", "parenright", "asterisk",
"plus", "comma", "hyphen", "period", "slash", "zero", "one", "two", "three",
"four", "five", "six", "seven", "eight", "nine", "colon", "semicolon", "less",
"equal", "greater", "question", "at", "A", "B", "C", "D", "E", "F", "G", "H",
"I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W",
"X", "Y", "Z", "bracketleft", "backslash", "bracketright", "asciicircum",
"underscore", "grave", "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k",
"l", "m", "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z",
"braceleft", "bar", "braceright", "asciitilde", "Adieresis", "Aring",
"Ccedilla", "Eacute", "Ntilde", "Odieresis", "Udieresis", "aacute", "agrave",
"acircumflex", "adieresis", "atilde", "aring", "ccedilla", "eacute", "egrave",
"ecircumflex", "edieresis", "iacute", "igrave", "icircumflex", "idieresis",
"ntilde", "oacute", "ograve", "ocircumflex", "odieresis", "otilde", "uacute",
"ugrave", "ucircumflex", "udieresis", "dagger", "degree", "cent", "sterling",
"section", "bullet", "paragraph", "germandbls", "registered", "copyright",
"trademark", "acute", "dieresis", "notequal", "AE", "Oslash", "infinity",
"plusminus", "lessequal", "greaterequal", "yen", "mu", "partialdiff", "Sigma",
"Pi", "pi", "integral", "ordfeminine", "ordmasculine", "Omega", "ae",
"oslash", "questiondown", "exclamdown", "logicalnot", "radical", "florin",
"approxequal", "Delta", "guillemotleft", "guillemotright", "ellipsis",
"nbspace", "Agrave", "Atilde", "Otilde", "OE", "oe", "endash", "emdash",
"quotedblleft", "quotedblright", "quoteleft", "quoteright", "divide",
"lozenge", "ydieresis", "Ydieresis", "fraction", "currency", "guilsinglleft",
"guilsinglright", "fi", "fl", "daggerdbl", "middot", "quotesinglbase",
"quotedblbase", "perthousand", "Acircumflex", "Ecircumflex", "Aacute",
"Edieresis", "Egrave", "Iacute", "Icircumflex", "Idieresis", "Igrave",
"Oacute", "Ocircumflex", "applelogo", "Ograve", "Uacute", "Ucircumflex",
"Ugrave", "dotlessi", "circumflex", "tilde", "overscore", "breve",
"dotaccent", "ring", "cedilla", "hungarumlaut", "ogonek", "caron", "Lslash",
"lslash", "Scaron", "scaron", "Zcaron", "zcaron", "brokenbar", "Eth", "eth",
"Yacute", "yacute", "Thorn", "thorn", "minus", "multiply", "onesuperior",
"twosuperior", "threesuperior", "onehalf", "onequarter", "threequarters",
"franc", "Gbreve", "gbreve", "Idot", "Scedilla", "scedilla", "Cacute",
"cacute", "Ccaron", "ccaron", "dmacron"
};

static char *newtabnames[] = {
    "OS/2",
    "PCLT",
    "cmap",
    "cvt ",
    "fpgm",
    "glyf",
    "head",
    "hhea",
    "hmtx",
    "loca",
    "maxp",
    "name",
    "post",
    "prep"
};

static char ttf_embedded_prefix[] = "Embedded";

#define DEFAULT_NTABS       14

static unsigned char addchksm(unsigned char b)
{
    checksum += (b << (8*chkcount--));
    if (chkcount < 0)
        chkcount = 3;
    return b;
}

static long putnum(int s, long n)
{
    long i = n;
    char buf[TTF_LONG_SIZE + 1], *p = buf;
    while (s-- > 0) {
        *p++ = i & 0xFF;
        i >>= 8;
    }
    p--;
    while (p >= buf)
        xputc(addchksm(*p--), OUTFILE);
    return n;
}

static long getnum(int s)
{
    long i = 0;
    int c;
    while (s > 0) {
        if ((c = TTF_GETCHAR()) < 0)
            FAIL("unexpected EOF");
        i = (i << 8) + c;
        s--;
    }
    return i;
}

static long funit(long n)
{
    if (n < 0)
        return -((-n/upem)*1000 + ((-n%upem)*1000)/upem);
    else
        return (n/upem)*1000 + ((n%upem)*1000)/upem;
}

static void ncopy(int n)
{
    while (n-- > 0)
        COPY_BYTE();
}

static tabdir_entry *name_lookup(char *s, boolean required)
{
    tabdir_entry *tab;
    for (tab = dir_tab; tab - dir_tab < ntabs; tab++)
        if (strncmp(tab->tag, s, 4) == 0)
            break;
    if (tab - dir_tab == ntabs)
        if (required)
            FAIL("can't find table `%s'" AND s);
        else
            tab = 0;
    return tab;
}

static tabdir_entry *seek_tab(char *name, TTF_LONG offset)
{
    tabdir_entry *tab = name_lookup(name, true);
    sprintf(printf_buf, "%s while reading table `%s'", filename, name);
    xfseek(INFILE, tab->offset + offset, SEEK_SET, printf_buf);
    return tab;
}

static void seek_off(char *name, TTF_LONG offset)
{
    sprintf(printf_buf, "%s while reading table %s", filename, name);
    xfseek(INFILE, offset, SEEK_SET, printf_buf);
}

static void copy_encoding()
{
    int i;
    char **n = cur_glyph_names = enc_tab[fm_cur->encoding].glyph_names;
    ttfenc_entry *e = ttfenc_tab = XTALLOC(MAX_CHAR_NUM, ttfenc_entry);
    for (i = 0; i < MAX_CHAR_NUM; i++) {
        if (pdfischarused(tex_font, i))
            e->name = *n;
        else
            e->name = notdef;
        n++;
        e++;
    }
}

static void read_names()
{
    int i;
    tabdir_entry *tab;
    char *p;
    tab = seek_tab("name", TTF_USHORT_SIZE);
    names_record_num = GET_USHORT();
    names_tab = XTALLOC(names_record_num, name_record);
    names_storage_size = tab->length - 
         (3*TTF_USHORT_SIZE + names_record_num*6*TTF_USHORT_SIZE);
    names_storage = XTALLOC(names_storage_size, char);
    SKIP(TTF_USHORT_SIZE);
    for (i = 0; i < names_record_num; i++) {
        names_tab[i].platform_id = GET_USHORT();
        names_tab[i].encoding_id = GET_USHORT();
        names_tab[i].language_id = GET_USHORT();
        names_tab[i].name_id = GET_USHORT();
        names_tab[i].length = GET_USHORT();
        names_tab[i].offset = GET_USHORT();
    }
    for (p = names_storage; p - names_storage < names_storage_size;)
         *p++ = GET_CHAR();
}

static void read_font()
{
    long i, k, l;
    char *p;
    glyph_entry *glyph;
    tabdir_entry *tab;
    TTF_UFWORD last_advWidth;
    if ((i= GET_FIXED()) != 0x00010000)
        FAIL("unsupport version %8lx; can handle only version 1.0" AND i);
    dir_tab = XTALLOC(ntabs = GET_USHORT(), tabdir_entry);
    SKIP(3*TTF_USHORT_SIZE);
    for (tab = dir_tab; tab - dir_tab < ntabs; tab++) {
        for (i = 0; i < 4; i++)
            tab->tag[i] = GET_CHAR();
        tab->checksum = GET_ULONG();
        tab->offset = GET_ULONG();
        tab->length = GET_ULONG();
    }
    if (name_lookup("PCLT", false) == 0)
        new_ntabs--;
    if (name_lookup("fpgm", false) == 0)
        new_ntabs--;
    if (name_lookup("cvt ", false) == 0)
        new_ntabs--;
    if (name_lookup("prep", false) == 0)
        new_ntabs--;
    seek_tab("maxp", TTF_FIXED_SIZE);
    glyph_tab = XTALLOC(1 + (glyphs_count = GET_USHORT()), glyph_entry);
    for (glyph = glyph_tab; glyph - glyph_tab < glyphs_count; glyph++)
        glyph->newindex = -1;
    glyph_index = XTALLOC(glyphs_count, short);
    glyph_index[0] = 0; /* index of ".notdef" glyph */
    glyph_index[1] = 1; /* index of ".null" glyph */
    for (i = 0; i < MAX_KEY_CODE; i++)
        font_keys[i].valid = true;
    seek_tab("head", 2*TTF_FIXED_SIZE + 2*TTF_ULONG_SIZE + TTF_USHORT_SIZE);
    upem = GET_USHORT();
    SKIP(16);
    font_keys[FONTBBOX1_CODE].value.i = funit(GET_FWORD());
    font_keys[FONTBBOX2_CODE].value.i = funit(GET_FWORD());
    font_keys[FONTBBOX3_CODE].value.i = funit(GET_FWORD());
    font_keys[FONTBBOX4_CODE].value.i = funit(GET_FWORD());
    SKIP(2*TTF_USHORT_SIZE + TTF_SHORT_SIZE);
    loca_format = GET_SHORT();
    seek_tab("hhea", TTF_FIXED_SIZE);
    font_keys[ASCENT_CODE].value.i = funit(GET_FWORD());
    font_keys[DESCENT_CODE].value.i = funit(GET_FWORD());
    SKIP(TTF_FWORD_SIZE + TTF_UFWORD_SIZE + 3*TTF_FWORD_SIZE + 8*TTF_SHORT_SIZE);
    nhmtxs = GET_USHORT();
    if (name_lookup("PCLT", false) != 0) {
        seek_tab("PCLT", TTF_FIXED_SIZE + TTF_ULONG_SIZE + TTF_USHORT_SIZE);
        font_keys[XHEIGHT_CODE].value.i = funit(GET_USHORT());
        SKIP(2*TTF_USHORT_SIZE);
        font_keys[CAPHEIGHT_CODE].value.i = funit(GET_USHORT());
    }
    else {
        font_keys[XHEIGHT_CODE].valid = false;
        font_keys[CAPHEIGHT_CODE].valid = false;
    }
    seek_tab("hmtx", 0);
    for (glyph = glyph_tab; glyph - glyph_tab < nhmtxs; glyph++) {
        glyph->advWidth = GET_UFWORD();
        glyph->lsb = GET_UFWORD();
    }
    if (nhmtxs < glyphs_count) {
        last_advWidth = glyph[-1].advWidth;
        for (;glyph - glyph_tab <  glyphs_count; glyph++) {
            glyph->advWidth = last_advWidth;
            glyph->lsb = GET_UFWORD();
        }
    }
    tab = seek_tab("post", 0);
    post_format = GET_FIXED();
    font_keys[ITALIC_ANGLE_CODE].value.i = GET_FIXED() >> 16;
    SKIP(2*TTF_FWORD_SIZE + 5*TTF_ULONG_SIZE);
    switch (post_format) {
    case 0x10000:
        for (glyph = glyph_tab; glyph - glyph_tab < MAC_GLYPH_MAX; glyph++) {
            glyph->name = mac_glyph_names[glyph - glyph_tab];
            glyph->name_index = glyph - glyph_tab;
        }
        for (;glyph - glyph_tab < glyphs_count; glyph++) {
            glyph->name = notdef;
            glyph->name_index = 0;
        }
        break;
    case 0x20000:
        l = GET_USHORT(); /* some fonts have this value different from nglyphs */
        for (glyph = glyph_tab; glyph - glyph_tab < l; glyph++)
            glyph->name_index = GET_USHORT();
        i = tab->length - (xftell(INFILE, filename) - tab->offset);
        glyph_names_buf = XTALLOC(i, char);
        for (p = glyph_names_buf; p - glyph_names_buf < i;) {
            for (k = GET_BYTE(); k > 0; k--)
                *p++ = GET_CHAR();
            *p++ = 0;
        }
        for (glyph = glyph_tab; glyph - glyph_tab < l; glyph++) {
            if (glyph->name_index < MAC_GLYPH_MAX)
                glyph->name = mac_glyph_names[glyph->name_index];
            else {
                p = glyph_names_buf;
                k = glyph->name_index - MAC_GLYPH_MAX;
                for (; k > 0; k--)
                    p = (char *)strchr(p, 0) + 1;
                glyph->name = p;
            }
        }
        break;
    default:
        FAIL("unsupported format (%8X) of `post' table"
             AND (unsigned int) post_format);
    }
    seek_tab("loca", 0);
    if (loca_format != 0)
        for (glyph = glyph_tab; glyph - glyph_tab < glyphs_count + 1; glyph++)
            glyph->offset = GET_ULONG();
    else
        for (glyph = glyph_tab; glyph - glyph_tab < glyphs_count + 1; glyph++)
            glyph->offset = GET_USHORT() << 1;
    read_names();
}

#define RESET_CHKSM() do {                                 \
    checksum = 0L;                                         \
    chkcount = 3;                                          \
    tab->offset = xftell(OUTFILE, filename) - ttf_offset;  \
} while (0)

#define SET_CHKSM() do {                                   \
    tab->checksum = checksum;                               \
    tab->length = xftell(OUTFILE, filename) - tab->offset - ttf_offset; \
} while (0)

static void copytab(char *name)
{
    tabdir_entry *tab = seek_tab(name, 0);
    long i;
    RESET_CHKSM();
    for (i = tab->length; i > 0; i--)
        COPY_CHAR();
    SET_CHKSM();
}

static void reenc_font()
{
    ttfenc_entry *e;
    tabdir_entry *tab;
    tab = name_lookup("cmap", true);
    RESET_CHKSM();
    PUT_USHORT(0);  /* table version number (0) */
    PUT_USHORT(1);  /* number of encoding tables (1) */
    PUT_USHORT(1);  /* platform ID */
    PUT_USHORT(0);  /* encoding ID */
    PUT_ULONG(4*TTF_USHORT_SIZE + TTF_ULONG_SIZE); /* byte offset from beginning of table */
    if (max_glyph_index < 256) {
        PUT_USHORT(0);  /* format number (0: byte encoding table) */
        PUT_USHORT(256 + 3*TTF_USHORT_SIZE); /* length of table */
        PUT_USHORT(0);  /* version number */
        for (e = ttfenc_tab; e - ttfenc_tab < MAX_CHAR_NUM; e++)
            PUT_BYTE(e->newindex);
    }
    else {
        WARN(\
"embedded TrueType font contains more than 255 glyphs, so trimmed table\n\
mapping is used. The output PDF may not be handled by current version of\n\
Acrobat and GhostScript");
        PUT_USHORT(6);  /* format number (6): trimmed table mapping */
        PUT_USHORT(TTF_USHORT_SIZE*(5 + MAX_CHAR_NUM));
        PUT_USHORT(0);  /* version number (0) */
        PUT_USHORT(0);  /* first character code */
        PUT_USHORT(MAX_CHAR_NUM);  /* number of character code in table */
        for (e = ttfenc_tab; e - ttfenc_tab < MAX_CHAR_NUM; e++)
            PUT_USHORT(e->newindex);
    }
    SET_CHKSM();
}

static void copy_dirtab()
{
    tabdir_entry *tab;
    int i;
    fflush(OUTFILE);
    pdfgone += xftell(OUTFILE, filename) - ttf_offset;
    xfseek(OUTFILE, ttf_offset + TABDIR_OFF, SEEK_SET, filename);
    if (is_subsetted()) 
        for (i = 0; i < DEFAULT_NTABS; i++) {
            tab = name_lookup(newtabnames[i], false);
            if (tab == 0)
                continue;
            for (k = 0; k < 4; k++)
               PUT_CHAR(tab->tag[k]);
            PUT_ULONG(tab->checksum);
            PUT_ULONG(tab->offset);
            PUT_ULONG(tab->length);
        }
    else
        for (tab = dir_tab; tab - dir_tab < ntabs; tab++) {
            for (k = 0; k < 4; k++)
               PUT_CHAR(tab->tag[k]);
            PUT_ULONG(tab->checksum);
            PUT_ULONG(tab->offset);
            PUT_ULONG(tab->length);
        }
}

static void write_names()
{
    char *new_names_storage = 
        XTALLOC(names_storage_size + 
                names_record_num*(3 + strlen(ttf_embedded_prefix)), char);
    char *p = new_names_storage, *r, *s, *t=NULL;
    int i, l;
    for (l = i = 0; i < names_record_num; i++)
        if (names_tab[i].platform_id == 1 && names_tab[i].encoding_id == 0)
            l++;
    PUT_USHORT(0); /* Format selector (0) */
    PUT_USHORT(l); /* number of name records */
    PUT_USHORT(3*TTF_USHORT_SIZE + l*6*TTF_USHORT_SIZE);
    for (i = 0; i < names_record_num; i++) {
        if (names_tab[i].platform_id == 1 && names_tab[i].encoding_id == 0) {
            s = p;
            if (names_tab[i].name_id == 1 || names_tab[i].name_id == 4 ||
                names_tab[i].name_id == 6) {
                for (r = ttf_embedded_prefix; *r; *p++ = *r++);
                if (names_tab[i].name_id == 6) {
                    *p++ = '-';
                    t = p;
                }
                else
                    *p++ = ' ';
            }
            r = names_storage + names_tab[i].offset;
            l = names_tab[i].length;
            for (; l > 0; l--)
                *p++ = *r++;
            if (names_tab[i].name_id == 6) {
                *p = 0;
                font_keys[FONTNAME_CODE].value.s = xstrdup(t);
            }
            PUT_USHORT(names_tab[i].platform_id);
            PUT_USHORT(names_tab[i].encoding_id);
            PUT_USHORT(names_tab[i].language_id);
            PUT_USHORT(names_tab[i].name_id);
            PUT_USHORT(p - s);
            PUT_USHORT(s - new_names_storage);
        }
    }
    for (r = new_names_storage; r < p; r++)
        PUT_BYTE(*r);
    XFREE(new_names_storage);
}

static void subset_font()
{
    long i, k;
    char *p;
    TTF_USHORT flags;
    TTF_USHORT idx;
    tabdir_entry *tab;
    short *index;
    ttfenc_entry *e;
    glyph_entry *glyph;
    ttf_offset = xftell(OUTFILE, filename);
    /* 
     * reindexing glyphs: we append index of used glyphs to `glyph_index'
     * while going through `ttfenc_tab'. After appending a new entry to
     * `glyph_index' we set field `newindex' of corresponding entries in both
     * `glyph_tab' and `ttfenc_tab' to the newly created index
     * 
     */
    for (e = ttfenc_tab; e - ttfenc_tab < MAX_CHAR_NUM; e++) {
        e->newindex = 0; /* index of ".notdef" glyph */
        if (e->name != notdef) {
            for (glyph = glyph_tab; glyph - glyph_tab < glyphs_count; glyph++)
                if (!strcmp(e->name, glyph->name))
                    break;
            if (glyph - glyph_tab < glyphs_count) {
                glyph_index[new_glyphs_count] = glyph - glyph_tab;
                glyph->newindex = e->newindex = new_glyphs_count;
                new_glyphs_count++;
            }
            else
                WARN("glyph `%s' undefined" AND e->name);
        }
    }
    max_glyph_index = new_glyphs_count - 1;
    for (i = 1, k = 0; i <= new_ntabs; i <<= 1, k++);
    PUT_FIXED(0x00010000); /* font version */
    PUT_USHORT(new_ntabs); /* number of tables */
    PUT_USHORT(i << 3); /* search range */
    PUT_USHORT(k - 1); /* entry selector */
    PUT_USHORT((new_ntabs<<4) - (i<<3)); /* range shift */
    xfseek(OUTFILE, ttf_offset + TABDIR_OFF + new_ntabs*4*TTF_ULONG_SIZE,
        SEEK_SET, filename);
    copytab("OS/2");
    if (name_lookup("PCLT", false) != 0)
        copytab("PCLT");
    if (name_lookup("fpgm", false) != 0)
        copytab("fpgm");
    if (name_lookup("cvt ", false) != 0)
        copytab("cvt ");
    if (name_lookup("prep", false) != 0)
        copytab("prep");
    reenc_font();
    i = (tab = name_lookup("glyf", true))->offset;
    RESET_CHKSM();
    loca_format = 0;
    for (index = glyph_index; index - glyph_index < new_glyphs_count; index++) {
        if ((glyph_tab[*index].newoffset = xftell(OUTFILE, filename) - tab->offset - ttf_offset) > 0x00020000 ||
            (glyph_tab[*index].newoffset & 1))
            loca_format = 1;
        if (glyph_tab[*index].offset != glyph_tab[*index + 1].offset) {
            seek_off("glyph", i + glyph_tab[*index].offset);
            k = COPY_SHORT();
            ncopy(4*TTF_FWORD_SIZE);
            if (k < 0) {
                do {
                    flags = COPY_USHORT();
                    idx = GET_USHORT();
                    if (glyph_tab[idx].newindex < 0) {
                        glyph_tab[idx].newindex = new_glyphs_count;
                        glyph_index[new_glyphs_count++] = idx;
                        /* 
                            NOTICE: Here we change `new_glyphs_count',
                            which appears in the condition of the `for' loop
                        */
                    }
                    PUT_USHORT(glyph_tab[idx].newindex);
                    if (flags & ARG_1_AND_2_ARE_WORDS)
                        ncopy(2*TTF_SHORT_SIZE);
                    else
                        ncopy(TTF_USHORT_SIZE);
                    if (flags & WE_HAVE_A_SCALE)
                        ncopy(TTF_F2DOT14_SIZE);
                    else if (flags & WE_HAVE_AN_X_AND_Y_SCALE)
                        ncopy(2*TTF_F2DOT14_SIZE);
                    else if (flags & WE_HAVE_A_TWO_BY_TWO)
                        ncopy(4*TTF_F2DOT14_SIZE);
                } while (flags & MORE_COMPONENTS);
                if (flags & WE_HAVE_INSTRUCTIONS)
                    ncopy(COPY_USHORT());
            }
            else
                ncopy(glyph_tab[*index + 1].offset - glyph_tab[*index].offset - 
                    TTF_USHORT_SIZE - 4*TTF_FWORD_SIZE);
        }
    }
    SET_CHKSM();
    if ((i = tab->length) > 0x00020000 || i & 1)
        loca_format = 1;
    tab = seek_tab("head", 0);
    RESET_CHKSM();
    ncopy(2*TTF_FIXED_SIZE + 2*TTF_ULONG_SIZE + 2*TTF_USHORT_SIZE + 16 + 
        4*TTF_FWORD_SIZE + 2*TTF_USHORT_SIZE + TTF_SHORT_SIZE);
    if (loca_format)
        PUT_SHORT(1);
    else
        PUT_SHORT(0);
    PUT_SHORT(0);
    SET_CHKSM();
    tab = seek_tab("hhea", 0);
    RESET_CHKSM();
    ncopy(TTF_FIXED_SIZE + 3*TTF_FWORD_SIZE + TTF_UFWORD_SIZE + 3*TTF_FWORD_SIZE + 8*TTF_SHORT_SIZE);
    PUT_USHORT(new_glyphs_count);
    SET_CHKSM();
    tab = seek_tab("hmtx", 0);
    RESET_CHKSM();
    for (index = glyph_index; index - glyph_index < new_glyphs_count; index++) {
        PUT_UFWORD(glyph_tab[*index].advWidth);
        PUT_UFWORD(glyph_tab[*index].lsb);
    }
    SET_CHKSM();
    tab = seek_tab("loca", 0);
    RESET_CHKSM();
    if (loca_format != 0) {
        for (index = glyph_index; index - glyph_index < new_glyphs_count; index++)
            PUT_ULONG(glyph_tab[*index].newoffset);
        PUT_ULONG(i);
    }
    else {
        for (index = glyph_index; index - glyph_index < new_glyphs_count; index++)
            PUT_USHORT(glyph_tab[*index].newoffset/2);
        PUT_USHORT(i/2);
    }
    SET_CHKSM();
    tab = seek_tab("maxp", TTF_FIXED_SIZE + TTF_USHORT_SIZE);
    RESET_CHKSM();
    PUT_FIXED(0x00010000);
    PUT_USHORT(new_glyphs_count);
    ncopy(13*TTF_USHORT_SIZE);
    SET_CHKSM();
    tab = seek_tab("name", 0);
    RESET_CHKSM();
    write_names();
    SET_CHKSM();
    tab = seek_tab("post", TTF_FIXED_SIZE);
    RESET_CHKSM();
    PUT_FIXED(0x00020000);
    ncopy(TTF_FIXED_SIZE + 2*TTF_FWORD_SIZE + 5*TTF_ULONG_SIZE);
    PUT_USHORT(new_glyphs_count);
    for (k = 0, index = glyph_index; index - glyph_index < new_glyphs_count; index++)
        if (glyph_tab[*index].name_index < MAC_GLYPH_MAX)
            PUT_USHORT(glyph_tab[*index].name_index);
        else
            PUT_USHORT(MAC_GLYPH_MAX + k++);
    for (index = glyph_index; index - glyph_index < new_glyphs_count; index++)
        if (glyph_tab[*index].name_index >= MAC_GLYPH_MAX) {
            PUT_BYTE(k = strlen(p = glyph_tab[*index].name));
            while (k-- > 0)
                PUT_CHAR(*p++);
        }
    SET_CHKSM();
    copy_dirtab();
}

static void copy_font()
{
    ttfenc_entry *e;
    tabdir_entry *tab;
    glyph_entry *glyph;
    int i;
    ttf_offset = xftell(OUTFILE, filename);
    for (i = 1, k = 0; i <= ntabs; i <<= 1, k++);
    PUT_FIXED(0x00010000); /* font version */
    PUT_USHORT(ntabs); /* number of tables */
    PUT_USHORT(i << 3); /* search range */
    PUT_USHORT(k - 1); /* entry selector */
    PUT_USHORT((ntabs<<4) - (i<<3)); /* range shift */
    xfseek(OUTFILE, ttf_offset + TABDIR_OFF + ntabs*4*TTF_ULONG_SIZE,
        SEEK_SET, filename);
    for (i = 0; i < ntabs; i++) {
        if (!is_reencoded() || (strncmp(dir_tab[i].tag, "cmap", 4) != 0)) {
            copytab(dir_tab[i].tag);
            continue;
        }
        tab = name_lookup("cmap", true);
        max_glyph_index = 0;
        for (e = ttfenc_tab; e - ttfenc_tab < MAX_CHAR_NUM; e++) {
            e->newindex = 0; /* index of ".notdef" glyph */
            if (e->name != notdef) {
                for (glyph = glyph_tab; glyph - glyph_tab < glyphs_count; glyph++)
                    if (strcmp(e->name, glyph->name) == 0)
                        break;
                if (glyph - glyph_tab < glyphs_count)
                    e->newindex = glyph - glyph_tab;
                else
                    WARN("glyph `%s' undefined" AND e->name);
                if (e->newindex > max_glyph_index)
                    max_glyph_index = e->newindex;
            }
        }
        reenc_font();
    }
    copy_dirtab();
}

void writettf()
{
    filename = fm_cur->ff_name;
    packfilename(maketexstring(filename), getnullstr(), getnullstr());
    if (!TTF_OPEN()) {
        WARN("cannot open TrueType font file for reading");
        font_file_not_found = true;
        return;
    }
    if (is_subsetted() && !is_reencoded()) {
        WARN("encoding vector required for TrueType font subsetting");
        return;
    }
    TEX_PRINTF("<%s" AND fm_cur->ff_name);
    glyph_names_buf = 0;
    new_glyphs_count = 2;
    new_ntabs = DEFAULT_NTABS;
    dir_tab = 0;
    glyph_tab = 0;
    glyph_index = 0;
    glyph_names_buf = 0;
    names_tab = 0;
    names_storage = 0;
    ttfenc_tab = 0;
    read_font();
    if (is_included()) {
        pdfsaveoffset = pdfoffset();
        pdfflush();
        copy_encoding();
        if (!is_subsetted())
            copy_font();
        else
            subset_font();
        xfflush(OUTFILE);
        xfseek(OUTFILE, pdfgone, SEEK_SET, filename);
        ttf_length = pdfoffset() - pdfsaveoffset;
        XFREE(ttfenc_tab);
    }
    XFREE(dir_tab);
    XFREE(glyph_tab);
    XFREE(glyph_index);
    XFREE(glyph_names_buf);
    XFREE(names_tab);
    XFREE(names_storage);
    TEX_PRINTF(">");
}
